import QtQuick 1.0
import "scripts/settings.js" as Settings
import quickwidgets 1.0

Item {
    id: widget

    property string _ACTIVE_COLOR_HIGH : "#6382c6"
    property string _ACTIVE_COLOR_LOW : "#3d6be0"
    property string _BACKGROUND_COLOR : "black"
    property string _GRADIENT_COLOR_HIGH : "#524e4e"
    property string _GRADIENT_COLOR_LOW : "black"
    property string _TEXT_COLOR : "white"
    property int _SMALL_FONT_SIZE : 18
    property int _STANDARD_FONT_SIZE : 24
    property int _LARGE_FONT_SIZE : 36

    function getSearchFeed(query) {
        videoListModel.loading = true;

        var doc = new XMLHttpRequest();
        doc.onreadystatechange = function() {
            if (doc.readyState == XMLHttpRequest.DONE) {
                var xml = doc.responseText;
                videoListModel.setXml(xml);

                videoListModel.loading = false;
                videoList.positionViewAtIndex(0, ListView.Beginning);
            }
        }
        doc.open("GET", "http://gdata.youtube.com/feeds/api/videos?v=2&max-results=50&safeSearch=none&q=" + encodeURIComponent(query).replace(" ", "+") + "&orderby=relevance");
        doc.send();
    }

    function playVideo(playerUrl) {
        videoPlayer.command = "/opt/usr/bin/qmltube " + playerUrl;
        videoPlayer.run();
    }

    width: 360
    height: 320

    Component.onCompleted: getSearchFeed("nokia n900")

    Process {
        id: videoPlayer
    }

    Rectangle {
        anchors.fill: widget
        color: widget._BACKGROUND_COLOR
        radius: 10
        border.width: 2
        border.color: widget._ACTIVE_COLOR_LOW
        opacity: 0.5
    }

    Rectangle {
        height: 30
        anchors { top: widget.top; topMargin: 5; left: widget.left; leftMargin: 10; right: widget.right; rightMargin: 50 }
        color:  "white"
        border.width: 2
        border.color: widget._ACTIVE_COLOR_LOW
        radius: 5

        TextInput {
            id: searchInput

            anchors { fill: parent; margins: 2 }
            focus: true
            font.pixelSize: widget._SMALL_FONT_SIZE
            selectByMouse: true
            selectionColor: widget._ACTIVE_COLOR_LOW
            Keys.onEnterPressed: {
                if (searchInput.text != "") {
                    getSearchFeed(searchInput.text);
                }
            }
            Keys.onUpPressed: videoList.decrementCurrentIndex()
            Keys.onDownPressed: videoList.incrementCurrentIndex()
        }
    }

    ToolButton {
        id: searchButton

        width: 30
        height: 30
        anchors { top: widget.top; topMargin: 5; right: widget.right; rightMargin: 10 }
        icon: "ui-images/searchicon.png"
        iconWidth: 25
        iconHeight: 25
        onButtonClicked: {
            if (searchInput.text != "") {
                getSearchFeed(searchInput.text);
            }
        }
    }

    Text {
        id: noResultsText

        anchors.centerIn: widget
        font.pixelSize: _STANDARD_FONT_SIZE
        font.bold: true
        color: "grey"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("No videos found")
        visible: false

        Timer {
            interval: 5000
            running: (!videoListModel.loading) && (videoListModel.count == 0)
            onTriggered: {
                if (videoListModel.count == 0) {
                    noResultsText.visible = true;
                }
            }
        }
    }

    ListView {
        id: videoList

        anchors { fill: widget; topMargin: 40; bottomMargin: 30; leftMargin: 2; rightMargin: 2 }
        boundsBehavior: Flickable.DragOverBounds
        highlightMoveDuration: 500
        preferredHighlightBegin: 0
        preferredHighlightEnd: 80
        highlightRangeMode: ListView.StrictlyEnforceRange
        cacheBuffer: 1000
        interactive: false
        clip: true

        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

        model: VideoListModel {
            id: videoListModel

            property bool loading : true

            onLoadingChanged: {
                if (videoListModel.loading) {
                    noResultsText.visible = false;
                }
            }
        }

        delegate: WidgetDelegate {
            id: delegate

            onPlayClicked: playVideo(playerUrl)
        }
    }

    Row {
        anchors { top: videoList.bottom; left: widget.left; right: widget.right; leftMargin: 70; rightMargin: 70 }
        spacing: 140

        Image {
            id: upButton

            width: 40
            height: 30
            source: upMouseArea.pressed ? "ui-images/upicon2.png" : "ui-images/upicon.png"
            sourceSize.width: upButton.width
            sourceSize.height: upButton.height
            smooth: true

            MouseArea {
                id: upMouseArea

                anchors.fill: upButton
                onClicked: videoList.decrementCurrentIndex()
            }
        }

        Image {
            id: downButton

            width: 40
            height: 30
            source: downMouseArea.pressed ? "ui-images/downicon2.png" : "ui-images/downicon.png"
            sourceSize.width: downButton.width
            sourceSize.height: downButton.height
            smooth: true

            MouseArea {
                id: downMouseArea

                anchors.fill: downButton
                onClicked: videoList.incrementCurrentIndex()
            }
        }
    }

    Rectangle {
        width: 1
        height: 20
        anchors { bottom: widget.bottom; bottomMargin: 5; horizontalCenter: widget.horizontalCenter }
        color: widget._ACTIVE_COLOR_LOW
        opacity: 0.5
    }

}
