import QtQuick 1.0
import "scripts/videoinfoscripts.js" as Scripts
import "scripts/menu.js" as Menu
import "scripts/settings.js" as Settings
import "scripts/createobject.js" as ObjectCreator

Rectangle {
    id: window

    property variant video
    property string videoId
    property string playerUrl
    property string title
    property string author
    property string date
    property string description : qsTr("No description")
    property string thumbnail
    property string duration
    property string views : "0"
    property string comments : "0"
    property string likes : "0"
    property string dislikes : "0"
    property variant tags : []

    signal authorClicked(string username)
//    signal playVideo(variant video)
    signal goToVideo(variant video)
    signal search(string query, string order)
    signal dialogClose

    function setVideo(videoObject) {

        video = videoObject;
        videoId = video.videoId;
        playerUrl = video.playerUrl;
        title = video.title;
        author = video.author;
        date = video.uploadDate.split("T")[0];
        thumbnail = video.largeThumbnail;
        duration = Scripts.getDuration(video.duration);
        if (video.description != "") {
            description = video.description;
        }
        if (video.views != "") {
            views = video.views;
        }
        if (video.likes != "") {
            likes = video.likes;
        }
        if (video.dislikes != "") {
            dislikes = video.dislikes;
        }
        if (video.comments != "") {
            comments = video.comments;
        }
        if (video.tags != "") {
            tags = video.tags.split(", ");
        }
    }

    function toggleMenu() {
        Menu.toggleMenu();
    }

    color: _BACKGROUND_COLOR

    Connections {
        target: YouTube

        onCommentAdded: {
            displayMessage(qsTr("Your comment has been added"));
            commentsModel.reload();
        }
        onVideoRated: displayMessage(qsTr("Your rating has been added"))
    }

    Connections {
        target: Sharing

        onAlert: displayMessage(message)
        onPostedToFacebook: displayMessage(qsTr("Video shared on facebook"))
        onRenewFacebookToken: {
            Scripts.closeDialogs();
            Settings.deleteAccessToken("Facebook");
            Scripts.checkFacebookAccess();
        }

    }

    Rectangle {
        id: dimmer

        anchors.fill: window
        color: window.color

        Loader {
            id: menuLoader

            z: 20
            width: dimmer.width / 2
            anchors { left: dimmer.left; bottom: dimmer.bottom; bottomMargin: 60 }

            onLoaded: menuLoader.item.addMenuItems([{ name: qsTr("Add to favourites") }, { name: qsTr("Add to playlist") }, { name: qsTr("Copy to clipboard") },
                                                    { name: qsTr("Share via facebook") } ])

            Timer {
                id: menuTimer

                interval: 600
                onTriggered: menuLoader.source = ""
            }

            Connections {
                target: menuLoader.item

                onItemClicked: {
                    toggleMenu();
                    if (index == 0) {
                        if (userIsSignedIn()) {
                            YouTube.addToFavourites(videoId);
                        }
                        else {
                            displayMessage(qsTr("You are not signed in to a YouTube account"));
                        }
                    }
                    else if (index == 1) {
                        if (userIsSignedIn()) {
                            Scripts.showPlaylistDialog();
                        }
                        else {
                            displayMessage(qsTr("You are not signed in to a YouTube account"));
                        }
                    }
                    else if (index == 2) {
                        Controller.copyToClipboard(playerUrl.split("&")[0]);
                    }
                    else if (index == 3) {
                        Scripts.shareVideo("Facebook");
                    }
                }
            }
        }

        Rectangle {
            id: frame

            width: Math.floor(window.width / 3.2)
            height: Math.floor(frame.width / (4 / 3))
            anchors { left: dimmer.left; leftMargin: 10; top: dimmer.top; topMargin: 60 }
            color: _BACKGROUND_COLOR
            border.width: 2
            border.color: "white"

            Image {
                id: thumb

                anchors { fill: frame; margins: 2 }
                source: thumbnail
                sourceSize.width: thumb.width
                sourceSize.height: thumb.height
                smooth: true

                Rectangle {
                    id: durationLabel

                    width: Math.floor(thumb.width / 2.5)
                    height: Math.floor(durationLabel.width / 2.5)
                    anchors { bottom: thumb.bottom; right: thumb.right }
                    color: "black"
                    opacity: 0.5
                    smooth: true
                }

                Text {
                    id: durationText

                    anchors.fill: durationLabel
                    text: duration
                    color: "white"
                    font.pixelSize: _STANDARD_FONT_SIZE
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }

                Rectangle {
                    width: Math.floor(thumb.width / 3.5)
                    height: width
                    anchors.centerIn: thumb
                    color: frameMouseArea.pressed ? _ACTIVE_COLOR_LOW : "black"
                    opacity: 0.5
                    radius: 5
                    smooth: true

                    Image {
                        id: playIcon

                        anchors { fill: parent; margins: 5 }
                        smooth: true
                        source: "ui-images/playicon.png"
                        sourceSize.width: playIcon.width
                        sourceSize.height: playIcon.height
                    }
                }
            }

            MouseArea {
                id: frameMouseArea

                anchors.fill: frame
                onClicked: YouTube.getVideoUrl(playerUrl)  //playVideo([video])
            }

            Grid {
                id: buttonGrid

                anchors { left: frame.left; top: frame.bottom; topMargin: 10 }
                columns: 2
                spacing: 10

                PushButton {
                    id: videoButton

                    width: Math.floor(frame.width / 2) - 5
                    icon: "ui-images/videodownloadicon.png"
                    iconWidth: 65
                    iconHeight: 65
                    onButtonClicked: addDownload(video)
                }

                PushButton {
                    id: audioButton

                    width: videoButton.width
                    height: videoButton.height
                    icon: "ui-images/audiodownloadicon.png"
                    iconWidth: 65
                    iconHeight: 65
                    onButtonClicked: addAudioDownload(video)
                }
            }
        }

        Item {
            id: tabItem

            anchors { fill: dimmer; leftMargin: frame.width + 20; rightMargin: 10; topMargin: 60; bottomMargin: 60 }

            Row {
                id: tabRow

                Rectangle {
                    id: infoTab

                    width: tabItem.width / 3
                    height: 40
                    color: _BACKGROUND_COLOR

                    BorderImage {
                        anchors.fill: infoTab
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 0
                    }

                    Text {
                        anchors.fill: infoTab
                        font.pixelSize: _STANDARD_FONT_SIZE
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 0 ? _TEXT_COLOR : "grey"
                        text: qsTr("Info")
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: infoTab.bottom; left: infoTab.left; right: infoTab.right }
                        color: _ACTIVE_COLOR_HIGH
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 0)
                    }

//                    MouseArea {
//                        id: infoMouseArea

//                        anchors.fill: infoTab
//                        onClicked: tabView.currentIndex = 0
//                    }
                }

                Rectangle {
                    id: commentsTab

                    width: tabItem.width / 3
                    height: 40
                    color: _BACKGROUND_COLOR

                    BorderImage {
                        anchors.fill: parent
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 1
                    }

                    Text {
                        anchors.fill: commentsTab
                        font.pixelSize: _STANDARD_FONT_SIZE
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 1 ? _TEXT_COLOR : "grey"
                        text: qsTr("Comments")
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: commentsTab.bottom; left: commentsTab.left; right: commentsTab.right }
                        color: _ACTIVE_COLOR_HIGH
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 1)
                    }

//                    MouseArea {
//                        id: commentsMouseArea

//                        anchors.fill: commentsTab
//                        onClicked: tabView.currentIndex = 1
//                    }
                }

                Rectangle {
                    id: relatedTab

                    width: tabItem.width / 3
                    height: 40
                    color: _BACKGROUND_COLOR

                    BorderImage {
                        anchors.fill: parent
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 2
                    }

                    Text {
                        anchors.fill: relatedTab
                        font.pixelSize: _STANDARD_FONT_SIZE
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 2 ? _TEXT_COLOR : "grey"
                        text: qsTr("Related")
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: relatedTab.bottom; left: relatedTab.left; right: relatedTab.right }
                        color: _ACTIVE_COLOR_HIGH
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 2)
                    }

//                    MouseArea {
//                        id: relatedMouseArea

//                        anchors.fill: relatedTab
//                        onClicked: tabView.currentIndex = 2
//                    }
                }
            }

            BusyDialog {
                anchors.centerIn: tabItem
                opacity: (tabView.currentIndex == 1 && commentsModel.status == XmlListModel.Loading) ||
                         (tabView.currentIndex == 2 && relatedModel.status == XmlListModel.Loading) ? 1 : 0
            }

            ListView {
                id: tabView

                anchors { left: tabItem.left; right: tabItem.right; top: tabRow.bottom; bottom: tabItem.bottom }
                orientation: ListView.Horizontal
                highlightMoveDuration: 200
                highlightRangeMode: ListView.StrictlyEnforceRange
                snapMode: ListView.SnapOneItem
                flickDeceleration: 500
                boundsBehavior: Flickable.StopAtBounds
                model: tabModel
                clip: true

                onCurrentIndexChanged: {
                    if ((tabView.currentIndex == 1) && (comments != "0") && (!commentsList.loaded)) {
                        commentsList.loadComments();
                    }
                    else if ((tabView.currentIndex == 2) && (!relatedView.loaded) && (relatedView.count == 0)) {
                        relatedView.loadVideos();
                    }
                }
            }
        }

        VisualItemModel {
            id: tabModel

            Flickable {
                id: scrollArea

                width: tabView.width
                height: tabView.height
                clip: true
                contentWidth: textColumn.width
                contentHeight: textColumn.height
                flickableDirection: Flickable.VerticalFlick
                boundsBehavior: Flickable.DragOverBounds
                interactive: visibleArea.heightRatio < 1
                visible: tabView.currentIndex == 0

                Column {
                    id: textColumn

                    spacing: 10
                    width: tabView.width
                    height: childrenRect.height

                    Text {
                        id: titleText

                        width: textColumn.width
                        text: title
                        color: _TEXT_COLOR
                        font.pixelSize: _STANDARD_FONT_SIZE
                        wrapMode: TextEdit.WordWrap
                    }

                    Text {
                        id: authorText

                        width: textColumn.width
                        color: _TEXT_COLOR
                        font.pixelSize: _SMALL_FONT_SIZE
                        textFormat: Text.StyledText
                        wrapMode: TextEdit.WordWrap
                        text: authorMouseArea.pressed ? qsTr("By ") + "<font color='"
                                                        + _ACTIVE_COLOR_HIGH + "'>" + author
                                                        + "</font> on " + date
                                                      : qsTr("By ") + "<font color='"
                                                        + _ACTIVE_COLOR_LOW + "'>" + author
                                                        + "</font> on " + date

                        MouseArea {
                            id: authorMouseArea

                            anchors.fill: authorText
                            onClicked: authorClicked(author)
                        }
                    }

                    Row {

                        spacing: 10

                        ToolButton {
                            id: likeButton

                            icon: "ui-images/likeicon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        YouTube.rateVideo(videoId, "like");
                                    }
                                    else {
                                        displayMessage(qsTr("You are not signed in to a YouTube account"));
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "grey"
                            text: likes
                        }

                        ToolButton {
                            id: dislikeButton

                            icon: "ui-images/dislikeicon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        YouTube.rateVideo(videoId, "dislike");
                                    }
                                    else {
                                        displayMessage(qsTr("You are not signed in to a YouTube account"));
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "grey"
                            text: dislikes
                        }

                        Text {
                            y: 20
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: _TEXT_COLOR
                            text: qsTr("Views")
                        }

                        Text {
                            y: 20
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "grey"
                            text: views
                        }
                    }

                    Text {
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: _TEXT_COLOR
                        text: qsTr("Description")
                    }

                    Text {
                        id: descriptionText

                        width: textColumn.width
                        text: description
                        color: "grey"
                        font.pixelSize: _SMALL_FONT_SIZE
                        wrapMode: TextEdit.WordWrap
                    }

                    Text {
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: _TEXT_COLOR
                        text: qsTr("Tags")
                    }

                    Flow {
                        spacing: 10
                        width: parent.width

                        Repeater {
                            model: tags

                            Text {
                                font.pixelSize: _SMALL_FONT_SIZE
                                color: children[0].pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                                text: modelData

                                MouseArea {
                                    anchors.fill: parent
                                    onClicked: search(parent.text, Settings.getSetting("searchOrder").toLowerCase())
                                }

                                Text {
                                    anchors.left: parent.right
                                    font.pixelSize: _SMALL_FONT_SIZE
                                    color: "grey"
                                    text: ","
                                    visible: index < (tags.length - 1)
                                }
                            }
                        }
                    }

                }
            }

            Item {
                id: commentsItem

                width: tabView.width
                height: tabView.height

                Column {
                    y: 10
                    spacing: 10

                    Row {
                        spacing: 10

                        ToolButton {
                            id: commentButton

                            icon: "ui-images/commenticon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        Scripts.showAddCommentDialog();
                                    }
                                    else {
                                        displayMessage(qsTr("You are not signed in to a YouTube account"));
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "grey"
                            horizontalAlignment: Text.AlignHCenter
                            verticalAlignment: Text.AlignVCenter
                            text: comments == "0" ? qsTr("No comments") : comments + qsTr(" comments")
                        }
                    }

                    ListView {
                        id: commentsList

                        property bool loaded : false // True if comments have been loaded

                        function loadComments() {
                            var doc = new XMLHttpRequest();
                            doc.onreadystatechange = function() {
                                if (doc.readyState == XMLHttpRequest.DONE) {
                                    var xml = doc.responseText;
                                    commentsModel.xml = xml;
                                    commentsList.loaded = true;
                                }
                            }
                            doc.open("GET", "http://gdata.youtube.com/feeds/api/videos/" + videoId + "/comments?v=2&max-results=50");
                            doc.send();
                        }

                        width: commentsItem.width
                        height: commentsItem.height - 60
                        clip: true
                        interactive: visibleArea.heightRatio < 1
                        visible: tabView.currentIndex == 1

                        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

                        model: CommentsModel {
                            id: commentsModel
                        }

                        delegate: CommentsDelegate {
                            id: commentDelegate

                            onCommentClicked: authorClicked(author);
                        }
                    }
                }
            }

            ListView {
                id: relatedView

                property bool loaded : false // True if related videos have been loaded

                function loadVideos() {
                    var doc = new XMLHttpRequest();
                    doc.onreadystatechange = function() {
                        if (doc.readyState == XMLHttpRequest.DONE) {
                            var xml = doc.responseText;
                            relatedModel.setXml(xml);
                            relatedView.loaded = true;
                        }
                    }
                    doc.open("GET", "http://gdata.youtube.com/feeds/api/videos/" + videoId + "/related?v=2&max-results=50");
                    doc.send();
                }

                width: tabView.width
                height: tabView.height
                boundsBehavior: Flickable.DragOverBounds
                highlightMoveDuration: 500
                preferredHighlightBegin: 0
                preferredHighlightEnd: 100
                highlightRangeMode: ListView.StrictlyEnforceRange
                cacheBuffer: 2500
                interactive: visibleArea.heightRatio < 1
                clip: true
                visible: tabView.currentIndex == 2

                model: VideoListModel {
                    id: relatedModel
                }

                delegate: VideoListDelegate {
                    id: delegate

                    onDelegateClicked: goToVideo(relatedModel.get(index))
                    onPlayClicked: YouTube.getVideoUrl(playerUrl)
                }

                Text {
                    id: noResultsText

                    anchors { top: relatedView.top; topMargin: 60; horizontalCenter: relatedView.horizontalCenter }
                    font.pixelSize: _STANDARD_FONT_SIZE
                    font.bold: true
                    color: "grey"
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    text: qsTr("No related videos")
                    visible: false

                    Timer {
                        interval: 5000
                        running: (relatedView.loaded) && (relatedModel.count == 0)
                        onTriggered: {
                            if (relatedModel.count == 0) {
                                noResultsText.visible = true;
                            }
                        }
                    }
                }

                ScrollBar {}
            }
        }

        MouseArea {
            id: mouseArea

            anchors.fill: dimmer
            enabled: false
            onClicked: Scripts.closeDialogs()
        }

        states: State {
            name: "dim"
            PropertyChanges { target: dimmer; opacity: 0.3 }
        }

        transitions: Transition {
            PropertyAnimation { properties: "opacity"; duration: 500 }
        }
    }

    states: State {
        name: "portrait"
        when: window.height > window.width
        PropertyChanges { target: frame; width: Math.floor(window.width / 1.9) }
        PropertyChanges { target: tabItem; anchors { leftMargin: 10; rightMargin: 10; topMargin: frame.height + 70 } }
        PropertyChanges { target: videoButton; width: window.width - (frame.width + 30); height: Math.floor(frame.height / 2) - 5 }
        AnchorChanges { target: buttonGrid; anchors { left: frame.right; top: frame.top } }
        PropertyChanges { target: buttonGrid; anchors.leftMargin: 10; anchors.topMargin: 0; columns: 1 }
    }
}

