import QtQuick 1.0
import "videoinfoscripts.js" as Scripts
import "menu.js" as Menu
import "settings.js" as Settings

Rectangle {
    id: window

    property variant video
    property string videoId
    property string playerUrl
    property string title
    property string author
    property string date
    property string description : "No description"
    property string thumbnail
    property string duration
    property string views : "0"
    property string comments : "0"
    property string likes : "0"
    property string dislikes : "0"
    property variant tags : []

    signal authorClicked(string username)
//    signal playVideo(variant video)
    signal goToVideo(variant video)
    signal search(string query, string order)

    function setVideo(videoObject) {

        video = videoObject;
        videoId = video.videoId;
        playerUrl = video.playerUrl;
        title = video.title;
        author = video.author;
        date = video.uploadDate.split("T")[0];
        thumbnail = video.largeThumbnail;
        duration = Scripts.getDuration(video.duration);
        if (video.description != "") {
            description = video.description;
        }
        if (video.views != "") {
            views = video.views;
        }
        if (video.likes != "") {
            likes = video.likes;
        }
        if (video.dislikes != "") {
            dislikes = video.dislikes;
        }
        if (video.comments != "") {
            comments = video.comments;
        }
        if (video.tags != "") {
            tags = video.tags.split(", ");
        }
    }

    function toggleMenu() {
        Menu.toggleMenu();
    }

    color: "black"
    onStateChanged: Scripts.rotateDialogs()

    Connections {
        target: YouTube

        onCommentAdded: {
            displayMessage("Your comment has been added");
            commentsModel.reload();
        }
        onVideoRated: displayMessage("Your rating has been added")
    }

    Loader {
        id: dialog

        anchors { top: window.bottom; topMargin: 10; horizontalCenter: window.horizontalCenter }
        z: 1
        onLoaded: {
            dialog.item.state = window.state;
            dialog.state = "show";
        }

        Timer {
            id: timer

            interval: 600
            onTriggered: dialog.source = ""
        }

        states: [
            State {
                name: "show"
                AnchorChanges { target: dialog; anchors { top: undefined; verticalCenter: window.verticalCenter } }
            }
        ]

        transitions: [
            Transition {
                AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
            }
        ]
    }

    Rectangle {
        id: dimmer

        anchors.fill: window
        color: window.color

        Loader {
            id: menuLoader

            z: 20
            width: dimmer.width / 2
            anchors { left: dimmer.left; bottom: dimmer.bottom; bottomMargin: 60 }

            onLoaded: menuLoader.item.addMenuItems([{ name: "Add to favourites" }, { name: "Add to playlist" }, { name: "Copy to clipboard" } ])

            Timer {
                id: menuTimer

                interval: 600
                onTriggered: menuLoader.source = ""
            }

            Connections {
                target: menuLoader.item

                onItemClicked: {
                    toggleMenu();
                    if (index == 0) {
                        if (userIsSignedIn()) {
                            YouTube.addToFavourites(videoId);
                        }
                        else {
                            displayMessage("You are not signed in to a YouTube account");
                        }
                    }
                    else if (index == 1) {
                        if (userIsSignedIn()) {
                            Scripts.showPlaylistDialog();
                        }
                        else {
                            displayMessage("You are not signed in to a YouTube account");
                        }
                    }
                    else if (index == 2) {
                        Controller.copyToClipboard(playerUrl.split("&")[0]);
                    }
                }
            }
        }

        Rectangle {
            id: frame

            width: 250
            height: 190
            anchors { left: dimmer.left; leftMargin: 10; top: dimmer.top; topMargin: 60 }
            color: "black"
            border.width: 3
            border.color: frameMouseArea.pressed ? activeColorLow : "white"

            Image {
                id: thumb

                anchors { fill: frame; margins: 3 }
                source: thumbnail
                sourceSize.width: thumb.width
                sourceSize.height: thumb.height
                smooth: true

                Rectangle {
                    id: durationLabel

                    width: 100
                    height: 44
                    anchors { bottom: thumb.bottom; right: thumb.right }
                    color: "black"
                    opacity: 0.5

                }

                Text {
                    id: durationText

                    anchors.fill: durationLabel
                    text: duration
                    color: "white"
                    font.pixelSize: standardFontSize
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                }
            }

            MouseArea {
                id: frameMouseArea

                anchors.fill: frame
                onClicked: YouTube.getVideoUrl(playerUrl)  //playVideo([video])
            }

            Grid {
                id: buttonGrid

                anchors { left: frame.left; top: frame.bottom; topMargin: 10 }
                columns: 2
                spacing: 10

                PushButton {
                    id: videoButton

                    width: 120
                    icon: "ui-images/videodownloadicon.png"
                    iconWidth: 65
                    iconHeight: 65
                    onButtonClicked: addDownload(video)
                }

                PushButton {
                    id: audioButton

                    width: 120
                    icon: "ui-images/audiodownloadicon.png"
                    iconWidth: 65
                    iconHeight: 65
                    onButtonClicked: addAudioDownload(video)
                }
            }
        }

        Item {
            id: tabItem

            anchors { fill: dimmer; leftMargin: 270; rightMargin: 10; topMargin: 60 }

            Row {
                id: tabRow

                Rectangle {
                    id: infoTab

                    width: tabItem.width / 3
                    height: 40
                    color: "black"

                    BorderImage {
                        anchors.fill: infoTab
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 0
                    }

                    Text {
                        anchors.fill: infoTab
                        font.pixelSize: standardFontSize
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 0 ? "white" : "grey"
                        text: "Info"
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: infoTab.bottom; left: infoTab.left; right: infoTab.right }
                        color: activeColor
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 0)
                    }

                    MouseArea {
                        id: infoMouseArea

                        anchors.fill: infoTab
                        onClicked: tabView.currentIndex = 0
                    }
                }

                Rectangle {
                    id: commentsTab

                    width: tabItem.width / 3
                    height: 40
                    color: "black"

                    BorderImage {
                        anchors.fill: parent
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 1
                    }

                    Text {
                        anchors.fill: commentsTab
                        font.pixelSize: standardFontSize
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 1 ? "white" : "grey"
                        text: "Comments"
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: commentsTab.bottom; left: commentsTab.left; right: commentsTab.right }
                        color: activeColor
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 1)
                    }

                    MouseArea {
                        id: commentsMouseArea

                        anchors.fill: commentsTab
                        onClicked: tabView.currentIndex = 1
                    }
                }

                Rectangle {
                    id: relatedTab

                    width: tabItem.width / 3
                    height: 40
                    color: "black"

                    BorderImage {
                        anchors.fill: parent
                        source: "ui-images/tab.png"
                        smooth: true
                        visible: tabView.currentIndex == 2
                    }

                    Text {
                        anchors.fill: relatedTab
                        font.pixelSize: standardFontSize
                        horizontalAlignment: Text.AlignHCenter
                        verticalAlignment: Text.AlignVCenter
                        color: tabView.currentIndex == 2 ? "white" : "grey"
                        text: "Related"
                    }

                    Rectangle {
                        height: 1
                        anchors { bottom: relatedTab.bottom; left: relatedTab.left; right: relatedTab.right }
                        color: activeColor
                        opacity: 0.5
                        visible: !(tabView.currentIndex == 2)
                    }

                    MouseArea {
                        id: relatedMouseArea

                        anchors.fill: relatedTab
                        onClicked: tabView.currentIndex = 2
                    }
                }
            }

            BusyDialog {
                anchors.centerIn: tabItem
                opacity: (tabView.currentIndex == 1 && commentsModel.status == XmlListModel.Loading) ||
                         (tabView.currentIndex == 2 && relatedModel.status == XmlListModel.Loading) ? 1 : 0
            }

            ListView {
                id: tabView

                anchors { left: tabItem.left; right: tabItem.right; top: tabRow.bottom; bottom: tabItem.bottom }
                orientation: ListView.Horizontal
                highlightMoveDuration: 200
                highlightRangeMode: ListView.StrictlyEnforceRange
                snapMode: ListView.SnapOneItem
                flickDeceleration: 500
                boundsBehavior: Flickable.StopAtBounds
                model: tabModel
                clip: true

                onCurrentIndexChanged: {
                    if ((tabView.currentIndex == 1) && (comments != "0") && (!commentsList.loaded)) {
                        commentsList.loadComments();
                    }
                    else if ((tabView.currentIndex == 2) && (!relatedView.loaded) && (relatedView.count == 0)) {
                        relatedView.loadVideos();
                    }
                }
            }
        }

        VisualItemModel {
            id: tabModel

            Flickable {
                id: scrollArea

                width: 520
                height: 320
                clip: true
                contentWidth: textColumn.width
                contentHeight: textColumn.height
                flickableDirection: Flickable.VerticalFlick
                boundsBehavior: Flickable.DragOverBounds
                interactive: visibleArea.heightRatio < 1
                visible: tabView.currentIndex == 0

                Column {
                    id: textColumn

                    spacing: 10
                    width: 520
                    height: childrenRect.height

                    Text {
                        id: titleText

                        width: textColumn.width
                        text: title
                        color: "white"
                        font.pixelSize: standardFontSize
                        wrapMode: TextEdit.WordWrap
                    }

                    Text {
                        id: authorText

                        width: textColumn.width
                        color: "white"
                        font.pixelSize: smallFontSize
                        textFormat: Text.StyledText
                        wrapMode: TextEdit.WordWrap
                        text: authorMouseArea.pressed ? "By <font color='"
                                                        + activeColor + "'>" + author
                                                        + "</font> on " + date
                                                      : "By <font color='"
                                                        + activeColorLow + "'>" + author
                                                        + "</font> on " + date

                        MouseArea {
                            id: authorMouseArea

                            anchors.fill: authorText
                            onClicked: authorClicked(author)
                        }
                    }

                    Row {

                        spacing: 10

                        ToolButton {
                            id: likeButton

                            icon: "ui-images/likeicon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        YouTube.rateVideo(videoId, "like");
                                    }
                                    else {
                                        displayMessage("You are not signed in to a YouTube account");
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: smallFontSize
                            color: "grey"
                            text: likes
                        }

                        ToolButton {
                            id: dislikeButton

                            icon: "ui-images/dislikeicon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        YouTube.rateVideo(videoId, "dislike");
                                    }
                                    else {
                                        displayMessage("You are not signed in to a YouTube account");
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: smallFontSize
                            color: "grey"
                            text: dislikes
                        }

                        Text {
                            y: 20
                            font.pixelSize: smallFontSize
                            color: "white"
                            text: "Views"
                        }

                        Text {
                            y: 20
                            font.pixelSize: smallFontSize
                            color: "grey"
                            text: views
                        }
                    }

                    Text {
                        font.pixelSize: smallFontSize
                        color: "white"
                        text: "Description"
                    }

                    Text {
                        id: descriptionText

                        width: textColumn.width
                        text: description
                        color: "grey"
                        font.pixelSize: smallFontSize
                        wrapMode: TextEdit.WordWrap
                    }

                    Text {
                        font.pixelSize: smallFontSize
                        color: "white"
                        text: "Tags"
                    }

                    Flow {
                        spacing: 10
                        width: parent.width

                        Repeater {
                            model: tags

                            Text {
                                font.pixelSize: smallFontSize
                                color: children[0].pressed ? activeColor : activeColorLow
                                text: modelData

                                MouseArea {
                                    anchors.fill: parent
                                    onClicked: search(parent.text, Settings.getSetting("searchOrder").toLowerCase())
                                }

                                Text {
                                    anchors.left: parent.right
                                    font.pixelSize: smallFontSize
                                    color: "grey"
                                    text: ","
                                }
                            }
                        }
                    }

                }
            }

            Item {
                id: commentsItem

                width: 520
                height: 320

                Column {
                    y: 10
                    spacing: 10

                    Row {
                        spacing: 10

                        ToolButton {
                            id: commentButton

                            icon: "ui-images/commenticon.png"

                            Connections {
                                onButtonClicked: {
                                    if (userIsSignedIn()) {
                                        Scripts.showAddCommentDialog();
                                    }
                                    else {
                                        displayMessage("You are not signed in to a YouTube account");
                                    }
                                }
                            }
                        }

                        Text {
                            y: 20
                            font.pixelSize: smallFontSize
                            color: "grey"
                            horizontalAlignment: Text.AlignHCenter
                            verticalAlignment: Text.AlignVCenter
                            text: comments == "0" ? "No comments" : comments + " comments"
                        }
                    }

                    ListView {
                        id: commentsList

                        property bool loaded : false // True if comments have been loaded

                        function loadComments() {
                            commentsModel.source = "http://gdata.youtube.com/feeds/api/videos/" + videoId + "/comments?v=2&max-results=50";
                            commentsModel.reload();
                            commentsList.loaded = true;
                        }

                        width: commentsItem.width
                        height: commentsItem.height - 60
                        clip: true
                        interactive: visibleArea.heightRatio < 1
                        visible: tabView.currentIndex == 1

                        Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 500 } }

                        model: CommentsModel {
                            id: commentsModel
                        }

                        delegate: CommentsDelegate {
                            id: commentDelegate

                            onCommentClicked: authorClicked(author);
                        }
                    }
                }
            }

            ListView {
                id: relatedView

                property bool loaded : false // True if related videos have been loaded

                function loadVideos() {
                    var doc = new XMLHttpRequest();
                    doc.onreadystatechange = function() {
                        if (doc.readyState == XMLHttpRequest.DONE) {
                            var xml = doc.responseText;
                            relatedModel.setXml(xml);
                            relatedView.loaded = true;
                        }
                    }
                    doc.open("GET", "http://gdata.youtube.com/feeds/api/videos/" + videoId + "/related?v=2&max-results=50");
                    doc.send();
                }

                width: 520
                height: 320
                boundsBehavior: Flickable.DragOverBounds
                highlightMoveDuration: 500
                preferredHighlightBegin: 0
                preferredHighlightEnd: 100
                highlightRangeMode: ListView.StrictlyEnforceRange
                cacheBuffer: 1000
                interactive: visibleArea.heightRatio < 1
                clip: true
                visible: tabView.currentIndex == 2

                model: VideoListModel {
                    id: relatedModel
                }

                delegate: VideoListDelegate {
                    id: delegate

                    Connections {
                        onDelegateClicked: goToVideo(relatedModel.get(index))
                    }
                }

                Text {
                    id: noResultsText

                    anchors { top: relatedView.top; topMargin: 60; horizontalCenter: relatedView.horizontalCenter }
                    font.pixelSize: standardFontSize
                    font.bold: true
                    color: "grey"
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    text: "No related videos"
                    visible: (relatedModel.status == XmlListModel.Ready) && (relatedModel.count == 0)
                }

                ScrollBar {}
            }
        }

        MouseArea {
            id: mouseArea

            anchors.fill: dimmer
            enabled: false
            onClicked: Scripts.closeDialogs()
        }

        states: State {
            name: "dim"
            PropertyChanges { target: dimmer; opacity: 0.3 }
        }

        transitions: Transition {
            PropertyAnimation { properties: "opacity"; duration: 500 }
        }
    }

    states: State {
        name: "portrait"
        when: window.height > window.width
        PropertyChanges { target: tabItem; anchors { leftMargin: 10; rightMargin: 10; topMargin: 260 } }
        PropertyChanges { target: scrollArea; width: 460; height: 440 }
        PropertyChanges { target: textColumn; width: 460 }
        PropertyChanges { target: commentsItem; width: 460; height: 440 }
        PropertyChanges { target: relatedView; width: 460; height: 440 }
        PropertyChanges { target: videoButton; width: 200 }
        PropertyChanges { target: audioButton; width: 200 }
        AnchorChanges { target: buttonGrid; anchors { left: frame.right; top: frame.top } }
        PropertyChanges { target: buttonGrid; anchors.leftMargin: 10; anchors.topMargin: 0; columns: 1 }
    }
}

