import QtQuick 1.0
import "mainscripts.js" as Scripts
import "settings.js" as Settings

Rectangle {
    id: window

    Component.onCompleted: Scripts.restoreSettings()

    property string activeColor : "#6382c6"
    property string activeColorLow : "#3d6be0"
    property int standardFontSize : 24
    property int smallFontSize : 18
    property int largeFontSize : 36

    function userIsSignedIn() {
        /* Check if the user is signed in */

        var signedIn = false;
        if (YouTube.currentUser != "") {
            signedIn = true;
        }
        else if (Settings.getDefaultAccount() != "unknown") {
            Scripts.signInToDefaultAccount();
            signedIn = true;
        }
        return signedIn;
    }

    function displayMessage(message) {
        /* Display a notification using the message banner */

        messageBanner.source = "";
        messageBanner.source = "MessageBanner.qml";
        messageBanner.item.message = message;
        toggleBusy(false);
    }

    function toggleBusy(isBusy) {
        notificationArea.isBusy = isBusy;
    }

    function addDownload(video) {
        /* Add a video to the download list model */

        var downloadItem = {
            filePath: "/home/user/MyDocs/.cutetube/" + video.title.replace(/[@&~=\/:?#|]/g, "_") + ".mp4",
            title: video.title,
            thumbnail: video.thumbnail,
            playerUrl: video.playerUrl.split("&")[0],
            status: Settings.getSetting("downloadStatus"),
            quality: Settings.getSetting("downloadQuality"),
            bytesReceived: 0,
            totalBytes: 100
        }
        downloadModel.appendDownload(downloadItem);
        displayMessage("Added to download queue");
    }

    width: 800
    height: 480
    color: "black"
    onStateChanged: {
        if (messageBanner.source != "") {
            messageBanner.item.state = window.state;
        }
    }

    Connections {
        /* Connect to signals from C++ object YouTube */

        target: YouTube

        onAlert: {
            displayMessage(message);
        }
        onAddedToFavourites: displayMessage("Video added to favourites")
        onAddedToPlaylist: displayMessage("Video added to playlist")
    }

    Connections {
        /* Connect to signals from C++ object DownloadManager */

        target: DownloadManager
        onAlert: displayMessage(message)
        onDownloadCompleted: {
            Scripts.moveToArchive(downloadModel.get(downloadModel.currentDownload));
            downloadModel.remove(downloadModel.currentDownload);
            downloadModel.getNextDownload();
        }
        onDownloadCancelled: {
            Settings.removeStoredDownload(downloadModel.get(downloadModel.currentDownload));
            downloadModel.remove(downloadModel.currentDownload);
            downloadModel.getNextDownload();
        }
        onQualityChanged: downloadModel.setProperty(downloadModel.currentDownload, "quality", quality)
        onStatusChanged: downloadModel.setProperty(downloadModel.currentDownload, "status", status)
        onProgressChanged: {
            downloadModel.setProperty(downloadModel.currentDownload, "bytesReceived", bytesReceived);
            downloadModel.setProperty(downloadModel.currentDownload, "totalBytes", bytesTotal);
        }
    }

    ListModel {
        /* Download list model */
        id: downloadModel

        property int currentDownload : 0

        function appendDownload(downloadItem) {
            downloadModel.append(downloadItem);
            if ((downloadItem.status == "Queued") && (!DownloadManager.isDownloading)) {
                DownloadManager.startDownload(downloadItem.filePath, downloadItem.playerUrl);
                downloadModel.currentDownload = downloadModel.count - 1;
            }
        }

        function resumeDownload(index) {
            downloadModel.setProperty(index, "status", "Queued");
            if (!DownloadManager.isDownloading) {
                var downloadItem = downloadModel.get(index);
                DownloadManager.startDownload(downloadItem.filePath, downloadItem.playerUrl);
                downloadModel.currentDownload = index;
            }
        }

        function pauseDownload(index) {
            if ((index == downloadModel.currentDownload) && (DownloadManager.isDownloading)) {
                DownloadManager.pauseDownload();
            }
            else {
                downloadModel.setProperty(index, "status", "Paused");
            }
        }

        function cancelDownload(index) {
            if ((index == downloadModel.currentDownload) && (DownloadManager.isDownloading)) {
                DownloadManager.cancelDownload();
            }
            else {
                Settings.removeStoredDownload(downloadModel.get(index));
                downloadModel.remove(index);
            }
        }

        function getNextDownload() {
            var i = 0;
            while ((i < downloadModel.count) && (!DownloadManager.isDownloading)) {
                var downloadItem = downloadModel.get(i);
                if (downloadItem.status == "Queued") {
                    DownloadManager.startDownload(downloadItem.filePath, downloadItem.playerUrl);
                    downloadModel.currentDownload = i;
                }
                i++;
            }
        }
    }

    ListModel {
        /* Archive list model */

        id: archiveModel
    }

    PlaylistModel {
        /* Holds the users YouTube playlists */

        id: playlistModel
    }

    SubscriptionsModel {
        /* Holds the users YouTube subscriptions */

        id: subscriptionsModel
    }

    NotificationArea {
        id: notificationArea

        property variant titleList : ["cuteTube"]

        function addTitle(title) {
            var titles = titleList;
            titles.push(title);
            titleList = titles;
        }

        function removeTitle() {
            var titles = titleList;
            titles.pop();
            titleList = titles;
        }

        z: 100
        viewTitle: titleList[windowView.currentIndex]

        Connections {
            onStartSearch: Scripts.search(query)
            onGoToVideo: Scripts.loadVideoInfo(videoId)
            onGoToDownloads: {
                if (windowView.currentIndex != viewsModel.count - 1) {
                    Scripts.loadDownloads()
                }
            }
        }
    }

    Loader {
        id: messageBanner

        width: window.width
        anchors.bottom: window.top
        z: 1
        onLoaded: {
            messageBanner.item.state = window.state;
            timer.running = true;
            messageBanner.state = "show"
        }

        Timer {
            id: timer

            interval: 2500
            onTriggered: messageBanner.state = ""
        }

        states: State {
            name: "show"
            AnchorChanges { target: messageBanner; anchors { bottom: undefined; top: notificationArea.bottom } }
        }

        transitions: Transition {
            AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
        }

    }

    MenuBar {
        id: menuBar

        showBackButton: windowView.currentIndex != 0
        showQuitButton: windowView.currentIndex == 0

        Connections {
            onMenuClicked: Scripts.toggleMenu()
            onQuitClicked: Qt.quit()
            onBackClicked: Scripts.goToPreviousView()
            onHomeClicked: Scripts.goHome()
        }
    }

    VisualItemModel {
        id: viewsModel

        HomeView {
            id: homeView

            width: window.width
            height: window.height

            property string mostRecentFeed : "http://gdata.youtube.com/feeds/api/standardfeeds/most_recent?v=2&max-results=50"
            property string mostViewedFeed : "http://gdata.youtube.com/feeds/api/standardfeeds/most_viewed?v=2&max-results=50&time=today"

            Connections {
                onMyChannel: Scripts.loadAccountView()
                onMostRecent: Scripts.loadVideos(homeView.mostRecentFeed, "Most Recent")
                onMostViewed: Scripts.loadVideos(homeView.mostViewedFeed, "Most Viewed")
                onArchive: Scripts.loadArchiveView()
            }
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }

        Loader {
            width: window.width
            height: window.height
        }
    }

    ListView {
        id: windowView

        anchors.fill: window
        model: viewsModel
        orientation: ListView.Horizontal
        interactive: false
        highlightMoveDuration: 300
        onCurrentIndexChanged: windowView.currentItem.visible = true
    }

    states: State {
        name: "portrait"
        when: window.height > window.width
    }
}
