#include "widget.h"

Widget::Widget(QWidget *parent)
    : QWidget(parent)
{
    megaphoneActive = false;
    transmitterActive = (getFMTxProperties("state").toString() == "enabled") ? true : false;
    lastVolumeValue = 100;

    setGeometry(0, 0, 800, 480);

    QDBusConnection::systemBus().connect("com.nokia.FMTx",
                                         "/com/nokia/fmtx/default",
                                         "com.nokia.policy",
                                         "info", this,
                                         SLOT(onTransmitterChanged(QString, const QString &, QStringList)));
    QDBusConnection::systemBus().connect("com.nokia.FMTx",
                                         "/com/nokia/fmtx/default",
                                         "com.nokia.FMTx.Device",
                                         "Changed", this,
                                         SLOT(onFMTxPropChanged()));
    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Coming",
                                         this, SLOT(callIncoming()));

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Created",
                                         this, SLOT(callIncoming()));

    freqLabel = new QPushButton(this);
    freqLabel->setStyleSheet("font:18pt;");
    freqLabel->setGeometry(10, 140, 780, 80);
    freqLabel->setFlat(true);

    connect(freqLabel, SIGNAL(clicked()), this, SLOT(onFreqLabelClicked()));

    _freq = getFMTxProperties("frequency").toUInt();
    setFreqLabel(_freq);

    button1 = new QPushButton("Enable megaphone", this);
    button1->setGeometry(10, 40, 320, 260);
    connect(button1, SIGNAL(clicked()), this, SLOT(toggleMegaphone()));

    button2 = new QPushButton("Enable FM transmitter", this);
    button2->setGeometry(470, 40, 320, 260);
    connect(button2, SIGNAL(clicked()), this, SLOT(toggleTransmitter()));

    volumeSlider = new QSlider(Qt::Horizontal, this);
    volumeSlider->setGeometry(30, 350, 740, 50);
    volumeSlider->setMinimum(0);
    volumeSlider->setMaximum(100);
    volumeSlider->setValue(100);
    connect(volumeSlider, SIGNAL(valueChanged(int)), this, SLOT(onVolumeSliderChanged(int)));

    uint freq_min, freq_max;
    getFreqRange(freq_min, freq_max);

    freqWidget = new FreqWidget(freq_min, freq_max, _freq, this);
    freqWidget->hide();

    connect(freqWidget, SIGNAL(freqChanged(uint)), this, SLOT(setFrequency(uint)));
    connect(freqWidget, SIGNAL(clicked()), freqWidget, SLOT(hide()));
}

Widget::~Widget()
{
    if(megaphoneActive)
       disableMegaphone();
}

void Widget::toggleMegaphone()
{
    if(megaphoneActive)
        megaphoneActive = disableMegaphone();
    else
        megaphoneActive = enableMegaphone();

    button1->setText(megaphoneActive ? "Disable megaphone" : "Enable megaphone");
}

bool Widget::enableMegaphone()
{
    gst_init(NULL, NULL);

    pipeline = gst_pipeline_new("pipeline");
    GstElement *sink;
    source = gst_element_factory_make("pulsesrc", "source");
    sink = gst_element_factory_make("pulsesink", "sink");

    pa_proplist *proplist = pa_proplist_new();
    pa_proplist_sets(proplist, PA_PROP_EVENT_ID, "ringtone-preview");
    g_object_set(sink, "proplist", proplist, NULL);

    gst_bin_add_many(GST_BIN(pipeline), source, sink, NULL);
    if(!gst_element_link(source, sink))
        return false;

    gst_element_set_state(pipeline, GST_STATE_PLAYING);
    return true;
}

bool Widget::disableMegaphone()
{
    setVolume(100);
    volumeSlider->setValue(100);
    gst_element_set_state(pipeline, GST_STATE_NULL);
    gst_object_unref(pipeline);
    return false;
}

void Widget::toggleTransmitter()
{
    if(transmitterActive)
        setFMTxProperties("state", QDBusVariant("disabled"));
    else
        setFMTxProperties("state", QDBusVariant("enabled"));
}

void Widget::callIncoming()
{
    if(megaphoneActive)
    {
        megaphoneActive = disableMegaphone();
        button1->setText("Enable megaphone");
    }
}

void Widget::setFrequency(const uint &freq)
{
    setFMTxProperties("frequency", QDBusVariant(freq));
}

void Widget::setFMTxProperties(const QString &prop, const QDBusVariant &value) //http://maemo.org/api_refs/5.0/5.0-final/fmtx-middleware/fmtx_api.html
{
    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.FMTx",
                                                      "/com/nokia/fmtx/default",
                                                      "org.freedesktop.DBus.Properties",
                                                      "Set");
    QVariantList list;
    list.append("com.nokia.FMTx.device");
    list.append(prop);
    list.append(QVariant::fromValue(value));
    msg.setArguments(list);
    QDBusConnection::systemBus().asyncCall(msg);
}

QVariant Widget::getFMTxProperties(const QString &prop)
{
    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.FMTx",
                                                      "/com/nokia/fmtx/default",
                                                      "org.freedesktop.DBus.Properties",
                                                      "Get");
    QVariantList list;
    list.append("com.nokia.FMTx.device");
    list.append(prop);
    msg.setArguments(list);
    QDBusReply<QVariant> reply = QDBusConnection::systemBus().call(msg);
    if(reply.isValid())
        return reply.value();
    else return QVariant();
}
void Widget::getFreqRange(uint &freq_min, uint &freq_max)
{
    freq_min = getFMTxProperties("freq_min").toUInt();
    freq_max = getFMTxProperties("freq_max").toUInt();

    qDebug() << "freq_min: " << freq_min << " freq_max: " << freq_max;
}

void Widget::setFreqLabel(const uint &freq)
{
    freqLabel->setText("Freq:\n" + QString::number(freq/1000) + "." + QString::number((freq%1000)/10)  + "MHz");
}

void Widget::setVolume(const int &percent)
{
    QtConcurrent::run(this, &Widget::setVolumeThread, percent);
}

void Widget::setVolumeThread(const int &percent)
{
    if(percent <= 100 && percent >= 0)
    {
        QProcess proc;
        proc.start(QString("amixer set Capture %1%").arg(percent));
        proc.waitForFinished();
    }
}


void Widget::onVolumeSliderChanged(const int &value)
{
    if(value > 95)
    {
        setVolume(100);
        lastVolumeValue = 100;
    }
    else if(lastVolumeValue + 5 <= value || lastVolumeValue - 5 >= value)
    {
        setVolume(value);
        lastVolumeValue = value;
    }
}

void Widget::onFreqLabelClicked()
{
    freqWidget->setFreq(_freq);
    freqWidget->show();
}

void Widget::onTransmitterChanged(QString , const QString &msg, QStringList)
{
    transmitterActive = msg.toInt();

    if(transmitterActive)
        button2->setText("Disable FM transmitter");
    else
        button2->setText("Enable FM transmitter");
}

void Widget::onFMTxPropChanged()
{
    _freq = getFMTxProperties("frequency").toUInt();
    setFreqLabel(_freq);
}

