/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/
#include "CDlgMenu.h"
#include "CDlgGps.h"
#include "CDlgGpsSetup.h"
#include "CDlgMap.h"
#include "CDlgWpt.h"
#include "CDlgWptEdit.h"
#include "CDlgWptDel.h"
#include "CDlgTrackEdit.h"
#include "CDlgTrack.h"
#include "CTrack.h"
#include "CTrackDB.h"
#include "CDlgSystem.h"
#include "CWptDB.h"
#include "CWpt.h"
#include "CMenu.h"

#include <QtGui>
#include <gdal.h>
#include <projects.h>

#ifndef _MKSTR_1
#define _MKSTR_1(x)    #x
#define _MKSTR(x)      _MKSTR_1(x)
#endif

#define VER_STR       _MKSTR(VER_MAJOR)"."_MKSTR(VER_MINOR)"."_MKSTR(VER_STEP)

CDlgMenu * CDlgMenu::m_self = 0;

CDlgMenu::CDlgMenu(QWidget * parent)
: QDialog(parent)
{
    m_self = this;
    setupUi(this);
    setupKeys(this);
    setWindowFlags(Qt::WindowStaysOnTopHint | Qt::Dialog);
    show();

    pushKey1->setText(tr("map"));
    connect(pushKey1, SIGNAL(clicked()), this, SLOT(slotKey1()));
    pushKey2->setText(tr(""));
    connect(pushKey2, SIGNAL(clicked()), this, SLOT(slotKey2()));
    pushKey3->setText(tr(""));
    pushKey3->setIcon(QIcon(":/icons/prev_dialog32x32.png"));
    connect(pushKey3, SIGNAL(clicked()), this, SLOT(slotKey3()));
    pushKey4->setText(tr(""));
    pushKey4->setIcon(QIcon(":/icons/next_dialog32x32.png"));
    connect(pushKey4, SIGNAL(clicked()), this, SLOT(slotKey4()));

    labelVersionQL->setText(VER_STR);
    labelVersionQT->setText(qVersion());

    QString gdalVersion = GDALVersionInfo("--version");
    labelVersionGDAL->setText(gdalVersion.split(",")[0]);
    labelVersionPROJ4->setText(QString::number(PJ_VERSION));

    dlgGps = new CDlgGps(stackedWidget);
    stackedWidget->insertWidget(eGps, dlgGps);

    dlgMap = new CDlgMap(stackedWidget);
    stackedWidget->insertWidget(eMap, dlgMap);

    dlgWpt = new CDlgWpt(stackedWidget);
    stackedWidget->insertWidget(eWpt, dlgWpt);

    dlgTrack = new CDlgTrack(stackedWidget);
    stackedWidget->insertWidget(eTrack, dlgTrack);

#ifdef WINCE
    pushReboot->hide();
#else
    spacerKeyboard->changeSize(20,0,QSizePolicy::Fixed,QSizePolicy::Fixed);
#endif
    connect(pushReboot, SIGNAL(clicked()), this, SLOT(slotReboot()));
    connect(pushQuit, SIGNAL(clicked()), this, SLOT(slotRestart()));

    QSettings cfg;
    setPage(cfg.value("environment/dialog", eGps).toInt());
}


CDlgMenu::~CDlgMenu()
{
    m_self = 0;
    QSettings cfg;
    cfg.setValue("environment/dialog", stackedWidget->currentIndex());
}


void CDlgMenu::slotRestart()
{
    qApp->quit();
}


void CDlgMenu::slotReboot()
{
#ifdef WINCE
#else
    system("reboot");
#endif

}


void CDlgMenu::slotKey1()
{
    close();
}


void CDlgMenu::slotKey2()
{
    int idx = stackedWidget->currentIndex();
    if(idx == eGps) {
        CDlgGpsSetup dlg(this);
        dlg.exec();
    }
    else if(idx == eAbout) {
        CDlgSystem dlg(this);
        dlg.exec();
    }
    else if(idx == eWpt) {

        CMenu menu(this);

        int y = 10;
        int x = (width() - menu.width()) >> 1;

        QString key = dlgWpt->getCurrentKey();
        if(!key.isEmpty()){
            menu.addAction(5, tr("edit"), QIcon(":/icons/edit64x64.png"), this, SLOT(slotWptEdit()));
            menu.addAction(1, tr("delete"), QIcon(":/icons/trash64x64.png"), this, SLOT(slotWptDelete()));
        }
        menu.addAction(7, tr("del. all"), QIcon(":/icons/trash64x64.png"), this, SLOT(slotWptDeleteAll()));
        menu.addAction(4, tr("back"), QIcon(":/icons/back64x64.png"));
        menu.exec(QPoint(x,y));
    }
    else if(idx == eTrack) {
        CMenu menu(this);

        int y = 10;
        int x = (width() - menu.width()) >> 1;

        QString key = dlgTrack->getCurrentKey();
        if(!key.isEmpty()){
            menu.addAction(5, tr("edit"), QIcon(":/icons/edit64x64.png"), this, SLOT(slotTrkEdit()));
            menu.addAction(1, tr("delete"), QIcon(":/icons/trash64x64.png"), this, SLOT(slotTrkDelete()));
        }
        menu.addAction(7, tr("del. all"), QIcon(":/icons/trash64x64.png"), this, SLOT(slotTrkDeleteAll()));
        menu.addAction(4, tr("back"), QIcon(":/icons/back64x64.png"));

        menu.exec(QPoint(x,y));

    }
}


void CDlgMenu::slotKey3()
{
    int idx = stackedWidget->currentIndex() - 1;
    if(idx < 0) idx = eMaxPage - 1;

    setPage(idx);
}


void CDlgMenu::slotKey4()
{
    int idx = stackedWidget->currentIndex() + 1;
    if(idx >= eMaxPage) idx = 0;

    setPage(idx);
}


void CDlgMenu::setPage(int idx)
{
    stackedWidget->setCurrentIndex(idx);
    if(idx == eAbout) {
        pushKey2->setText(tr("system"));
        pushKey2->setIcon(QIcon(""));
    }
    else if(idx == eGps) {
        pushKey2->setText(tr("setup"));
        pushKey2->setIcon(QIcon(""));
    }
    else if(idx == eWpt) {
        pushKey2->setText(tr(""));
        pushKey2->setIcon(QIcon(":/icons/menu32x32.png"));
    }
    else if(idx == eTrack) {
        pushKey2->setText(tr(""));
        pushKey2->setIcon(QIcon(":/icons/menu32x32.png"));
    }
    else {
        pushKey2->setText("");
        pushKey2->setIcon(QIcon(""));
    }
    update();
}


void CDlgMenu::paintEvent(QPaintEvent * e)
{
    QPainter p;
    p.begin(this);
    p.end();

    QDialog::paintEvent(e);
}

void CDlgMenu::slotWptDelete()
{
    QString key = dlgWpt->getCurrentKey();
    if(key.isEmpty()) return;

    CWpt wpt;
    QString errmsg;

    if(CWptDB::self().get(key, wpt, errmsg)){
        QMessageBox::StandardButton res = QMessageBox::question(this, tr("Delete waypoint..."), tr("Really delete: %1").arg(wpt.name), QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok);
        if(res != QMessageBox::Ok){
            return;
        }

        QString errmsg;
        if(!CWptDB::self().del(wpt.key(), errmsg)){
            QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
            return;
        }
    }
    else{
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
    }
}

void CDlgMenu::slotWptDeleteAll()
{
    CDlgWptDel dlg(this);
    dlg.exec();
}

void CDlgMenu::slotWptEdit()
{

    QString key = dlgWpt->getCurrentKey();
    if(key.isEmpty()) return;

    CWpt wpt;
    QString errmsg;

    if(CWptDB::self().get(key, wpt, errmsg)){
        CDlgWptEdit dlg(wpt, this);
        dlg.exec();
    }
    else{
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
    }
}

void CDlgMenu::slotTrkDelete()
{
    QString errmsg;
    QString key = dlgTrack->getCurrentKey();
    if(key.isEmpty()) return;

    CTrack track;
    if(!CTrackDB::self().getTrackInfo(key, track, errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }

    QMessageBox::StandardButton res = QMessageBox::question(this, tr("Delete track..."), tr("Really delete: %1").arg(track.name), QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok);
    if(res != QMessageBox::Ok){
        return;
    }

    if(!CTrackDB::self().del(track.trackid, errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }

}

void CDlgMenu::slotTrkDeleteAll()
{
    QString errmsg;

    QMessageBox::StandardButton res = QMessageBox::question(this, tr("Delete track..."), tr("Really delete all tracks?"), QMessageBox::Ok|QMessageBox::Cancel, QMessageBox::Ok);
    if(res != QMessageBox::Ok){
        return;
    }

    if(!CTrackDB::self().delAll(errmsg)){
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
        return;
    }

}

void CDlgMenu::slotTrkEdit()
{
    QString key = dlgTrack->getCurrentKey();
    if(key.isEmpty()) return;

    CTrack trk;
    QString errmsg;

    if(CTrackDB::self().get(key, trk, errmsg)){
        CDlgTrackEdit dlg(trk, this);
        dlg.exec();
    }
    else{
        QMessageBox::critical(this, tr("Error..."), tr("Database error: %1").arg(errmsg), QMessageBox::Abort, QMessageBox::Abort);
    }
}

