/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/
#include "CMapWorld.h"

#include <QtGui>
#include <projects.h>
#include <math.h>

CMapWorld::CMapWorld(QObject * parent)
: IMap(parent)
, zoomfactor(1.0)
, xscale(scales[6])
, yscale(-scales[6] / 2.116)
, x(0)
, y(0)
, xref1(-180.0 * DEG_TO_RAD)
, yref1(85.0511 *DEG_TO_RAD)
, xref2(180.0 * DEG_TO_RAD)
, yref2(-85.0511 *DEG_TO_RAD)
{

}


CMapWorld::~CMapWorld()
{

}


void CMapWorld::resize(const QSize& s)
{
    IMap::resize(s);
    buffer.fill(QColor("#ffffcc"));

    qreal width     = buffer.width()  * xscale * zoomfactor;
    qreal height    = buffer.height() * yscale * zoomfactor;
    viewport        = QRectF(x, y, width, -height);

}


void CMapWorld::draw(QPainter& p)
{
    if(needsRedraw) {
        qreal width     = buffer.width()  * xscale * zoomfactor;
        qreal height    = buffer.height() * yscale * zoomfactor;
        viewport        = QRectF(x, y, width, -height);
        needsRedraw     = false;
    }
    p.drawPixmap(0,0,buffer);
}


void CMapWorld::convertRad2Pt(qreal& u, qreal& v)
{
    u = (u - x) / (xscale * zoomfactor);
    v = (v - y) / (yscale * zoomfactor) + buffer.height();
}


void CMapWorld::convertPt2Rad(qreal& u, qreal& v)
{
    u = x + u * xscale * zoomfactor;
    v = y + (v - buffer.height()) *  yscale * zoomfactor;
}


void CMapWorld::move(const QPoint& old, const QPoint& next)
{
    qreal u1 = old.x();
    qreal v1 = old.y();
    convertPt2Rad(u1,v1);

    qreal u2 = next.x();
    qreal v2 = next.y();
    convertPt2Rad(u2,v2);

    qDebug() << u1 << v1 <<u2 << v2;

    // move top left point by difference
    x += u1 - u2;
    y += v1 - v2;

    qreal width     = buffer.width()  * xscale * zoomfactor;
    qreal height    = buffer.height() * yscale * zoomfactor;
    viewport        = QRectF(x, y, width, -height);

    needsRedraw = true;

}


int CMapWorld::zoom(int idx)
{

    if(idx < 0) idx = 0;
    if(idx >= numScales) idx = numScales - 1;

    qreal desired = scales[idx];

    zoomfactor  = desired/xscale;
    if(zoomfactor >= 1.0) {
        zoomfactor = (int)(zoomfactor + 0.5);
    }

    needsRedraw = true;

    return idx;
}
