/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "engine.h"
#include "qimsysdebug.h"
#include <QIcon>
#include "translator.h"
#include <QSettings>

class Engine::Private : private QObject
{
    Q_OBJECT
public:
    Private(Engine *parent);
    ~Private();

private:
    Engine *q;

public:
    QString inputString;
    QimsysConversionItemList conversionList;
    QimsysConversionItemList candidateList;
    QimsysConversionItemList predictionList;
};

Engine::Private::Private(Engine *parent)
    : QObject(parent)
    , q(parent)
{
    qimsysDebugIn() << parent;
    q->setAvailable(true);
    q->setCategoryType(CanBeNone);
    TR(q, "categoryName", this, QT_TR_NOOP("Input/Conversion Engine"));
    q->setIcon(QIcon(":/engine.png"));
    TR(q, "name", this, QT_TR_NOOP("Example"));
    TR(q, "author", this, QT_TR_NOOP("Tasuku Suzuki"));
    TR(q, "translator", this, QT_TR_NOOP("None"));
    qimsysDebugOut();
}

Engine::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

Engine::Engine(QObject *parent)
    : QimsysEngine(parent)
{
    qimsysDebugIn() << parent;
    d = new Private(this);
    qimsysDebugOut();
}

Engine::~Engine()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

QString Engine::identifier() const
{
    return QLatin1String("engine");
}

void Engine::setRawString(const QString &rawString)
{
    Q_UNUSED(rawString);
}

void Engine::setInputString(const QString &inputString)
{
    if (d->inputString == inputString) return;
    qimsysDebugIn() << inputString;
    d->inputString = inputString;

    d->predictionList.clear();
    if (!d->inputString.isEmpty()) {
        for (int i = 0; i < 5; i++) {
            QimsysConversionItem item;
            item.from = inputString;
            item.to = QString("%1(%2)").arg(inputString).arg(i);
            item.index = i;
            d->predictionList.append(item);
        }
    }
    emit predicted(d->predictionList);
    qimsysDebugOut();
}

void Engine::convert()
{
    qimsysDebugIn();
    d->conversionList.clear();
    QString inputString = d->inputString;
    while (!inputString.isEmpty()) {
        int len = qMin(inputString.length(), qrand() % 4 + 1);
        QString from = inputString.left(len);
        inputString = inputString.mid(len);
        QimsysConversionItem item;
        item.index = 0;
        item.from = from;
        item.to = QString("%1<%2>").arg(from).arg(len);
        d->conversionList.append(item);
    }
    emit conversionsChanged(d->conversionList);
    qimsysDebugOut();
}

void Engine::resize(int index, int delta)
{
    if (index > d->conversionList.count() - 1) return;
    if (delta > 0) {
        if (index == d->conversionList.count() - 1) return;
    } else {
        if (d->conversionList[index].from.length() + delta < 1) return;
    }
    qimsysDebugIn() << index << delta;

    while (d->conversionList.count() > index + 1) {
        d->conversionList.removeLast();
    }
    QimsysConversionItem last = d->conversionList.takeLast();
    QString inputString = d->inputString;
    foreach(const QimsysConversionItem &item, d->conversionList) {
        inputString = inputString.mid(item.from.length());
    }
    bool first = true;
    while (!inputString.isEmpty()) {
        int len = qMin(inputString.length(), qrand() % 4 + 1);
        if (first) {
            len = last.from.length() + delta;
            first = false;
        }
        QString from = inputString.left(len);
        inputString = inputString.mid(len);
        QimsysConversionItem item;
        item.index = 0;
        item.from = from;
        item.to = QString("%1<%2>").arg(from).arg(len);
        d->conversionList.append(item);
    }
    emit conversionsChanged(d->conversionList);
    qimsysDebugOut();
}

QimsysConversionItemList Engine::candidates(int index)
{
    d->candidateList.clear();
    if (index < 0) return d->candidateList;
    if (index > d->conversionList.count() - 1) return d->candidateList;

    qimsysDebugIn() << index;
    QString from = d->conversionList[index].from;
    for (int i = 0; i < 5; i++) {
        QimsysConversionItem item;
        item.index = i;
        item.from = from;
        item.to = QString("%1{%2}").arg(from).arg(i);
        d->candidateList.append(item);
    }
    qimsysDebugOut() << d->candidateList;
    return d->candidateList;
}

void Engine::commit(const QimsysConversionItemList &result)
{
    qimsysDebugIn() << result;
    if (!d->conversionList.isEmpty()) {
        d->conversionList.clear();
        d->candidateList.clear();
    } else if (!d->predictionList.isEmpty()) {
        d->predictionList.clear();
        emit predicted(d->predictionList);
    }
    qimsysDebugOut();
}

#include "engine.moc"
