#include <QtGui/qpainter.h>
#include <QSettings>
#include <QAction>
#include <QDebug>
#include <QMenuBar>
#include <QCoreApplication>
#include <QTranslator>
#include <QLocale>


#include "qflipclock-standalone.h"
#include "settings.h"
#include "config.h"

#ifdef Q_WS_MAEMO_5
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

static QDBusConnection dBusConnection = QDBusConnection::systemBus();
#endif

QFlipClock_App::QFlipClock_App()
        : QFlipClock()
{
    QString localeName = QLocale::system().name();

    QTranslator translator;
    translator.load("qflipclock_" + localeName,
                    TRANSLATIONS_DIR);
    QCoreApplication::installTranslator(&translator);

    setWindowTitle(tr("Flip Clock"));
#ifdef Q_WS_MAEMO_5
    _dBusInterface = new QDBusInterface(MCE_SERVICE, MCE_REQUEST_PATH,
                                       MCE_REQUEST_IF, dBusConnection, this);

    dBusConnection.connect(MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                           MCE_DISPLAY_SIG, this, SLOT(screenChange(const QDBusMessage &)));
#endif
    QSettings settings(ORGANIZATION_NAME, APPLICATION_NAME);
    toggleSize();
    setTransition(TransitionsEnum(
            settings.value("Main/Transition", Transition_Flip).toInt()));

    if (settings.value("Main/KeepBacklightOn", true).toBool())
        startBacklightOn();

    QAction *settingsAction = new QAction(tr("&Settings"), this);
    connect(settingsAction, SIGNAL(triggered()), SLOT(showSettingsDialog()));

#ifdef Q_WS_MAEMO_5
    menuBar()->addAction(settingsAction);
    setContextMenuPolicy(Qt::ActionsContextMenu);
#else
    addAction(settingsAction);
    setContextMenuPolicy(Qt::ActionsContextMenu);
#endif
}

void QFlipClock_App::showSettingsDialog()
{
    Settings dlg;
    if (dlg.exec() == QDialog::Accepted)
    {
        setNewTransition(dlg.getTransition());
        setNewTheme(dlg.getTheme());
        setNewKeepBacklight(dlg.getKeepBacklight());
        dlg.close();
    }
}

void QFlipClock_App::setNewTransition(TransitionsEnum transition)
{
    QSettings  settings(ORGANIZATION_NAME, APPLICATION_NAME);
    settings.setValue("Main/Transition", transition);
    setTransition(transition);
}

void QFlipClock_App::setNewTheme(QString name)
{
    QSettings settings(ORGANIZATION_NAME, APPLICATION_NAME);
    settings.setValue("Main/Theme", name);
    settings.sync();
    updateTheme();
    repaint();
}

void QFlipClock_App::setNewKeepBacklight(bool value)
{
    QSettings  settings(ORGANIZATION_NAME, APPLICATION_NAME);
    settings.setValue("Main/KeepBacklightOn", value);
    if (value)
        startBacklightOn();
    else
        stopBacklightOn();
}


#ifdef Q_WS_MAEMO_5
void QFlipClock_App::screenChange(const QDBusMessage &message)
{
    QString state = message.arguments().at(0).toString();
    if (!state.isEmpty())
    {
        if (state == MCE_DISPLAY_OFF_STRING)
        {
            stopTimer();
        }
        else if (state == MCE_DISPLAY_ON_STRING)
        {
            startTimer();
            updateTime(true);
        }
    }
}

void QFlipClock_App::startBacklightOn()
{
    if (!_backlightTimer.isActive())
        _backlightTimer.start(30000, this);
}

void QFlipClock_App::stopBacklightOn()
{
    if (_backlightTimer.isActive())
        _backlightTimer.stop();
}

void QFlipClock_App::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == _backlightTimer.timerId())
    {
        if (_dBusInterface->isValid())
        {
            _dBusInterface->call("req_display_blanking_pause");
        }
    }
    else
    {
        // Propagate event to parent
        event->ignore();
        QFlipClock::timerEvent(event);
    }
}

#endif
