#include "urlretriever.h"
#include "networkaccessmanager.h"
#include "pluginmanager.h"
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QFile>

UrlRetriever* UrlRetriever::m_instance = 0;

UrlRetriever::UrlRetriever() :
    QObject(),
    m_reply(0),
    m_index(0),
    m_total(0),
    m_cancelled(false)
{
    if (!m_instance) {
        m_instance = this;
    }
}

UrlRetriever::~UrlRetriever() {
    if (m_reply) {
        m_reply->deleteLater();
    }
}

UrlRetriever* UrlRetriever::instance() {
    return !m_instance ? new UrlRetriever : m_instance;
}

int UrlRetriever::progress() const {
    return (m_index <= 0) || (m_total <= 0) ? 0 : m_index * 100 / m_total;
}

void UrlRetriever::addUrlToQueue(const QUrl &url) {
    m_cancelled = false;
    m_urlQueue.enqueue(url);
    m_total = m_urlQueue.size();

    if (m_total == 1) {
        emit busy(tr("Retrieving URLs"), 100);
        emit progressChanged(0);
        this->getWebPage(m_urlQueue.dequeue());
    }
}

void UrlRetriever::addUrlToQueue(const QString &url) {
    m_cancelled = false;
    m_urlQueue.enqueue(url);
    m_total = m_urlQueue.size();

    if (m_total == 1) {
        emit busy(tr("Retrieving URLs"), 100);
        emit progressChanged(0);
        this->getWebPage(m_urlQueue.dequeue());
    }
}

void UrlRetriever::addUrlsToQueue(QList<QUrl> urls) {
    m_cancelled = false;

    foreach (QUrl url, urls) {
        m_urlQueue.enqueue(url);
    }

    m_total = m_urlQueue.size();

    if (m_total == urls.size()) {
        emit busy(tr("Retrieving URLs"), 100);
        emit progressChanged(0);
        this->getWebPage(m_urlQueue.dequeue());
    }
}

void UrlRetriever::addUrlsToQueue(QStringList urls) {
    m_cancelled = false;

    foreach (QString url, urls) {
        m_urlQueue.enqueue(url);
    }

    m_total = m_urlQueue.size();

    if (m_total == urls.size()) {
        emit busy(tr("Retrieving URLs"), 100);
        emit progressChanged(0);
        this->getWebPage(m_urlQueue.dequeue());
    }
}

void UrlRetriever::parseUrlsFromText(const QString &text) {
    QStringList urlStrings = text.split(QRegExp("\\s"), QString::SkipEmptyParts);
    this->addUrlsToQueue(urlStrings);
}

void UrlRetriever::importUrlsFromTextFile(const QString &filePath) {
    QFile textFile(filePath);

    if (textFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QString urlString = textFile.readAll();
        textFile.close();
        this->parseUrlsFromText(urlString);
    }
}

void UrlRetriever::cancel() {
    if (!m_cancelled) {
        m_cancelled = true;

        if (m_reply) {
            m_reply->deleteLater();
            m_reply = 0;
        }

        m_urlQueue.clear();
        m_index = 0;

        emit cancelled();
    }
}

void UrlRetriever::clearResults() {
    m_results.clear();
}

void UrlRetriever::getWebPage(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux i686) AppleWebKit/537.22 (KHTML, like Gecko) Ubuntu Chromium/25.0.1364.160 Chrome/25.0.1364.160 Safari/537.22");
    m_reply = NetworkAccessManager::instance()->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(checkWebPage()));
}

void UrlRetriever::checkWebPage() {
    if (!m_reply) {
        if (!m_cancelled) {
            this->cancel();
        }

        return;
    }

    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        redirect = m_reply->header(QNetworkRequest::LocationHeader).toUrl();
    }

    if (!redirect.isEmpty()) {
        if (redirect.scheme().isEmpty()) {
            redirect.setScheme(m_reply->request().url().scheme());
        }

        if (redirect.host().isEmpty()) {
            redirect.setHost(m_reply->request().url().host());
        }

        m_reply->deleteLater();
        m_reply = 0;
        this->getWebPage(redirect);
        return;
    }

    QString response(m_reply->readAll());
    QRegExp re("http(s|)://[^'\"<\\s]+");
    int pos = 0;

    while ((pos = re.indexIn(response, pos)) != -1) {
        QUrl url(re.cap(0));

        if (PluginManager::instance()->servicePluginExists(url)) {
            m_results.append(url);
        }

        pos += re.matchedLength();
    }

    m_reply->deleteLater();
    m_reply = 0;

    m_index++;
    emit progressChanged(this->progress());

    if ((!m_urlQueue.isEmpty()) && (!m_cancelled)) {
        this->getWebPage(m_urlQueue.dequeue());
    }
    else {
        emit finished();
        m_index = 0;
        m_total = 0;
    }
}
