#ifndef PLUGINMANAGER_H
#define PLUGINMANAGER_H

#include <QObject>
#include <QQueue>
#include <QStringList>

class ServicePlugin;
class DecaptchaPlugin;
class RecaptchaPlugin;
class QUrl;

class PluginManager : public QObject
{
    Q_OBJECT

    Q_PROPERTY(int progress
               READ progress
               NOTIFY progressChanged)

public:
    ~PluginManager();

    inline int progress() const { return m_progress; }

    bool servicePluginExists(const QString &serviceName) const;
    bool servicePluginExists(const QUrl &url) const;
    ServicePlugin* getServicePlugin(const QString &serviceName) const;
    ServicePlugin* getServicePlugin(const QUrl &url) const;
    ServicePlugin* createServicePlugin(const QString &serviceName);
    ServicePlugin* createServicePlugin(const QUrl &url);
    inline QList<ServicePlugin*> servicePlugins() const { return m_services; }
    inline QStringList servicePluginNames() const { return m_serviceNames; }

    DecaptchaPlugin* getDecaptchaPlugin(const QString &serviceName) const;
    DecaptchaPlugin* createDecaptchaPlugin(const QString &serviceName);
    inline QList<DecaptchaPlugin*> decaptchaPlugins() const { return m_decaptchaServices; }
    inline QStringList decaptchaPluginNames() const { return m_decaptchaNames; }

    RecaptchaPlugin* getRecaptchaPlugin(const QString &serviceName) const;
    RecaptchaPlugin* createRecaptchaPlugin(const QString &serviceName);
    inline QList<RecaptchaPlugin*> recaptchaPlugins() const { return m_recaptchaServices; }
    inline QStringList recaptchaPluginNames() const { return m_recaptchaNames; }

    static PluginManager* instance();

public slots:
    void loadPlugins();
    void cancel();
    
private:
    PluginManager();

    void loadServicePlugins();
    void loadDecaptchaPlugins();
    void loadRecaptchaPlugins();
    void loginToAccounts();

private slots:
    void onAccountLogin(bool ok);

signals:
    void busy(const QString &message, int numberOfOperations);
    void progressChanged(int progress);
    void cancelled();
    void pluginsReady();
    
private:
    static PluginManager *m_instance;

    QList<ServicePlugin*> m_services;
    QStringList m_serviceNames;

    QList<DecaptchaPlugin*> m_decaptchaServices;
    QStringList m_decaptchaNames;

    QList<RecaptchaPlugin*> m_recaptchaServices;
    QStringList m_recaptchaNames;

    QList<QString> m_paths;
    QQueue<ServicePlugin*> m_loginQueue;
    int m_progress;
    bool m_cancelled;
};

#endif // PLUGINMANAGER_H
