#include "solvemedia.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>

using namespace QtJson;

SolveMedia::SolveMedia(QObject *parent) :
    RecaptchaPlugin(parent)
{
}

void SolveMedia::getCaptcha(const QString &key) {
    m_key = key;
    QUrl url("http://api.solvemedia.com/papi/_challenge.js");
    url.addQueryItem("k", key);

    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaDownloaded()));
}

void SolveMedia::onCaptchaDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode != 200) {
        this->reportError(statusCode);
    }
    else {
        QString response(reply->readAll());
        bool ok = false;
        QVariantMap map = Json::parse(response, ok).toMap();

        if (ok) {
            QString challenge = map.value("ACChallengeResult").toMap().value("chid").toString();

            if (challenge.isEmpty()) {
                emit error(CaptchaNotFound);
            }
            else {
                this->downloadCaptchaImage(challenge);
            }
        }
        else {
            emit error(CaptchaNotFound);
        }
    }

    reply->deleteLater();
}

void SolveMedia::downloadCaptchaImage(const QString &challenge) {
    m_challenge = challenge;
    QUrl url("http://api.solvemedia.com/papi/media");
    url.addQueryItem("c", challenge);

    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaImageDownloaded()));
}

void SolveMedia::onCaptchaImageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        emit gotCaptcha(reply->readAll());
    }
    else {
        this->reportError(statusCode);
    }

    reply->deleteLater();
}

void SolveMedia::reportError(int errorCode) {
    switch (errorCode) {
    case 404:
        emit error(CaptchaNotFound);
        break;
    case 503:
        emit error(ServiceUnavailable);
        break;
    case 500:
        emit error(InternalError);
        break;
    case 403:
        emit error(Unauthorised);
        break;
    default:
        emit error(UnknownError);
    }
}

Q_EXPORT_PLUGIN2(solvemedia, SolveMedia)
