#include "mediafire.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QRegExp>

using namespace QtJson;

MediaFire::MediaFire(QObject *parent) :
    ServicePlugin(parent)
{
}

QRegExp MediaFire::urlPattern() const {
    return QRegExp("http(s|)://(www.|)mediafire.com/\\?\\w+", Qt::CaseInsensitive);
}

bool MediaFire::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void MediaFire::login(const QString &username, const QString &password) {
    QString data = QString("login_email=%1&login_pass=%2").arg(username).arg(password);
    QUrl url("http://www.mediafire.com/dynamic/login.php");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void MediaFire::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn(statusCode == 200);

    reply->deleteLater();
}

void MediaFire::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void MediaFire::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();
    QRegExp re("http://\\d+.\\d+.\\d+.\\d+/\\w+/\\w+/[^'\"]+");

    if ((!redirect.isEmpty()) && (re.indexIn(redirect) == -1)) {
        this->checkUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (response.contains("thisfolder_root")) {
            this->getFolderFileLinks(reply->request().url().toString().section('?', -1));
        }
        else {
            QString fileName = response.section("<title>", 1, 1).section('<', 0, 0);

            if (fileName.isEmpty()) {
                emit urlChecked(false);
            }
            else {
                emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
            }
        }
    }

    reply->deleteLater();
}

void MediaFire::getFolderFileLinks(const QString &folderId) {
    QUrl url("http://www.mediafire.com/api/folder/get_content.php");
    url.addQueryItem("content_type", "files");
    url.addQueryItem("order_by", "name");
    url.addQueryItem("order_direction", "asc");
    url.addQueryItem("chunk", "1");
    url.addQueryItem("version", "2");
    url.addQueryItem("folder_key", folderId);
    url.addQueryItem("response_format", "json");
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFolderFileLinks()));
}

void MediaFire::checkFolderFileLinks() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString response(reply->readAll());
    QVariantMap folder = Json::parse(response).toMap();
    QVariantList files = folder.value("response").toMap().value("folder_content").toMap().value("files").toList();

    if (files.isEmpty()) {
        emit urlChecked(false);
    }
    else {
        while (!files.isEmpty()) {
            QVariantMap file = files.takeFirst().toMap();
            QString id = file.value("quickkey").toString();
            QString fileName = file.value("filename").toString();

            if ((!id.isEmpty()) && (!fileName.isEmpty())) {
                QUrl url("http://www.mediafire.com/?" + id);
                emit urlChecked(true, url, this->serviceName(), fileName, files.isEmpty());
            }
            else if (files.isEmpty()) {
                emit urlChecked(false);
            }
        }
    }

    reply->deleteLater();
}

void MediaFire::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    m_url = webUrl;
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onWebPageDownloaded()));
}

void MediaFire::onWebPageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\d+.\\d+.\\d+.\\d+/\\w+/\\w+/[^'\"]+");
    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (re.indexIn(redirect) == 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (!redirect.isEmpty()) {
        this->getDownloadUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (re.indexIn(response) >= 0) {
            QNetworkRequest request;
            request.setUrl(QUrl(re.cap()));
            emit downloadRequestReady(request);
        }
        else {
            m_captchaKey = response.section("http://api.solvemedia.com/papi/challenge.script?k=", 1, 1).section('"', 0, 0);

            if (m_captchaKey.isEmpty()) {
                emit error(UnknownError);
            }
            else {
                emit statusChanged(CaptchaRequired);
            }
        }
    }

    reply->deleteLater();
}

void MediaFire::submitCaptchaResponse(const QString &challenge, const QString &response) {
    QString data = QString("adcopy_challenge=%1&adcopy_response=%2").arg(challenge).arg(response);
    QNetworkRequest request(m_url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaSubmitted()));
}

void MediaFire::onCaptchaSubmitted() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\d+.\\d+.\\d+.\\d+/\\w+/\\w+/[^'\"]+");
    QString response(reply->readAll());

    if (re.indexIn(response) >= 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (response.contains("Your entry was incorrect")) {
        emit error(CaptchaError);
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(mediafire, MediaFire)
