#include "dailymotion.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QRegExp>
#include <QSettings>

using namespace QtJson;

Dailymotion::Dailymotion(QObject *parent) :
    ServicePlugin(parent)
{
    m_formatList << "720" << "480" << "380";
}

QRegExp Dailymotion::urlPattern() const {
    return QRegExp("(http(s|)://(www.|)dailymotion.com/(video|playlist)/|http://dai.ly/)\\w{6}", Qt::CaseInsensitive);
}

bool Dailymotion::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void Dailymotion::checkUrl(const QUrl &webUrl) {
    QString urlString = webUrl.toString();
    QRegExp re("\\w{6,6}(_|$)");
    re.indexIn(urlString);
    QString id = re.cap();
    QUrl url;

    if (urlString.contains("/playlist/")) {
        url.setUrl(QString("https://api.dailymotion.com/playlist/%1/videos").arg(id));
        url.addQueryItem("limit", "100");
    }
    else {
        url.setUrl("https://api.dailymotion.com/video/" + id);
    }

    url.addQueryItem("fields", "title,url");
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void Dailymotion::checkPlaylistVideoUrls(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void Dailymotion::checkUrlIsValid() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString response(reply->readAll());
    QVariantMap result = Json::parse(response).toMap();

    if (reply->request().url().hasQueryItem("limit")) {
        QVariantList videos = result.value("list").toList();
        bool moreResults = result.value("has_more").toBool();

        if (videos.isEmpty()) {
            emit urlChecked(false);
        }
        else {
            while (!videos.isEmpty()) {
                QVariantMap video = videos.takeFirst().toMap();
                QUrl url(video.value("url").toString());
                QString title = video.value("title").toString().trimmed();
                emit urlChecked((url.isValid()) && (!title.isEmpty()), url, this->serviceName(), title + ".mp4", (videos.isEmpty()) && (!moreResults));
            }

            if (moreResults) {
                QString urlString = reply->request().url().toString();
                QUrl playlistUrl(urlString.section("&page=", 0, 0));
                playlistUrl.addQueryItem("page", QString::number(result.value("page").toInt() + 1));
                this->checkPlaylistVideoUrls(playlistUrl);
            }
        }
    }
    else {
        QUrl url(result.value("url").toString());
        QString title = result.value("title").toString().trimmed();
        emit urlChecked((url.isValid()) && (!title.isEmpty()), url, this->serviceName(), title + ".mp4");
    }

    reply->deleteLater();
}

void Dailymotion::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    QString id = webUrl.toString().section('/', -1).left(6);
    QUrl url("http://www.dailymotion.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void Dailymotion::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(QByteArray::fromPercentEncoding(reply->readAll()).replace("\\", ""));
    QString manifestUrl = response.section("autoURL\":\"", 1, 1).section('"', 0, 0);

    if (manifestUrl.startsWith("http://")) {
        this->getFormatList(QUrl(manifestUrl));
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

void Dailymotion::getFormatList(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFormatList()));
}

void Dailymotion::checkFormatList() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QVariantMap formats = Json::parse(response).toMap();
    QVariantList formatList = formats.value("alternates").toList();
    QString preferredFormat = QSettings("QDL", "QDL").value("Dailymotion/videoFormat", "380").toString();
    QString manifestUrl;
    int i = m_formatList.indexOf(preferredFormat);
    int ii = 0;

    while ((manifestUrl.isEmpty()) && (i < m_formatList.size())) {
        while ((manifestUrl.isEmpty()) && (ii < formatList.size())) {
            QVariantMap map = formatList.at(ii).toMap();
            QString name = map.value("name").toString();

            if (name == m_formatList.at(i)) {
                manifestUrl = map.value("template").toString();
            }

            ii++;
        }

        ii = 0;
        i++;
    }

    if (manifestUrl.startsWith("http://")) {
        this->getVideoUrl(QUrl(manifestUrl));
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

void Dailymotion::getVideoUrl(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVideoUrl()));
}

void Dailymotion::checkVideoUrl() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QVariantMap map = Json::parse(response).toMap();
    QString urlTemplate = map.value("template").toString().remove("frag($fragment$)/");

    if (urlTemplate.isEmpty()) {
        emit error(UnknownError);
    }
    else {
        QString host = reply->request().url().host();
        QUrl url(QString("http://%1%2").arg(host).arg(urlTemplate));
        emit downloadRequestReady(QNetworkRequest(url));
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(dailymotion, Dailymotion)
