/*
# =============================================================================
# Rtree spatial index. Copyright (C) 2007 Sean C. Gillies
#
# This library is free software; you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation; either version 2.1 of the License, or (at your option)
# any later version.
#
# This library is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License 
# along with this library; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contact email: sgillies@frii.com
# =============================================================================
*/

#include "gispyspatialindex.h"
#include "Python.h"
#include <SpatialIndex/SpatialIndex.h>

using namespace SpatialIndex;

class PyListVisitor : public IVisitor
{
public:
    PyListVisitor(PyObject *o, bool region = false)
    {
        ids = o;
	incl_reg = region;
        Py_INCREF(ids);
    }

    ~PyListVisitor()
    {
        Py_DECREF(ids);
    }

    void visitNode(const INode & n) {}

    void visitData(const IData & d)
    {
        SpatialIndex::Region region;
        SpatialIndex::IShape *shape = NULL;

        PyObject *tp = NULL;
        PyObject *tpregion = NULL;
        PyObject *idx = NULL;



	idx = PyLong_FromLongLong(d.getIdentifier());

	if(idx == NULL)
		goto error;

	if(incl_reg) {
		d.getShape(&shape);

		if(shape == NULL)
			goto error;

		shape->getMBR(region);

		if(region.getLow(0) == region.getHigh(0) &&
			region.getLow(1) == region.getHigh(1))
			tpregion = Py_BuildValue("(dd)",
				region.getLow(0), region.getLow(1));
		else
			tpregion = Py_BuildValue("(dddd)", region.getLow(0),
				region.getLow(1), region.getHigh(0),
				region.getHigh(1));

		if(tpregion == NULL)
			goto error;

		tp = PyTuple_New(2);

		if(tp == NULL)
			goto error;

		PyTuple_SetItem(tp, 0, idx);
		PyTuple_SetItem(tp, 1, tpregion);

		PyList_Append(ids, tp);

	} else {

		PyList_Append(ids, idx);
	}

	if(shape)
		delete shape;

	return;

error:
	Py_XDECREF(tp);
	Py_XDECREF(tpregion);
	Py_XDECREF(idx);

	if(shape)
		delete shape;
    }

    void visitData(std::vector<const IData*>& v) {}

private:
    PyObject *ids;
    bool incl_reg;
};


extern "C"
GISPySpatialIndex *
RtreeIndex_new(char* filename, unsigned long nPageLength, int load)
{
    if (!filename)
        return new GISPySpatialIndex;
    else
    {
        if (load == 1)
        {
            return new GISPySpatialIndex(filename);
        }
        else
        {
            if (!nPageLength) nPageLength=4096;
            return new GISPySpatialIndex(filename, nPageLength);
        }
    }
}

extern "C"
void
RtreeIndex_del(GISPySpatialIndex *index)
{
    delete index;
}

extern "C"
int
RtreeIndex_insertData(GISPySpatialIndex *index, long id,
                      double *min, double *max)
{
    try {	
        index->index().insertData(0, 0, SpatialIndex::Region(min, max, 2), id);
        return 1;
    }
    catch (Tools::Exception& e) {
        PyErr_SetString(PyExc_TypeError, e.what().c_str());
        return 0;
    }
}

extern "C"
int
RtreeIndex_deleteData(GISPySpatialIndex *index, long id,
                      double *min, double *max)
{
    try {	
        index->index().deleteData(SpatialIndex::Region(min, max, 2), id);
        return 1;
    }
    catch (Tools::Exception& e) {
        PyErr_SetString(PyExc_TypeError, e.what().c_str());
        return NULL;
    }
}

extern "C"
PyObject *
RtreeIndex_intersects(GISPySpatialIndex *index, double *min, double *max, int incl_reg)
{
    /* get intersecting data */
    int count=0;
    PyObject *ids;

    ids = PyList_New((size_t)count);
    PyListVisitor *visitor = new PyListVisitor(ids, incl_reg);

    try {	
        const SpatialIndex::Region *region = new SpatialIndex::Region(min, max, 2);
        index->index().intersectsWithQuery((*region), (*visitor));
        delete region;
        delete visitor;
        return ids;
    }
    catch (Tools::Exception& e) {
        PyErr_SetString(PyExc_TypeError, e.what().c_str());
        delete visitor;
        return NULL;
    }

}

extern "C"
PyObject *
RtreeIndex_contains(GISPySpatialIndex *index, double *min, double *max, int incl_reg)
{
    /* get intersecting data */
    int count=0;
    PyObject *ids;

    ids = PyList_New((size_t)count);
    PyListVisitor *visitor = new PyListVisitor(ids, incl_reg);

    try {	
        const SpatialIndex::Region *region = new SpatialIndex::Region(min, max, 2);
        index->index().containsWhatQuery((*region), (*visitor));
        delete region;
        delete visitor;
        return ids;
    }
    catch (Tools::Exception& e) {
        PyErr_SetString(PyExc_TypeError, e.what().c_str());
        delete visitor;
        return NULL;
    }
}

extern "C"
int
RtreeIndex_isValid(GISPySpatialIndex *index)
{
  try {	
      return (int) index->index().isIndexValid();
  }
  catch (...) {
     // isIndexValid throws an exception for empty indexes which we'll assume is valid
	return 1;
  }
}

extern "C"
PyObject *
RtreeIndex_nearestNeighbors(GISPySpatialIndex *index, uint32_t num_results, double *min, double *max, int incl_reg)
{
    /* get intersecting data */
    int count=0;
    PyObject *ids;

    ids = PyList_New((size_t)count);
    PyListVisitor *visitor = new PyListVisitor(ids, incl_reg);
    try {
        const SpatialIndex::Region *region = new SpatialIndex::Region(min, max, 2);
        index->index().nearestNeighborQuery(num_results, (*region), (*visitor));
        delete region;
        delete visitor;
        return ids;
    }
    catch (Tools::Exception& e) {
        PyErr_SetString(PyExc_TypeError, e.what().c_str());
        delete visitor;
        return NULL;
    }
}

extern "C"
PyObject *
RtreeIndex_getheight(GISPySpatialIndex *index)
{
	PyObject *ret;

	SpatialIndex::IStatistics* stats = NULL;
	index->index().getStatistics(&stats);

	if(stats)
	    ret = PyInt_FromSsize_t(stats->getTreeHeight());
	else
            ret = PyInt_FromSsize_t(-1);

	if(stats)
		delete stats;

	return ret;
}

extern "C"
PyObject *
RtreeIndex_getsize(GISPySpatialIndex *index)
{
	PyObject *ret;

	SpatialIndex::IStatistics* stats = NULL;
	index->index().getStatistics(&stats);

	if(stats)
	    ret = PyInt_FromSsize_t(stats->getNumberOfData());
	else
            ret = PyInt_FromSsize_t(-1);

	if(stats)
		delete stats;

	return ret;

}

extern "C"
PyObject *
RtreeIndex_getnodes(GISPySpatialIndex *index)
{
	PyObject *ret;

	SpatialIndex::IStatistics* stats = NULL;
	index->index().getStatistics(&stats);

	if(stats)
	    ret = PyInt_FromSsize_t(stats->getNumberOfNodes());
	else
            ret = PyInt_FromSsize_t(-1);

	if(stats)
		delete stats;

	return ret;
}

