#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import sys
import os
import cPickle
import logging
from pprint import pprint
from optparse import make_option

from terra_config import TerraConfig
from cli_utils import die, process_cmdline


log = logging.getLogger("terra.core.plugin_config")


class PluginConfig(object):
    def __init__(self, name, moddir, modname, filter_map, rank, enabled=True):
        if not isinstance(name, basestring) or \
           not isinstance(moddir, basestring) or \
           not isinstance(modname, basestring):
            raise TypeError("name, moddir and modname must be strings")

        if type(filter_map) is not list:
            raise TypeError("filter_map must be a list of tuples")
        for t in filter_map:
            if type(t) is not tuple or len(t) != 2 or \
               not isinstance(t[0], basestring) or \
               not isinstance(t[1], basestring):
                raise TypeError("filter_map items must be "
                                "tuples with 2 strings")

        if type(rank) is not int:
            raise TypeError("rank should be an int")
        if rank < 0:
            rank = 0
        if rank > 256:
            rank = 256

        if type(enabled) is not bool:
            raise TypeError("enabled must be a boolean")

        self.name = name
        self.moddir = moddir
        self.modname = modname
        self.filter_map = filter_map
        self.rank = rank
        self.enabled = enabled

    def pprint(self):
        def mprint(label, msg):
            sys.stdout.write("%s: " % label)
            pprint(msg)

        mprint("Name", self.name)
        mprint("Moddir", self.moddir)
        mprint("Modname", self.modname)
        mprint("Filter map", self.filter_map)
        mprint("Rank", self.rank)
        mprint("Enabled", self.enabled)

    def __eq__(self, other):
        # don't check 'rank' and 'enabled' attributes
        return self.name == other.name and \
               self.moddir == other.moddir and \
               self.modname == other.modname and \
               self.filter_map == other.filter_map

    def __ne__(self, other):
        return not self.__eq__(other)

    def __str__(self):
        return "%s(name=%r, moddir=%r, modname=%r, filter_map=%r, " \
               "rank=%d, enabled=%s)" % (self.__class__.__name__, self.name,
                                         self.moddir, self.modname,
                                         self.filter_map, self.rank,
                                         self.enabled)


class PluginConfigManager(object):
    def _check_loaded(method):
        def wrap(*args, **kargs):
            if args[0]._confs is None:
                raise RuntimeError("config file not loaded yet")
            return method(*args, **kargs)

        return wrap

    def __init__(self, config_file):
        self._config_file = config_file
        self._confs = None

    def _get_confs(self):
        if self._confs is None:
            return None
        return self._confs.values()

    confs = property(_get_confs)

    def _load_from_file(self, filename):
        if not os.path.exists(filename):
            return {}

        confs = cPickle.load(open(filename, "rb"))

        if type(confs) is not dict:
            raise TypeError("config file %r should have a dict" % filename)

        for c in confs.itervalues():
            if type(c) is not PluginConfig:
                raise TypeError("each plugin config should be instance"
                                " of PluginConfig")

        return confs

    def load(self):
        if self._confs is not None:
            return self._confs.values()

        self._confs = self._load_from_file(self._config_file)
        return self._confs.values()

    @_check_loaded
    def save(self):
        cPickle.dump(self._confs, open(self._config_file, "wb"),
                     cPickle.HIGHEST_PROTOCOL)

    def _do_sync(self, cfg_file):
        oconfs = self._load_from_file(cfg_file)

        set_local = set(self._confs.iterkeys())
        set_other = set(oconfs.iterkeys())

        minus = set_local.difference(set_other)
        for name in minus:
            del self._confs[name]
        log.debug("sync: removed plugins %r", minus)

        add = set_other.difference(set_local)
        for name in add:
            self._confs[name] = oconfs[name]
        log.debug("sync: added plugins %r", add)

        check = set_local.intersection(set_other)
        for name in check:
            cl = self._confs[name]
            co = oconfs[name]
            if cl != co:
                cl, cl.enabled = co, cl.enabled
                self._confs[name] = cl
                log.debug("sync: updated plugin %r", name)
        log.debug("sync: checked plugins %r", check)

    @_check_loaded
    def sync(self, cfg_file):
        mt_local = os.path.getmtime(self._config_file)
        mt_other = os.path.getmtime(cfg_file)
        if mt_other > mt_local:
            log.debug("syncing config file %r with %r",
                      self._config_file, cfg_file)
            self._do_sync(cfg_file)
            self.save()

    @_check_loaded
    def add(self, conf):
        if type(conf) is not PluginConfig:
            raise TypeError("conf must be a PluginConfig")

        res, self._confs[conf.name] = self._confs.get(conf.name, None), conf
        return res

    @_check_loaded
    def remove(self, name):
        if name not in self._confs:
            raise ValueError("no plugin %r registered" % name)

        del self._confs[name]

    def _change_enabled_for(self, name, value):
        if name not in self._confs:
            raise ValueError("no plugin %r registered" % name)

        self._confs[name].enabled = bool(value)

    @_check_loaded
    def enable(self, name):
        self._change_enabled_for(name, True)

    @_check_loaded
    def disable(self, name):
        self._change_enabled_for(name, False)


def get_user_plugins_cfg(cfg):
    app_user_dir = os.path.join(os.path.expanduser("~"), cfg.app_user_dir)
    if not os.path.exists(app_user_dir):
        os.makedirs(app_user_dir, 0700)
    plugins_cfg = os.path.join(app_user_dir, os.path.basename(cfg.plugins_cfg))
    return plugins_cfg


def _process_cmdline(options):
    parser, opts, args = process_cmdline(options)
    cfg = TerraConfig(opts.config_file)

    return cfg, opts, args, parser


def list_plugins():
    options = [make_option("-s", "--system", action="store_true", dest="system",
                           help="list plugins installed in the system")]

    cfg, opts, args, parser = _process_cmdline(options)
    if len(args) != 0:
        parser.error("unexpected arguments %r" % args)

    if opts.system:
        plugins_cfg = cfg.plugins_cfg
    else:
        plugins_cfg = get_user_plugins_cfg(cfg)

    try:
        pcm = PluginConfigManager(plugins_cfg)
        plgs = pcm.load()
    except Exception, e:
        die("couldn't read plugin configuration file %r: %s" % (plugins_cfg, e))

    print "PLUGINS:"
    for p in plgs:
        print "-" * 70
        p.pprint()


def enable_plugin():
    options = [make_option("-s", "--system", action="store_true", dest="system",
                           help="mark plugin as enabled in the system")]

    cfg, opts, args, parser = _process_cmdline(options)
    if len(args) != 1:
        parser.error("unexpected arguments %r" % args)

    if opts.system:
        plugins_cfg = cfg.plugins_cfg
    else:
        plugins_cfg = get_user_plugins_cfg(cfg)

    try:
        pcm = PluginConfigManager(plugins_cfg)
        pcm.load()
        pcm.enable(args[0])
        pcm.save()
    except Exception, e:
        die("couldn't enable plugin %r: %s" % (args[0], e))


def disable_plugin():
    options = [make_option("-s", "--system", action="store_true", dest="system",
                           help="mark plugin as disabled in the system")]

    cfg, opts, args, parser = _process_cmdline(options)
    if len(args) != 1:
        parser.error("unexpected arguments %r" % args)

    if opts.system:
        plugins_cfg = cfg.plugins_cfg
    else:
        plugins_cfg = get_user_plugins_cfg(cfg)

    try:
        pcm = PluginConfigManager(plugins_cfg)
        pcm.load()
        pcm.disable(args[0])
        pcm.save()
    except Exception, e:
        die("couldn't disable plugin %r: %s" % (args[0], e))
