/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qpaintengine_wrapper.h"

// Extra includes
#include <QVarLengthArray>
#include <qimage.h>
#include <qline.h>
#include <qpaintdevice.h>
#include <qpaintengine.h>
#include <qpainter.h>
#include <qpainterpath.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qrect.h>
#include <qregion.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QPaintEngineWrapper::QPaintEngineWrapper(QFlags<QPaintEngine::PaintEngineFeature> features) : QPaintEngine(features) {
    // ... middle
}

bool QPaintEngineWrapper::begin(QPaintDevice * pdev)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "begin"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.begin()' not implemented.");
        return bool(0);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QPaintDevice* >::toPython(pdev)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QPaintEngine.begin");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

QPoint QPaintEngineWrapper::coordinateOffset() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "coordinateOffset"));
    if (py_override.isNull()) {
        return this->QPaintEngine::coordinateOffset();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPoint();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPoint >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QPaintEngine.coordinateOffset");
        return QPoint()        ;
    }
    QPoint cpp_result(Shiboken::Converter<QPoint >::toCpp(py_result));
    return cpp_result;
}

void QPaintEngineWrapper::drawEllipse(const QRect & r)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawEllipse"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawEllipse(r);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QRect& >::toPython(r)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawEllipse(const QRectF & r)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawEllipse"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawEllipse(r);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QRectF& >::toPython(r)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawImage(const QRectF & r, const QImage & pm, const QRectF & sr, QFlags<Qt::ImageConversionFlag> flags)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawImage"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawImage(r, pm, sr, flags);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OOOO)",
        Shiboken::Converter<QRectF& >::toPython(r),
        Shiboken::Converter<QImage& >::toPython(pm),
        Shiboken::Converter<QRectF& >::toPython(sr),
        Shiboken::Converter<QFlags<Qt::ImageConversionFlag> >::toPython(flags)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawLines(const QLine * lines, int lineCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawLines"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawLines(lines, lineCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QLine* >::toPython(lines),
        lineCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawLines(const QLineF * lines, int lineCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawLines"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawLines(lines, lineCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QLineF* >::toPython(lines),
        lineCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPath(const QPainterPath & path)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPath"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawPath(path);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QPainterPath& >::toPython(path)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPixmap(const QRectF & r, const QPixmap & pm, const QRectF & sr)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPixmap"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.drawPixmap()' not implemented.");
        return ;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OOO)",
        Shiboken::Converter<QRectF& >::toPython(r),
        Shiboken::Converter<QPixmap& >::toPython(pm),
        Shiboken::Converter<QRectF& >::toPython(sr)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPoints(const QPoint * points, int pointCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPoints"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawPoints(points, pointCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QPoint* >::toPython(points),
        pointCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPoints(const QPointF * points, int pointCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPoints"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawPoints(points, pointCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QPointF* >::toPython(points),
        pointCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPolygon(const QPoint * points, int pointCount, QPaintEngine::PolygonDrawMode mode)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPolygon"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawPolygon(points, pointCount, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OiO)",
        Shiboken::Converter<QPoint* >::toPython(points),
        pointCount,
        Shiboken::Converter<QPaintEngine::PolygonDrawMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawPolygon(const QPointF * points, int pointCount, QPaintEngine::PolygonDrawMode mode)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawPolygon"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawPolygon(points, pointCount, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OiO)",
        Shiboken::Converter<QPointF* >::toPython(points),
        pointCount,
        Shiboken::Converter<QPaintEngine::PolygonDrawMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawRects(const QRect * rects, int rectCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawRects"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawRects(rects, rectCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QRect* >::toPython(rects),
        rectCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawRects(const QRectF * rects, int rectCount)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawRects"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawRects(rects, rectCount);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Oi)",
        Shiboken::Converter<QRectF* >::toPython(rects),
        rectCount
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QPaintEngineWrapper::drawTextItem(const QPointF & p, const QTextItem & textItem)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawTextItem"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawTextItem(p, textItem);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QPointF& >::toPython(p),
        Shiboken::Converter<QTextItem& >::toPython(textItem)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 1));
}

void QPaintEngineWrapper::drawTiledPixmap(const QRectF & r, const QPixmap & pixmap, const QPointF & s)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "drawTiledPixmap"));
    if (py_override.isNull()) {
        return this->QPaintEngine::drawTiledPixmap(r, pixmap, s);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OOO)",
        Shiboken::Converter<QRectF& >::toPython(r),
        Shiboken::Converter<QPixmap& >::toPython(pixmap),
        Shiboken::Converter<QPointF& >::toPython(s)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QPaintEngineWrapper::end()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "end"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.end()' not implemented.");
        return bool(0);
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QPaintEngine.end");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QPaintEngine::Type QPaintEngineWrapper::type() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "type"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.type()' not implemented.");
        return QPaintEngine::Type();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPaintEngine::Type();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPaintEngine::Type >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QPaintEngine.type");
        return QPaintEngine::Type()        ;
    }
    QPaintEngine::Type cpp_result(Shiboken::Converter<QPaintEngine::Type >::toCpp(py_result));
    return cpp_result;
}

void QPaintEngineWrapper::updateState(const QPaintEngineState & state)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "updateState"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.updateState()' not implemented.");
        return ;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QPaintEngineState& >::toPython(state)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QPaintEngineWrapper::~QPaintEngineWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQPaintEngine_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QPaintEngineWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    assert(!sbkSelf->cptr);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'QPaintEngine' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0};

    if (!PyArg_UnpackTuple(args, "QPaintEngine", 0, 1, &(pyargs[0])))
        return -1;

    if (numArgs == 0) {
        // QPaintEngine(QFlags<QPaintEngine::PaintEngineFeature>)
        cptr = new QPaintEngineWrapper();
    } else if (SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Check(pyargs[0])) {
        QFlags<QPaintEngine::PaintEngineFeature> cpp_arg0 = Shiboken::Converter<QFlags<QPaintEngine::PaintEngineFeature> >::toCpp(pyargs[0]);
        // QPaintEngine(QFlags<QPaintEngine::PaintEngineFeature>)
        cptr = new QPaintEngineWrapper(cpp_arg0);
    } else goto SbkQPaintEngine_Init_TypeError;

    if (PyErr_Occurred()) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQPaintEngine_Init_TypeError;

    sbkSelf->cptr = cptr;
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    sbkSelf->referredObjects = new Shiboken::RefCountMap;
    BindingManager::instance().registerWrapper(sbkSelf);

    return 1;

    SbkQPaintEngine_Init_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngine.PaintEngineFeatures = 0", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine", overloads);
        return -1;
}

static PyObject*
SbkQPaintEngineFunc_begin(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QPaintDevice* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPaintDevice* cpp_arg0 = Shiboken::Converter<QPaintDevice* >::toCpp(arg);
        // begin(QPaintDevice*)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.begin()' not implemented.");
            return 0;
        }
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->begin(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQPaintEngineFunc_begin_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQPaintEngineFunc_begin_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintDevice", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.begin", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_clearDirty(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (SbkPySide_QtGui_QPaintEngine_DirtyFlag_Check(arg)) {
        QFlags<QPaintEngine::DirtyFlag> cpp_arg0 = Shiboken::Converter<QFlags<QPaintEngine::DirtyFlag> >::toCpp(arg);
        // clearDirty(QFlags<QPaintEngine::DirtyFlag>)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::clearDirty(cpp_arg0);
    } else goto SbkQPaintEngineFunc_clearDirty_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_clearDirty_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngine.DirtyFlags", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.clearDirty", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_coordinateOffset(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // coordinateOffset()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QPoint cpp_result = cppSelf->QPaintEngine::coordinateOffset();
    py_result = Shiboken::Converter<QPoint >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_drawEllipse(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QRect& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(arg);
        // drawEllipse(QRect)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawEllipse(*cpp_arg0);
    } else if (Shiboken::Converter<QRectF& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(arg);
        if (!Shiboken::Converter<QRectF& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // drawEllipse(QRectF)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawEllipse(*cpp_arg0);
    } else goto SbkQPaintEngineFunc_drawEllipse_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawEllipse_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect", "PySide.QtCore.QRectF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.drawEllipse", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawImage(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "drawImage", 3, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]) && Shiboken::Converter<QImage& >::isConvertible(pyargs[1]) && Shiboken::Converter<QRectF& >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QImage* cpp_arg1 = Shiboken::Converter<QImage* >::toCpp(pyargs[1]);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg2_auto_ptr;
        QRectF* cpp_arg2 = Shiboken::Converter<QRectF* >::toCpp(pyargs[2]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[2]))
            cpp_arg2_auto_ptr = std::auto_ptr<QRectF >(cpp_arg2);
        if (numArgs == 3) {
            // drawImage(QRectF,QImage,QRectF,QFlags<Qt::ImageConversionFlag>)
            QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
            cppSelf->QPaintEngine::drawImage(*cpp_arg0, *cpp_arg1, *cpp_arg2);
        } else if (SbkPySide_QtCore_Qt_ImageConversionFlag_Check(pyargs[3])) {
            QFlags<Qt::ImageConversionFlag> cpp_arg3 = Shiboken::Converter<QFlags<Qt::ImageConversionFlag> >::toCpp(pyargs[3]);
            // drawImage(QRectF,QImage,QRectF,QFlags<Qt::ImageConversionFlag>)
            QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
            cppSelf->QPaintEngine::drawImage(*cpp_arg0, *cpp_arg1, *cpp_arg2, cpp_arg3);
        } else goto SbkQPaintEngineFunc_drawImage_TypeError;
    } else goto SbkQPaintEngineFunc_drawImage_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawImage_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF, PySide.QtGui.QImage, PySide.QtCore.QRectF, PySide.QtCore.Qt.ImageConversionFlags = Qt.AutoColor", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawImage", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawLines(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "drawLines", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QLine* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QLine* cpp_arg0 = Shiboken::Converter<QLine* >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawLines(const QLine*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawLines(cpp_arg0, cpp_arg1);
    } else if (numArgs == 2 && Shiboken::Converter<QLineF* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QLineF > cpp_arg0_auto_ptr;
        QLineF* cpp_arg0 = Shiboken::Converter<QLineF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QLineF* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QLineF >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawLines(const QLineF*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawLines(cpp_arg0, cpp_arg1);
    } else goto SbkQPaintEngineFunc_drawLines_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawLines_TypeError:
        const char* overloads[] = {"PySide.QtCore.QLine, int", "PySide.QtCore.QLineF, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawLines", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawPath(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QPainterPath& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(arg);
        // drawPath(QPainterPath)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawPath(*cpp_arg0);
    } else goto SbkQPaintEngineFunc_drawPath_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawPath_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.drawPath", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawPixmap(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "drawPixmap", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]) && Shiboken::Converter<QPixmap& >::isConvertible(pyargs[1]) && Shiboken::Converter<QRectF& >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QPixmap > cpp_arg1_auto_ptr;
        QPixmap* cpp_arg1 = Shiboken::Converter<QPixmap* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QPixmap& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QPixmap >(cpp_arg1);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg2_auto_ptr;
        QRectF* cpp_arg2 = Shiboken::Converter<QRectF* >::toCpp(pyargs[2]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[2]))
            cpp_arg2_auto_ptr = std::auto_ptr<QRectF >(cpp_arg2);
        // drawPixmap(QRectF,QPixmap,QRectF)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.drawPixmap()' not implemented.");
            return 0;
        }
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->drawPixmap(*cpp_arg0, *cpp_arg1, *cpp_arg2);
    } else goto SbkQPaintEngineFunc_drawPixmap_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawPixmap_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF, PySide.QtGui.QPixmap, PySide.QtCore.QRectF", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawPixmap", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawPoints(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "drawPoints", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QPoint* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPoint > cpp_arg0_auto_ptr;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPoint* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPoint >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawPoints(const QPoint*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawPoints(cpp_arg0, cpp_arg1);
    } else if (numArgs == 2 && Shiboken::Converter<QPointF* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawPoints(const QPointF*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawPoints(cpp_arg0, cpp_arg1);
    } else goto SbkQPaintEngineFunc_drawPoints_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawPoints_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint, int", "PySide.QtCore.QPointF, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawPoints", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawPolygon(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "drawPolygon", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QPoint* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && Shiboken::Converter<QPaintEngine::PolygonDrawMode >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPoint > cpp_arg0_auto_ptr;
        QPoint* cpp_arg0 = Shiboken::Converter<QPoint* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPoint* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPoint >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        QPaintEngine::PolygonDrawMode cpp_arg2 = Shiboken::Converter<QPaintEngine::PolygonDrawMode >::toCpp(pyargs[2]);
        // drawPolygon(const QPoint*,int,QPaintEngine::PolygonDrawMode)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawPolygon(cpp_arg0, cpp_arg1, cpp_arg2);
    } else if (numArgs == 3 && Shiboken::Converter<QPointF* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1]) && Shiboken::Converter<QPaintEngine::PolygonDrawMode >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        QPaintEngine::PolygonDrawMode cpp_arg2 = Shiboken::Converter<QPaintEngine::PolygonDrawMode >::toCpp(pyargs[2]);
        // drawPolygon(const QPointF*,int,QPaintEngine::PolygonDrawMode)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawPolygon(cpp_arg0, cpp_arg1, cpp_arg2);
    } else goto SbkQPaintEngineFunc_drawPolygon_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawPolygon_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPoint, int, PySide.QtGui.QPaintEngine.PolygonDrawMode", "PySide.QtCore.QPointF, int, PySide.QtGui.QPaintEngine.PolygonDrawMode", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawPolygon", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawRects(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "drawRects", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QRect* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawRects(const QRect*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawRects(cpp_arg0, cpp_arg1);
    } else if (numArgs == 2 && Shiboken::Converter<QRectF* >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF* >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        // drawRects(const QRectF*,int)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawRects(cpp_arg0, cpp_arg1);
    } else goto SbkQPaintEngineFunc_drawRects_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawRects_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect, int", "PySide.QtCore.QRectF, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawRects", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawTextItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "drawTextItem", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]) && Shiboken::Converter<QTextItem& >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QTextItem* cpp_arg1 = Shiboken::Converter<QTextItem* >::toCpp(pyargs[1]);
        // drawTextItem(QPointF,QTextItem)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawTextItem(*cpp_arg0, *cpp_arg1);
    } else goto SbkQPaintEngineFunc_drawTextItem_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawTextItem_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF, PySide.QtGui.QTextItem", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawTextItem", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_drawTiledPixmap(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "drawTiledPixmap", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]) && Shiboken::Converter<QPixmap& >::isConvertible(pyargs[1]) && Shiboken::Converter<QPointF& >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QPixmap > cpp_arg1_auto_ptr;
        QPixmap* cpp_arg1 = Shiboken::Converter<QPixmap* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QPixmap& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QPixmap >(cpp_arg1);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg2_auto_ptr;
        QPointF* cpp_arg2 = Shiboken::Converter<QPointF* >::toCpp(pyargs[2]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[2]))
            cpp_arg2_auto_ptr = std::auto_ptr<QPointF >(cpp_arg2);
        // drawTiledPixmap(QRectF,QPixmap,QPointF)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::drawTiledPixmap(*cpp_arg0, *cpp_arg1, *cpp_arg2);
    } else goto SbkQPaintEngineFunc_drawTiledPixmap_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_drawTiledPixmap_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF, PySide.QtGui.QPixmap, PySide.QtCore.QPointF", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QPaintEngine.drawTiledPixmap", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_end(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // end()
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.end()' not implemented.");
        return 0;
    }
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->end();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_hasFeature(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Check(arg)) {
        QFlags<QPaintEngine::PaintEngineFeature> cpp_arg0 = Shiboken::Converter<QFlags<QPaintEngine::PaintEngineFeature> >::toCpp(arg);
        // hasFeature(QFlags<QPaintEngine::PaintEngineFeature>)const
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QPaintEngine::hasFeature(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQPaintEngineFunc_hasFeature_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQPaintEngineFunc_hasFeature_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngine.PaintEngineFeatures", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.hasFeature", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_isActive(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isActive()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QPaintEngine::isActive();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_isExtended(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isExtended()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QPaintEngine::isExtended();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_paintDevice(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // paintDevice()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QPaintDevice * cpp_result = cppSelf->QPaintEngine::paintDevice();
    py_result = Shiboken::Converter<QPaintDevice* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_painter(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // painter()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QPainter * cpp_result = cppSelf->QPaintEngine::painter();
    py_result = Shiboken::Converter<QPainter* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_setActive(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setActive(bool)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::setActive(cpp_arg0);
    } else goto SbkQPaintEngineFunc_setActive_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_setActive_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.setActive", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_setDirty(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (SbkPySide_QtGui_QPaintEngine_DirtyFlag_Check(arg)) {
        QFlags<QPaintEngine::DirtyFlag> cpp_arg0 = Shiboken::Converter<QFlags<QPaintEngine::DirtyFlag> >::toCpp(arg);
        // setDirty(QFlags<QPaintEngine::DirtyFlag>)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::setDirty(cpp_arg0);
    } else goto SbkQPaintEngineFunc_setDirty_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_setDirty_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngine.DirtyFlags", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.setDirty", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_setSystemClip(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QRegion& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QRegion > cpp_arg0_auto_ptr;
        QRegion* cpp_arg0 = Shiboken::Converter<QRegion* >::toCpp(arg);
        if (!Shiboken::Converter<QRegion& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QRegion >(cpp_arg0);
        // setSystemClip(QRegion)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::setSystemClip(*cpp_arg0);
    } else goto SbkQPaintEngineFunc_setSystemClip_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_setSystemClip_TypeError:
        const char* overloads[] = {"PySide.QtGui.QRegion", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.setSystemClip", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_setSystemRect(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QRect& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QRect* cpp_arg0 = Shiboken::Converter<QRect* >::toCpp(arg);
        // setSystemRect(QRect)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->QPaintEngine::setSystemRect(*cpp_arg0);
    } else goto SbkQPaintEngineFunc_setSystemRect_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_setSystemRect_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRect", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.setSystemRect", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_syncState(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // syncState()
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    cppSelf->QPaintEngine::syncState();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQPaintEngineFunc_systemClip(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // systemClip()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QRegion cpp_result = cppSelf->QPaintEngine::systemClip();
    py_result = Shiboken::Converter<QRegion >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_systemRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // systemRect()const
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QRect cpp_result = cppSelf->QPaintEngine::systemRect();
    py_result = Shiboken::Converter<QRect >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_testDirty(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (SbkPySide_QtGui_QPaintEngine_DirtyFlag_Check(arg)) {
        QFlags<QPaintEngine::DirtyFlag> cpp_arg0 = Shiboken::Converter<QFlags<QPaintEngine::DirtyFlag> >::toCpp(arg);
        // testDirty(QFlags<QPaintEngine::DirtyFlag>)
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QPaintEngine::testDirty(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQPaintEngineFunc_testDirty_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQPaintEngineFunc_testDirty_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngine.DirtyFlags", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.testDirty", overloads);
        return 0;
}

static PyObject*
SbkQPaintEngineFunc_type(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // type()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.type()' not implemented.");
        return 0;
    }
    QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
    QPaintEngine::Type cpp_result = cppSelf->type();
    py_result = Shiboken::Converter<QPaintEngine::Type >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQPaintEngineFunc_updateState(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QPaintEngineState& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QPaintEngineState* cpp_arg0 = Shiboken::Converter<QPaintEngineState* >::toCpp(arg);
        // updateState(QPaintEngineState)
        if (SbkBaseWrapper_containsCppWrapper(self)) {
            PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QPaintEngine.updateState()' not implemented.");
            return 0;
        }
        QPaintEngine* cppSelf = Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self);
        cppSelf->updateState(*cpp_arg0);
    } else goto SbkQPaintEngineFunc_updateState_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQPaintEngineFunc_updateState_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPaintEngineState", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QPaintEngine.updateState", overloads);
        return 0;
}

static PyMethodDef SbkQPaintEngine_methods[] = {
    {"begin", (PyCFunction)SbkQPaintEngineFunc_begin, METH_O},
    {"clearDirty", (PyCFunction)SbkQPaintEngineFunc_clearDirty, METH_O},
    {"coordinateOffset", (PyCFunction)SbkQPaintEngineFunc_coordinateOffset, METH_NOARGS},
    {"drawEllipse", (PyCFunction)SbkQPaintEngineFunc_drawEllipse, METH_O},
    {"drawImage", (PyCFunction)SbkQPaintEngineFunc_drawImage, METH_VARARGS},
    {"drawLines", (PyCFunction)SbkQPaintEngineFunc_drawLines, METH_VARARGS},
    {"drawPath", (PyCFunction)SbkQPaintEngineFunc_drawPath, METH_O},
    {"drawPixmap", (PyCFunction)SbkQPaintEngineFunc_drawPixmap, METH_VARARGS},
    {"drawPoints", (PyCFunction)SbkQPaintEngineFunc_drawPoints, METH_VARARGS},
    {"drawPolygon", (PyCFunction)SbkQPaintEngineFunc_drawPolygon, METH_VARARGS},
    {"drawRects", (PyCFunction)SbkQPaintEngineFunc_drawRects, METH_VARARGS},
    {"drawTextItem", (PyCFunction)SbkQPaintEngineFunc_drawTextItem, METH_VARARGS},
    {"drawTiledPixmap", (PyCFunction)SbkQPaintEngineFunc_drawTiledPixmap, METH_VARARGS},
    {"end", (PyCFunction)SbkQPaintEngineFunc_end, METH_NOARGS},
    {"hasFeature", (PyCFunction)SbkQPaintEngineFunc_hasFeature, METH_O},
    {"isActive", (PyCFunction)SbkQPaintEngineFunc_isActive, METH_NOARGS},
    {"isExtended", (PyCFunction)SbkQPaintEngineFunc_isExtended, METH_NOARGS},
    {"paintDevice", (PyCFunction)SbkQPaintEngineFunc_paintDevice, METH_NOARGS},
    {"painter", (PyCFunction)SbkQPaintEngineFunc_painter, METH_NOARGS},
    {"setActive", (PyCFunction)SbkQPaintEngineFunc_setActive, METH_O},
    {"setDirty", (PyCFunction)SbkQPaintEngineFunc_setDirty, METH_O},
    {"setSystemClip", (PyCFunction)SbkQPaintEngineFunc_setSystemClip, METH_O},
    {"setSystemRect", (PyCFunction)SbkQPaintEngineFunc_setSystemRect, METH_O},
    {"syncState", (PyCFunction)SbkQPaintEngineFunc_syncState, METH_NOARGS},
    {"systemClip", (PyCFunction)SbkQPaintEngineFunc_systemClip, METH_NOARGS},
    {"systemRect", (PyCFunction)SbkQPaintEngineFunc_systemRect, METH_NOARGS},
    {"testDirty", (PyCFunction)SbkQPaintEngineFunc_testDirty, METH_O},
    {"type", (PyCFunction)SbkQPaintEngineFunc_type, METH_NOARGS},
    {"updateState", (PyCFunction)SbkQPaintEngineFunc_updateState, METH_O},
    {0} // Sentinel
};

static PyObject* SbkQPaintEngine_get_extended(SbkBaseWrapper* self)
{
    return Shiboken::Converter<unsigned int >::toPython(Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->extended);
}
static int SbkQPaintEngine_set_extended(SbkBaseWrapper* self, PyObject* value)
{
    if (value == 0) {
        PyErr_SetString(PyExc_TypeError, "'extended' may not be deleted");
        return -1;
    }
    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "wrong type attributed to 'extended', 'uint' or convertible type expected");
        return -1;
    }

    Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->extended = Shiboken::Converter<unsigned int >::toCpp(value);

    return 0;
}

static PyObject* SbkQPaintEngine_get_selfDestruct(SbkBaseWrapper* self)
{
    return Shiboken::Converter<unsigned int >::toPython(Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->selfDestruct);
}
static int SbkQPaintEngine_set_selfDestruct(SbkBaseWrapper* self, PyObject* value)
{
    if (value == 0) {
        PyErr_SetString(PyExc_TypeError, "'selfDestruct' may not be deleted");
        return -1;
    }
    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "wrong type attributed to 'selfDestruct', 'uint' or convertible type expected");
        return -1;
    }

    Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->selfDestruct = Shiboken::Converter<unsigned int >::toCpp(value);

    return 0;
}

static PyObject* SbkQPaintEngine_get_active(SbkBaseWrapper* self)
{
    return Shiboken::Converter<unsigned int >::toPython(Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->active);
}
static int SbkQPaintEngine_set_active(SbkBaseWrapper* self, PyObject* value)
{
    if (value == 0) {
        PyErr_SetString(PyExc_TypeError, "'active' may not be deleted");
        return -1;
    }
    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "wrong type attributed to 'active', 'uint' or convertible type expected");
        return -1;
    }

    Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->active = Shiboken::Converter<unsigned int >::toCpp(value);

    return 0;
}

static PyObject* SbkQPaintEngine_get_state(SbkBaseWrapper* self)
{
    return Shiboken::Converter<QPaintEngineState* >::toPython(Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->state);
}
static int SbkQPaintEngine_set_state(SbkBaseWrapper* self, PyObject* value)
{
    if (value == 0) {
        PyErr_SetString(PyExc_TypeError, "'state' may not be deleted");
        return -1;
    }
    if (!Shiboken::Converter<QPaintEngineState* >::isConvertible(value)) {
        PyErr_SetString(PyExc_TypeError, "wrong type attributed to 'state', 'QPaintEngineState' or convertible type expected");
        return -1;
    }

    Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->state = Shiboken::Converter<QPaintEngineState* >::toCpp(value);

    Shiboken::SbkBaseWrapper_keepReference(reinterpret_cast<SbkBaseWrapper*>(self), "state", value);

    return 0;
}

static PyObject* SbkQPaintEngine_get_gccaps(SbkBaseWrapper* self)
{
    return Shiboken::Converter<QFlags<QPaintEngine::PaintEngineFeature> >::toPython(Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->gccaps);
}
static int SbkQPaintEngine_set_gccaps(SbkBaseWrapper* self, PyObject* value)
{
    if (value == 0) {
        PyErr_SetString(PyExc_TypeError, "'gccaps' may not be deleted");
        return -1;
    }
    if (!SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "wrong type attributed to 'gccaps', 'PaintEngineFeatures' or convertible type expected");
        return -1;
    }

    Shiboken::Converter<QPaintEngine* >::toCpp((PyObject*)self)->gccaps = Shiboken::Converter<QFlags<QPaintEngine::PaintEngineFeature> >::toCpp(value);

    return 0;
}

// Getters and Setters for QPaintEngine
static PyGetSetDef SbkQPaintEngine_getsetlist[] = {
    {const_cast<char*>("extended"), (getter)SbkQPaintEngine_get_extended, (setter)SbkQPaintEngine_set_extended},
    {const_cast<char*>("selfDestruct"), (getter)SbkQPaintEngine_get_selfDestruct, (setter)SbkQPaintEngine_set_selfDestruct},
    {const_cast<char*>("active"), (getter)SbkQPaintEngine_get_active, (setter)SbkQPaintEngine_set_active},
    {const_cast<char*>("state"), (getter)SbkQPaintEngine_get_state, (setter)SbkQPaintEngine_set_state},
    {const_cast<char*>("gccaps"), (getter)SbkQPaintEngine_get_gccaps, (setter)SbkQPaintEngine_set_gccaps},
    {0}  // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQPaintEngine_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QPaintEngine",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::SbkBaseWrapper_Dealloc< QPaintEngineWrapper >,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQPaintEngine_methods,
    /*tp_members*/          0,
    /*tp_getset*/           SbkQPaintEngine_getsetlist,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQPaintEngine_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0
};

static SbkBaseWrapperType* SbkQPaintEngine_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQPaintEngine_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QPaintEngine*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

static PyGetSetDef SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PolygonDrawMode",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtGui_QPaintEngine_Type_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_Type_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_Type_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Type",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QPaintEngine_Type_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_Type_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_Type_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___and__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toPython(
        ((QPaintEngine::PaintEngineFeatures) ((SbkEnumObject*)self)->ob_ival)
        & Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___or__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toPython(
        ((QPaintEngine::PaintEngineFeatures) ((SbkEnumObject*)self)->ob_ival)
        | Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___xor__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toPython(
        ((QPaintEngine::PaintEngineFeatures) ((SbkEnumObject*)self)->ob_ival)
        ^ Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___invert__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toPython(
        ~ Shiboken::Converter< QPaintEngine::PaintEngineFeatures >::toCpp(self)
    );
}


static PyNumberMethods SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_as_number = {
    /*nb_add*/                  0,
    /*nb_subtract*/             0,
    /*nb_multiply*/             0,
    /*nb_divide*/               0,
    /*nb_remainder*/            0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             0,
    /*nb_positive*/             0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___invert__,
    /*nb_lshift*/               0,
    /*nb_rshift*/               0,
    /*nb_and*/                  (binaryfunc)SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___and__,
    /*nb_xor*/                  (binaryfunc)SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___xor__,
    /*nb_or*/                   (binaryfunc)SbkPySide_QtGui_QPaintEngine_PaintEngineFeature___or__,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          0,
    /*nb_inplace_subtract*/     0,
    /*nb_inplace_multiply*/     0,
    /*nb_inplace_divide*/       0,
    /*nb_inplace_remainder*/    0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       0,
    /*nb_inplace_rshift*/       0,
    /*nb_inplace_and*/          0,
    /*nb_inplace_xor*/          0,
    /*nb_inplace_or*/           0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};


static PyGetSetDef SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PaintEngineFeature",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        &SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PaintEngineFeatures",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             &SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    if (!PyType_IsSubtype(type, &SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type))
        return 0;

    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlag___and__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::DirtyFlags >::toPython(
        ((QPaintEngine::DirtyFlags) ((SbkEnumObject*)self)->ob_ival)
        & Shiboken::Converter< QPaintEngine::DirtyFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlag___or__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::DirtyFlags >::toPython(
        ((QPaintEngine::DirtyFlags) ((SbkEnumObject*)self)->ob_ival)
        | Shiboken::Converter< QPaintEngine::DirtyFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlag___xor__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::DirtyFlags >::toPython(
        ((QPaintEngine::DirtyFlags) ((SbkEnumObject*)self)->ob_ival)
        ^ Shiboken::Converter< QPaintEngine::DirtyFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlag___invert__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QPaintEngine::DirtyFlags >::toPython(
        ~ Shiboken::Converter< QPaintEngine::DirtyFlags >::toCpp(self)
    );
}


static PyNumberMethods SbkPySide_QtGui_QPaintEngine_DirtyFlag_as_number = {
    /*nb_add*/                  0,
    /*nb_subtract*/             0,
    /*nb_multiply*/             0,
    /*nb_divide*/               0,
    /*nb_remainder*/            0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             0,
    /*nb_positive*/             0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)SbkPySide_QtGui_QPaintEngine_DirtyFlag___invert__,
    /*nb_lshift*/               0,
    /*nb_rshift*/               0,
    /*nb_and*/                  (binaryfunc)SbkPySide_QtGui_QPaintEngine_DirtyFlag___and__,
    /*nb_xor*/                  (binaryfunc)SbkPySide_QtGui_QPaintEngine_DirtyFlag___xor__,
    /*nb_or*/                   (binaryfunc)SbkPySide_QtGui_QPaintEngine_DirtyFlag___or__,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          0,
    /*nb_inplace_subtract*/     0,
    /*nb_inplace_multiply*/     0,
    /*nb_inplace_divide*/       0,
    /*nb_inplace_remainder*/    0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       0,
    /*nb_inplace_rshift*/       0,
    /*nb_inplace_and*/          0,
    /*nb_inplace_xor*/          0,
    /*nb_inplace_or*/           0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};


static PyGetSetDef SbkPySide_QtGui_QPaintEngine_DirtyFlag_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_DirtyFlag_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "DirtyFlag",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        &SbkPySide_QtGui_QPaintEngine_DirtyFlag_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QPaintEngine_DirtyFlag_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_DirtyFlag_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlag_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QPaintEngine_DirtyFlags_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "DirtyFlags",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             &SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QPaintEngine_DirtyFlags_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QPaintEngine_DirtyFlags_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    if (!PyType_IsSubtype(type, &SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type))
        return 0;

    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}



PyAPI_FUNC(void)
init_QPaintEngine(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQPaintEngine_Type);

    // Fill type discovery information
    SbkQPaintEngine_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQPaintEngine_Type.type_discovery->addTypeDiscoveryFunction(&SbkQPaintEngine_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQPaintEngine_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQPaintEngine_Type));
    PyModule_AddObject(module, "QPaintEngine",
        ((PyObject*)&SbkQPaintEngine_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: PolygonDrawMode
    SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_POLYGONDRAWMODE_IDX] = &SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "PolygonDrawMode",((PyObject*)&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type,
        (long) QPaintEngine::OddEvenMode, "OddEvenMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "OddEvenMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type.tp_dict,
        "OddEvenMode", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type,
        (long) QPaintEngine::WindingMode, "WindingMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "WindingMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type.tp_dict,
        "WindingMode", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type,
        (long) QPaintEngine::ConvexMode, "ConvexMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "ConvexMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type.tp_dict,
        "ConvexMode", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type,
        (long) QPaintEngine::PolylineMode, "PolylineMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PolylineMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PolygonDrawMode_Type.tp_dict,
        "PolylineMode", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QPaintEngine::PolygonDrawMode");

    // init enum class: Type
    SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_TYPE_IDX] = &SbkPySide_QtGui_QPaintEngine_Type_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_Type_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_Type_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "Type",((PyObject*)&SbkPySide_QtGui_QPaintEngine_Type_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::X11, "X11");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "X11", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "X11", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::Windows, "Windows");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Windows", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "Windows", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::QuickDraw, "QuickDraw");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "QuickDraw", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "QuickDraw", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::CoreGraphics, "CoreGraphics");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "CoreGraphics", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "CoreGraphics", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::MacPrinter, "MacPrinter");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "MacPrinter", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "MacPrinter", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::QWindowSystem, "QWindowSystem");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "QWindowSystem", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "QWindowSystem", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::PostScript, "PostScript");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PostScript", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "PostScript", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::OpenGL, "OpenGL");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "OpenGL", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "OpenGL", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::Picture, "Picture");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Picture", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "Picture", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::SVG, "SVG");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "SVG", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "SVG", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::Raster, "Raster");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Raster", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "Raster", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::Direct3D, "Direct3D");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Direct3D", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "Direct3D", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::Pdf, "Pdf");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Pdf", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "Pdf", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::OpenVG, "OpenVG");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "OpenVG", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "OpenVG", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::OpenGL2, "OpenGL2");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "OpenGL2", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "OpenGL2", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::PaintBuffer, "PaintBuffer");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PaintBuffer", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "PaintBuffer", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::User, "User");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "User", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "User", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_Type_Type,
        (long) QPaintEngine::MaxUser, "MaxUser");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "MaxUser", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_Type_Type.tp_dict,
        "MaxUser", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QPaintEngine::Type");

    // init enum class: PaintEngineFeature
    SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_PAINTENGINEFEATURE_IDX] = &SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "PaintEngineFeature",((PyObject*)&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type));

    // init flags class: QFlags<QPaintEngine::PaintEngineFeature>
    SbkPySide_QtGuiTypes[SBK_QFLAGS_QPAINTENGINE_PAINTENGINEFEATURE__IDX] = &SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "PaintEngineFeatures",((PyObject*)&SbkPySide_QtGui_QPaintEngine_PaintEngineFeatures_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PrimitiveTransform, "PrimitiveTransform");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PrimitiveTransform", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PrimitiveTransform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PatternTransform, "PatternTransform");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PatternTransform", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PatternTransform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PixmapTransform, "PixmapTransform");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PixmapTransform", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PixmapTransform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PatternBrush, "PatternBrush");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PatternBrush", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PatternBrush", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::LinearGradientFill, "LinearGradientFill");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "LinearGradientFill", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "LinearGradientFill", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::RadialGradientFill, "RadialGradientFill");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "RadialGradientFill", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "RadialGradientFill", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::ConicalGradientFill, "ConicalGradientFill");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "ConicalGradientFill", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "ConicalGradientFill", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::AlphaBlend, "AlphaBlend");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "AlphaBlend", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "AlphaBlend", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PorterDuff, "PorterDuff");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PorterDuff", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PorterDuff", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PainterPaths, "PainterPaths");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PainterPaths", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PainterPaths", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::Antialiasing, "Antialiasing");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "Antialiasing", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "Antialiasing", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::BrushStroke, "BrushStroke");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "BrushStroke", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "BrushStroke", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::ConstantOpacity, "ConstantOpacity");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "ConstantOpacity", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "ConstantOpacity", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::MaskedBrush, "MaskedBrush");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "MaskedBrush", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "MaskedBrush", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PerspectiveTransform, "PerspectiveTransform");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PerspectiveTransform", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PerspectiveTransform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::BlendModes, "BlendModes");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "BlendModes", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "BlendModes", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::ObjectBoundingModeGradients, "ObjectBoundingModeGradients");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "ObjectBoundingModeGradients", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "ObjectBoundingModeGradients", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::RasterOpModes, "RasterOpModes");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "RasterOpModes", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "RasterOpModes", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::PaintOutsidePaintEvent, "PaintOutsidePaintEvent");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "PaintOutsidePaintEvent", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "PaintOutsidePaintEvent", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type,
        (long) QPaintEngine::AllFeatures, "AllFeatures");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "AllFeatures", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_PaintEngineFeature_Type.tp_dict,
        "AllFeatures", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QPaintEngine::PaintEngineFeature");

    // init enum class: DirtyFlag
    SbkPySide_QtGuiTypes[SBK_QPAINTENGINE_DIRTYFLAG_IDX] = &SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "DirtyFlag",((PyObject*)&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type));

    // init flags class: QFlags<QPaintEngine::DirtyFlag>
    SbkPySide_QtGuiTypes[SBK_QFLAGS_QPAINTENGINE_DIRTYFLAG__IDX] = &SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type);
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
            "DirtyFlags",((PyObject*)&SbkPySide_QtGui_QPaintEngine_DirtyFlags_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyPen, "DirtyPen");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyPen", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyPen", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyBrush, "DirtyBrush");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyBrush", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyBrush", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyBrushOrigin, "DirtyBrushOrigin");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyBrushOrigin", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyBrushOrigin", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyFont, "DirtyFont");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyFont", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyFont", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyBackground, "DirtyBackground");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyBackground", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyBackground", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyBackgroundMode, "DirtyBackgroundMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyBackgroundMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyBackgroundMode", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyTransform, "DirtyTransform");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyTransform", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyTransform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyClipRegion, "DirtyClipRegion");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyClipRegion", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyClipRegion", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyClipPath, "DirtyClipPath");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyClipPath", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyClipPath", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyHints, "DirtyHints");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyHints", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyHints", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyCompositionMode, "DirtyCompositionMode");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyCompositionMode", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyCompositionMode", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyClipEnabled, "DirtyClipEnabled");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyClipEnabled", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyClipEnabled", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::DirtyOpacity, "DirtyOpacity");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "DirtyOpacity", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "DirtyOpacity", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type,
        (long) QPaintEngine::AllDirty, "AllDirty");
    PyDict_SetItemString(SbkQPaintEngine_Type.super.ht_type.tp_dict,
        "AllDirty", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QPaintEngine_DirtyFlag_Type.tp_dict,
        "AllDirty", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QPaintEngine::DirtyFlag");

    Shiboken::TypeResolver::createObjectTypeResolver<QPaintEngine >("QPaintEngine*");
    Shiboken::TypeResolver::createObjectTypeResolver<QPaintEngine >(typeid(QPaintEngine).name());
}


} // extern "C"

namespace Shiboken
{
// Copy C++ object implementation
// Converter implementations
} // namespace Shiboken

