/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qgraphicsitem_wrapper.h"

// Extra includes
#include <QList>
#include <qcoreevent.h>
#include <qcursor.h>
#include <qevent.h>
#include <qgraphicseffect.h>
#include <qgraphicsitem.h>
#include <qgraphicsscene.h>
#include <qgraphicssceneevent.h>
#include <qgraphicstransform.h>
#include <qgraphicswidget.h>
#include <qpainter.h>
#include <qpainterpath.h>
#include <qpoint.h>
#include <qpolygon.h>
#include <qrect.h>
#include <qregion.h>
#include <qsize.h>
#include <qstring.h>
#include <qstyleoption.h>
#include <qtransform.h>
#include <qvariant.h>
#include <qwidget.h>

using namespace Shiboken;


// Native ---------------------------------------------------------

QGraphicsItemWrapper::QGraphicsItemWrapper(QGraphicsItem * parent, QGraphicsScene * scene) : QGraphicsItem(parent, scene) {
    // ... middle
}

void QGraphicsItemWrapper::advance(int phase)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "advance"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::advance(phase);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(i)",
        phase
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QRectF QGraphicsItemWrapper::boundingRect() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "boundingRect"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsItem.boundingRect()' not implemented.");
        return QRectF();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QRectF();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QRectF >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.boundingRect");
        return QRectF()        ;
    }
    QRectF cpp_result(Shiboken::Converter<QRectF >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsItemWrapper::collidesWithItem(const QGraphicsItem * other, Qt::ItemSelectionMode mode) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "collidesWithItem"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::collidesWithItem(other, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QGraphicsItem* >::toPython(other),
        Shiboken::Converter<Qt::ItemSelectionMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.collidesWithItem");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QGraphicsItemWrapper::collidesWithPath(const QPainterPath & path, Qt::ItemSelectionMode mode) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "collidesWithPath"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::collidesWithPath(path, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QPainterPath& >::toPython(path),
        Shiboken::Converter<Qt::ItemSelectionMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.collidesWithPath");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsItemWrapper::contains(const QPointF & point) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contains"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::contains(point);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QPointF& >::toPython(point)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.contains");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsItemWrapper::contextMenuEvent(QGraphicsSceneContextMenuEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contextMenuEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::contextMenuEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneContextMenuEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::dragEnterEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragEnterEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragEnterEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::dragLeaveEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragLeaveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragLeaveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::dragMoveEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::dropEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dropEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dropEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QVariant QGraphicsItemWrapper::extension(const QVariant & variant) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "extension"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::extension(variant);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QVariant& >::toPython(variant)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.extension");
        return QVariant()        ;
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsItemWrapper::focusInEvent(QFocusEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "focusInEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::focusInEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QFocusEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::focusOutEvent(QFocusEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "focusOutEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::focusOutEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QFocusEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::hoverEnterEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverEnterEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverEnterEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneHoverEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::hoverLeaveEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverLeaveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverLeaveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneHoverEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::hoverMoveEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneHoverEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::inputMethodEvent(QInputMethodEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "inputMethodEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::inputMethodEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QInputMethodEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QVariant QGraphicsItemWrapper::inputMethodQuery(Qt::InputMethodQuery query) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "inputMethodQuery"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::inputMethodQuery(query);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<Qt::InputMethodQuery >::toPython(query)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.inputMethodQuery");
        return QVariant()        ;
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsItemWrapper::isObscuredBy(const QGraphicsItem * item) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isObscuredBy"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::isObscuredBy(item);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsItem* >::toPython(item)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.isObscuredBy");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

QVariant QGraphicsItemWrapper::itemChange(QGraphicsItem::GraphicsItemChange change, const QVariant & value)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "itemChange"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::itemChange(change, value);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QGraphicsItem::GraphicsItemChange >::toPython(change),
        Shiboken::Converter<QVariant& >::toPython(value)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.itemChange");
        return QVariant()        ;
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsItemWrapper::keyPressEvent(QKeyEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "keyPressEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::keyPressEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QKeyEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::keyReleaseEvent(QKeyEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "keyReleaseEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::keyReleaseEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QKeyEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::mouseDoubleClickEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseDoubleClickEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseDoubleClickEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneMouseEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::mouseMoveEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneMouseEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::mousePressEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mousePressEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mousePressEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneMouseEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsItemWrapper::mouseReleaseEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseReleaseEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseReleaseEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneMouseEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QPainterPath QGraphicsItemWrapper::opaqueArea() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "opaqueArea"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::opaqueArea();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPainterPath();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPainterPath >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.opaqueArea");
        return QPainterPath()        ;
    }
    QPainterPath cpp_result(Shiboken::Converter<QPainterPath >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsItemWrapper::paint(QPainter * painter, const QStyleOptionGraphicsItem * option, QWidget * widget)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "paint"));
    if (py_override.isNull()) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsItem.paint()' not implemented.");
        return ;
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OOO)",
        Shiboken::Converter<QPainter* >::toPython(painter),
        Shiboken::Converter<QStyleOptionGraphicsItem* >::toPython(option),
        Shiboken::Converter<QWidget* >::toPython(widget)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

bool QGraphicsItemWrapper::sceneEvent(QEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "sceneEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::sceneEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.sceneEvent");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QGraphicsItemWrapper::sceneEventFilter(QGraphicsItem * watched, QEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "sceneEventFilter"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::sceneEventFilter(watched, event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QGraphicsItem* >::toPython(watched),
        Shiboken::Converter<QEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.sceneEventFilter");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 1));
    return cpp_result;
}

QPainterPath QGraphicsItemWrapper::shape() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "shape"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::shape();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPainterPath();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPainterPath >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.shape");
        return QPainterPath()        ;
    }
    QPainterPath cpp_result(Shiboken::Converter<QPainterPath >::toCpp(py_result));
    return cpp_result;
}

int QGraphicsItemWrapper::type() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "type"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::type();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QGraphicsItem.type");
        return int(0)        ;
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsItemWrapper::wheelEvent(QGraphicsSceneWheelEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "wheelEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::wheelEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QGraphicsSceneWheelEvent* >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QGraphicsItemWrapper::~QGraphicsItemWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQGraphicsItem_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QGraphicsItemWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    assert(!sbkSelf->cptr);
    SbkBaseWrapperType* type = reinterpret_cast<SbkBaseWrapperType*>(self->ob_type);
    SbkBaseWrapperType* myType = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_IDX]);
    if (type == myType) {
        PyErr_SetString(PyExc_NotImplementedError,
            "'QGraphicsItem' represents a C++ abstract class and cannot be instantiated");
        return -1;
    }


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "QGraphicsItem", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return -1;

    if (numArgs == 0) {
        // QGraphicsItem(QGraphicsItem*,QGraphicsScene*)
        cptr = new QGraphicsItemWrapper();
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // QGraphicsItem(QGraphicsItem*,QGraphicsScene*)
            cptr = new QGraphicsItemWrapper(cpp_arg0);
            Shiboken::setParent(pyargs[0], self);
        } else if (Shiboken::Converter<QGraphicsScene* >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            QGraphicsScene* cpp_arg1 = Shiboken::Converter<QGraphicsScene* >::toCpp(pyargs[1]);
            // QGraphicsItem(QGraphicsItem*,QGraphicsScene*)
            cptr = new QGraphicsItemWrapper(cpp_arg0, cpp_arg1);
            Shiboken::setParent(pyargs[0], self);
        } else goto SbkQGraphicsItem_Init_TypeError;
    } else goto SbkQGraphicsItem_Init_TypeError;

    if (PyErr_Occurred()) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQGraphicsItem_Init_TypeError;

    sbkSelf->cptr = cptr;
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf);

    return 1;

    SbkQGraphicsItem_Init_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem = None, PySide.QtGui.QGraphicsScene = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem", overloads);
        return -1;
}

static PyObject*
SbkQGraphicsItemFunc_acceptDrops(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // acceptDrops()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::acceptDrops();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_acceptHoverEvents(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // acceptHoverEvents()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::acceptHoverEvents();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_acceptTouchEvents(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // acceptTouchEvents()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::acceptTouchEvents();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_acceptedMouseButtons(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // acceptedMouseButtons()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QFlags<Qt::MouseButton> cpp_result = cppSelf->QGraphicsItem::acceptedMouseButtons();
    py_result = Shiboken::Converter<QFlags<Qt::MouseButton> >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_acceptsHoverEvents(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // acceptsHoverEvents()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::acceptsHoverEvents();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_addToIndex(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // addToIndex()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::addToIndex();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_advance(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // advance(int)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::advance(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_advance_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_advance_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.advance", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_boundingRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // boundingRect()const
    if (SbkBaseWrapper_containsCppWrapper(self)) {
        PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsItem.boundingRect()' not implemented.");
        return 0;
    }
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->boundingRect();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_boundingRegion(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QTransform& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTransform* cpp_arg0 = Shiboken::Converter<QTransform* >::toCpp(arg);
        // boundingRegion(QTransform)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRegion cpp_result = cppSelf->QGraphicsItem::boundingRegion(*cpp_arg0);
        py_result = Shiboken::Converter<QRegion >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_boundingRegion_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_boundingRegion_TypeError:
        const char* overloads[] = {"PySide.QtGui.QTransform", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.boundingRegion", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_boundingRegionGranularity(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // boundingRegionGranularity()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::boundingRegionGranularity();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_cacheMode(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // cacheMode()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem::CacheMode cpp_result = cppSelf->QGraphicsItem::cacheMode();
    py_result = Shiboken::Converter<QGraphicsItem::CacheMode >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_childItems(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // childItems()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QList<QGraphicsItem * > cpp_result = cppSelf->QGraphicsItem::childItems();
    py_result = Shiboken::Converter<QList<QGraphicsItem * > >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_childrenBoundingRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // childrenBoundingRect()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QGraphicsItem::childrenBoundingRect();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_clearFocus(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // clearFocus()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::clearFocus();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_clipPath(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // clipPath()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPainterPath cpp_result = cppSelf->QGraphicsItem::clipPath();
    py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_collidesWithItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "collidesWithItem", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // collidesWithItem(const QGraphicsItem*,Qt::ItemSelectionMode)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QGraphicsItem::collidesWithItem(cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else if (Shiboken::Converter<Qt::ItemSelectionMode >::isConvertible(pyargs[1])) {
            Qt::ItemSelectionMode cpp_arg1 = Shiboken::Converter<Qt::ItemSelectionMode >::toCpp(pyargs[1]);
            // collidesWithItem(const QGraphicsItem*,Qt::ItemSelectionMode)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QGraphicsItem::collidesWithItem(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_collidesWithItem_TypeError;
    } else goto SbkQGraphicsItemFunc_collidesWithItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_collidesWithItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtCore.Qt.ItemSelectionMode = Qt.IntersectsItemShape", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.collidesWithItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_collidesWithPath(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "collidesWithPath", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // collidesWithPath(QPainterPath,Qt::ItemSelectionMode)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QGraphicsItem::collidesWithPath(*cpp_arg0);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else if (Shiboken::Converter<Qt::ItemSelectionMode >::isConvertible(pyargs[1])) {
            Qt::ItemSelectionMode cpp_arg1 = Shiboken::Converter<Qt::ItemSelectionMode >::toCpp(pyargs[1]);
            // collidesWithPath(QPainterPath,Qt::ItemSelectionMode)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            bool cpp_result = cppSelf->QGraphicsItem::collidesWithPath(*cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_collidesWithPath_TypeError;
    } else goto SbkQGraphicsItemFunc_collidesWithPath_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_collidesWithPath_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath, PySide.QtCore.Qt.ItemSelectionMode = Qt.IntersectsItemShape", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.collidesWithPath", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_collidingItems(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // collidingItems(Qt::ItemSelectionMode)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QList<QGraphicsItem * > cpp_result = cppSelf->QGraphicsItem::collidingItems();
        py_result = Shiboken::Converter<QList<QGraphicsItem * > >::toPython(cpp_result);
    } else if (Shiboken::Converter<Qt::ItemSelectionMode >::isConvertible(arg)) {
        Qt::ItemSelectionMode cpp_arg0 = Shiboken::Converter<Qt::ItemSelectionMode >::toCpp(arg);
        // collidingItems(Qt::ItemSelectionMode)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QList<QGraphicsItem * > cpp_result = cppSelf->QGraphicsItem::collidingItems(cpp_arg0);
        py_result = Shiboken::Converter<QList<QGraphicsItem * > >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_collidingItems_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_collidingItems_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.ItemSelectionMode = Qt.IntersectsItemShape", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.collidingItems", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_commonAncestorItem(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // commonAncestorItem(const QGraphicsItem*)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::commonAncestorItem(cpp_arg0);
        py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
        Shiboken::setParent(self, py_result);
    } else goto SbkQGraphicsItemFunc_commonAncestorItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_commonAncestorItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.commonAncestorItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_contains(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QPointF& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(arg);
        if (!Shiboken::Converter<QPointF& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // contains(QPointF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::contains(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_contains_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_contains_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.contains", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_contextMenuEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneContextMenuEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneContextMenuEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneContextMenuEvent* >::toCpp(arg);
        // contextMenuEvent(QGraphicsSceneContextMenuEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::contextMenuEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_contextMenuEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_contextMenuEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneContextMenuEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.contextMenuEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_cursor(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // cursor()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QCursor cpp_result = cppSelf->QGraphicsItem::cursor();
    py_result = Shiboken::Converter<QCursor >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_data(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // data(int)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QGraphicsItem::data(cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_data_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_data_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.data", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_deviceTransform(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QTransform& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QTransform* cpp_arg0 = Shiboken::Converter<QTransform* >::toCpp(arg);
        // deviceTransform(QTransform)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QTransform cpp_result = cppSelf->QGraphicsItem::deviceTransform(*cpp_arg0);
        py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_deviceTransform_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_deviceTransform_TypeError:
        const char* overloads[] = {"PySide.QtGui.QTransform", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.deviceTransform", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_dragEnterEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneDragDropEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneDragDropEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toCpp(arg);
        // dragEnterEvent(QGraphicsSceneDragDropEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::dragEnterEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_dragEnterEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_dragEnterEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneDragDropEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.dragEnterEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_dragLeaveEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneDragDropEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneDragDropEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toCpp(arg);
        // dragLeaveEvent(QGraphicsSceneDragDropEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::dragLeaveEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_dragLeaveEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_dragLeaveEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneDragDropEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.dragLeaveEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_dragMoveEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneDragDropEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneDragDropEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toCpp(arg);
        // dragMoveEvent(QGraphicsSceneDragDropEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::dragMoveEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_dragMoveEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_dragMoveEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneDragDropEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.dragMoveEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_dropEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneDragDropEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneDragDropEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneDragDropEvent* >::toCpp(arg);
        // dropEvent(QGraphicsSceneDragDropEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::dropEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_dropEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_dropEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneDragDropEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.dropEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_effectiveOpacity(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // effectiveOpacity()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::effectiveOpacity();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_ensureVisible(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "ensureVisible", 0, 6, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4]), &(pyargs[5])))
        return 0;

    if (numArgs == 0) {
        // ensureVisible(QRectF,int,int)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::ensureVisible();
    } else if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        if (numArgs == 4) {
            // ensureVisible(qreal,qreal,qreal,qreal,int,int)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::ensureVisible(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        } else if (PyNumber_Check(pyargs[4])) {
            int cpp_arg4 = Shiboken::Converter<int >::toCpp(pyargs[4]);
            if (numArgs == 5) {
                // ensureVisible(qreal,qreal,qreal,qreal,int,int)
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                cppSelf->QGraphicsItem::ensureVisible(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
            } else if (PyNumber_Check(pyargs[5])) {
                int cpp_arg5 = Shiboken::Converter<int >::toCpp(pyargs[5]);
                // ensureVisible(qreal,qreal,qreal,qreal,int,int)
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                cppSelf->QGraphicsItem::ensureVisible(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4, cpp_arg5);
            } else goto SbkQGraphicsItemFunc_ensureVisible_TypeError;
        } else goto SbkQGraphicsItemFunc_ensureVisible_TypeError;
    } else if (Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        if (numArgs == 1) {
            // ensureVisible(QRectF,int,int)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::ensureVisible(*cpp_arg0);
        } else if (PyNumber_Check(pyargs[1])) {
            int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // ensureVisible(QRectF,int,int)
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                cppSelf->QGraphicsItem::ensureVisible(*cpp_arg0, cpp_arg1);
            } else if (PyNumber_Check(pyargs[2])) {
                int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
                // ensureVisible(QRectF,int,int)
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                cppSelf->QGraphicsItem::ensureVisible(*cpp_arg0, cpp_arg1, cpp_arg2);
            } else goto SbkQGraphicsItemFunc_ensureVisible_TypeError;
        } else goto SbkQGraphicsItemFunc_ensureVisible_TypeError;
    } else goto SbkQGraphicsItemFunc_ensureVisible_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_ensureVisible_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF = QRectF(), int = 50, int = 50", "float, float, float, float, int = 50, int = 50", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.ensureVisible", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_extension(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QVariant& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QVariant > cpp_arg0_auto_ptr;
        QVariant* cpp_arg0 = Shiboken::Converter<QVariant* >::toCpp(arg);
        if (!Shiboken::Converter<QVariant& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QVariant >(cpp_arg0);
        // extension(QVariant)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QGraphicsItem::extension(*cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_extension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_extension_TypeError:
        const char* overloads[] = {"PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.extension", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_filtersChildEvents(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // filtersChildEvents()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::filtersChildEvents();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_flags(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // flags()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QFlags<QGraphicsItem::GraphicsItemFlag> cpp_result = cppSelf->QGraphicsItem::flags();
    py_result = Shiboken::Converter<QFlags<QGraphicsItem::GraphicsItemFlag> >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_focusInEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QFocusEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QFocusEvent* cpp_arg0 = Shiboken::Converter<QFocusEvent* >::toCpp(arg);
        // focusInEvent(QFocusEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::focusInEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_focusInEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_focusInEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QFocusEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.focusInEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_focusItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // focusItem()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::focusItem();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_focusOutEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QFocusEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QFocusEvent* cpp_arg0 = Shiboken::Converter<QFocusEvent* >::toCpp(arg);
        // focusOutEvent(QFocusEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::focusOutEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_focusOutEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_focusOutEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QFocusEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.focusOutEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_focusProxy(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // focusProxy()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::focusProxy();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_focusScopeItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // focusScopeItem()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::focusScopeItem();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_grabKeyboard(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // grabKeyboard()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::grabKeyboard();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_grabMouse(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // grabMouse()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::grabMouse();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_graphicsEffect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // graphicsEffect()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsEffect * cpp_result = cppSelf->QGraphicsItem::graphicsEffect();
    py_result = Shiboken::Converter<QGraphicsEffect* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_group(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // group()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItemGroup * cpp_result = cppSelf->QGraphicsItem::group();
    py_result = Shiboken::Converter<QGraphicsItemGroup* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_handlesChildEvents(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // handlesChildEvents()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::handlesChildEvents();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_hasCursor(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // hasCursor()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::hasCursor();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_hasFocus(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // hasFocus()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::hasFocus();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_hide(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // hide()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::hide();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_hoverEnterEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneHoverEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneHoverEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneHoverEvent* >::toCpp(arg);
        // hoverEnterEvent(QGraphicsSceneHoverEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::hoverEnterEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_hoverEnterEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_hoverEnterEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneHoverEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.hoverEnterEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_hoverLeaveEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneHoverEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneHoverEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneHoverEvent* >::toCpp(arg);
        // hoverLeaveEvent(QGraphicsSceneHoverEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::hoverLeaveEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_hoverLeaveEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_hoverLeaveEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneHoverEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.hoverLeaveEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_hoverMoveEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneHoverEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneHoverEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneHoverEvent* >::toCpp(arg);
        // hoverMoveEvent(QGraphicsSceneHoverEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::hoverMoveEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_hoverMoveEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_hoverMoveEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneHoverEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.hoverMoveEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_inputMethodEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QInputMethodEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QInputMethodEvent* cpp_arg0 = Shiboken::Converter<QInputMethodEvent* >::toCpp(arg);
        // inputMethodEvent(QInputMethodEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::inputMethodEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_inputMethodEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_inputMethodEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QInputMethodEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.inputMethodEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_inputMethodHints(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // inputMethodHints()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QFlags<Qt::InputMethodHint> cpp_result = cppSelf->QGraphicsItem::inputMethodHints();
    py_result = Shiboken::Converter<QFlags<Qt::InputMethodHint> >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_inputMethodQuery(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<Qt::InputMethodQuery >::isConvertible(arg)) {
        Qt::InputMethodQuery cpp_arg0 = Shiboken::Converter<Qt::InputMethodQuery >::toCpp(arg);
        // inputMethodQuery(Qt::InputMethodQuery)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QGraphicsItem::inputMethodQuery(cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_inputMethodQuery_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_inputMethodQuery_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.InputMethodQuery", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.inputMethodQuery", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_installSceneEventFilter(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // installSceneEventFilter(QGraphicsItem*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::installSceneEventFilter(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_installSceneEventFilter_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_installSceneEventFilter_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.installSceneEventFilter", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_isActive(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isActive()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isActive();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isAncestorOf(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // isAncestorOf(const QGraphicsItem*)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isAncestorOf(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_isAncestorOf_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_isAncestorOf_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.isAncestorOf", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_isBlockedByModalPanel(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isBlockedByModalPanel(QGraphicsItem**)const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    // Begin code injection
    
    QGraphicsItem *item_ = NULL;
    bool retval_ = cppSelf->isBlockedByModalPanel(&item_);
    py_result = Shiboken::makeTuple(retval_, item_);
    
    // End of code injection



    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isClipped(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isClipped()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isClipped();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isEnabled(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isEnabled()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isEnabled();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isObscured(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_isObscured_TypeError;

    if (!PyArg_UnpackTuple(args, "isObscured", 0, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 0) {
        // isObscured()const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isObscured();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // isObscured(qreal,qreal,qreal,qreal)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isObscured(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // isObscured(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isObscured(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_isObscured_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_isObscured_TypeError:
        const char* overloads[] = {"", "PySide.QtCore.QRectF", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.isObscured", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_isObscuredBy(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // isObscuredBy(const QGraphicsItem*)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isObscuredBy(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_isObscuredBy_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_isObscuredBy_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.isObscuredBy", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_isPanel(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isPanel()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isPanel();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isSelected(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isSelected()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isSelected();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isUnderMouse(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isUnderMouse()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isUnderMouse();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isVisible(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isVisible()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isVisible();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isVisibleTo(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // isVisibleTo(const QGraphicsItem*)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::isVisibleTo(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_isVisibleTo_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_isVisibleTo_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.isVisibleTo", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_isWidget(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isWidget()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isWidget();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_isWindow(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isWindow()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QGraphicsItem::isWindow();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_itemChange(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "itemChange", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QGraphicsItem::GraphicsItemChange >::isConvertible(pyargs[0]) && Shiboken::Converter<QVariant& >::isConvertible(pyargs[1])) {
        QGraphicsItem::GraphicsItemChange cpp_arg0 = Shiboken::Converter<QGraphicsItem::GraphicsItemChange >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
        QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QVariant& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
        // itemChange(QGraphicsItem::GraphicsItemChange,QVariant)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QGraphicsItem::itemChange(cpp_arg0, *cpp_arg1);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_itemChange_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_itemChange_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.GraphicsItemChange, PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.itemChange", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_itemTransform(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "itemTransform", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // itemTransform(const QGraphicsItem*,bool*)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QTransform cpp_result = cppSelf->QGraphicsItem::itemTransform(cpp_arg0);
            py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);
        } else if (PyBool_Check(pyargs[1])) {
            bool * cpp_arg1 = Shiboken::Converter<bool * >::toCpp(pyargs[1]);
            // itemTransform(const QGraphicsItem*,bool*)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QTransform cpp_result = cppSelf->QGraphicsItem::itemTransform(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_itemTransform_TypeError;
    } else goto SbkQGraphicsItemFunc_itemTransform_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_itemTransform_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, bool = 0", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.itemTransform", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_keyPressEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QKeyEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QKeyEvent* cpp_arg0 = Shiboken::Converter<QKeyEvent* >::toCpp(arg);
        // keyPressEvent(QKeyEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::keyPressEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_keyPressEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_keyPressEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QKeyEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.keyPressEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_keyReleaseEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QKeyEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QKeyEvent* cpp_arg0 = Shiboken::Converter<QKeyEvent* >::toCpp(arg);
        // keyReleaseEvent(QKeyEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::keyReleaseEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_keyReleaseEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_keyReleaseEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QKeyEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.keyReleaseEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapFromItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 4)
        goto SbkQGraphicsItemFunc_mapFromItem_TypeError;

    if (!PyArg_UnpackTuple(args, "mapFromItem", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
            qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            if (numArgs == 3) {
                // mapFromItem(const QGraphicsItem*,qreal,qreal)const
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                QPointF cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
            } else if (numArgs == 5 && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4])) {
                qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
                qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
                // mapFromItem(const QGraphicsItem*,qreal,qreal,qreal,qreal)const
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
                py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
            } else goto SbkQGraphicsItemFunc_mapFromItem_TypeError;
        } else if (numArgs == 2 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QRectF > cpp_arg1_auto_ptr;
            QRectF* cpp_arg1 = Shiboken::Converter<QRectF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QRectF >(cpp_arg1);
            // mapFromItem(const QGraphicsItem*,QRectF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QPolygonF > cpp_arg1_auto_ptr;
            QPolygonF* cpp_arg1 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg1);
            // mapFromItem(const QGraphicsItem*,QPolygonF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QPointF > cpp_arg1_auto_ptr;
            QPointF* cpp_arg1 = Shiboken::Converter<QPointF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QPointF >(cpp_arg1);
            // mapFromItem(const QGraphicsItem*,QPointF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QPainterPath* cpp_arg1 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[1]);
            // mapFromItem(const QGraphicsItem*,QPainterPath)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPainterPath cpp_result = cppSelf->QGraphicsItem::mapFromItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapFromItem_TypeError;
    } else goto SbkQGraphicsItemFunc_mapFromItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapFromItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtGui.QPainterPath", "PySide.QtGui.QGraphicsItem, PySide.QtCore.QPointF", "PySide.QtGui.QGraphicsItem, PySide.QtGui.QPolygonF", "PySide.QtGui.QGraphicsItem, PySide.QtCore.QRectF", "PySide.QtGui.QGraphicsItem, float, float", "PySide.QtGui.QGraphicsItem, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapFromItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapFromParent(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3)
        goto SbkQGraphicsItemFunc_mapFromParent_TypeError;

    if (!PyArg_UnpackTuple(args, "mapFromParent", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // mapFromParent(qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapFromParent(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 4 && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            // mapFromParent(qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromParent(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapFromParent_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapFromParent(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPolygonF > cpp_arg0_auto_ptr;
        QPolygonF* cpp_arg0 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg0);
        // mapFromParent(QPolygonF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // mapFromParent(QPointF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPointF cpp_result = cppSelf->QGraphicsItem::mapFromParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[0]);
        // mapFromParent(QPainterPath)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPainterPath cpp_result = cppSelf->QGraphicsItem::mapFromParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapFromParent_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapFromParent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath", "PySide.QtCore.QPointF", "PySide.QtGui.QPolygonF", "PySide.QtCore.QRectF", "float, float", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapFromParent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapFromScene(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3)
        goto SbkQGraphicsItemFunc_mapFromScene_TypeError;

    if (!PyArg_UnpackTuple(args, "mapFromScene", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // mapFromScene(qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapFromScene(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 4 && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            // mapFromScene(qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromScene(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapFromScene_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapFromScene(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPolygonF > cpp_arg0_auto_ptr;
        QPolygonF* cpp_arg0 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg0);
        // mapFromScene(QPolygonF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapFromScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // mapFromScene(QPointF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPointF cpp_result = cppSelf->QGraphicsItem::mapFromScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[0]);
        // mapFromScene(QPainterPath)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPainterPath cpp_result = cppSelf->QGraphicsItem::mapFromScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapFromScene_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapFromScene_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath", "PySide.QtCore.QPointF", "PySide.QtGui.QPolygonF", "PySide.QtCore.QRectF", "float, float", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapFromScene", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectFromItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3 || numArgs == 4)
        goto SbkQGraphicsItemFunc_mapRectFromItem_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectFromItem", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 5 && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4])) {
            qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
            // mapRectFromItem(const QGraphicsItem*,qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromItem(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
            py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QRectF > cpp_arg1_auto_ptr;
            QRectF* cpp_arg1 = Shiboken::Converter<QRectF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QRectF >(cpp_arg1);
            // mapRectFromItem(const QGraphicsItem*,QRectF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapRectFromItem_TypeError;
    } else goto SbkQGraphicsItemFunc_mapRectFromItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectFromItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtCore.QRectF", "PySide.QtGui.QGraphicsItem, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectFromItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectFromParent(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_mapRectFromParent_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectFromParent", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // mapRectFromParent(qreal,qreal,qreal,qreal)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromParent(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapRectFromParent(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromParent(*cpp_arg0);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapRectFromParent_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectFromParent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectFromParent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectFromScene(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_mapRectFromScene_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectFromScene", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // mapRectFromScene(qreal,qreal,qreal,qreal)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromScene(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapRectFromScene(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectFromScene(*cpp_arg0);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapRectFromScene_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectFromScene_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectFromScene", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectToItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3 || numArgs == 4)
        goto SbkQGraphicsItemFunc_mapRectToItem_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectToItem", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 5 && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4])) {
            qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
            // mapRectToItem(const QGraphicsItem*,qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToItem(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
            py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QRectF > cpp_arg1_auto_ptr;
            QRectF* cpp_arg1 = Shiboken::Converter<QRectF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QRectF >(cpp_arg1);
            // mapRectToItem(const QGraphicsItem*,QRectF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapRectToItem_TypeError;
    } else goto SbkQGraphicsItemFunc_mapRectToItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectToItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtCore.QRectF", "PySide.QtGui.QGraphicsItem, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectToItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectToParent(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_mapRectToParent_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectToParent", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // mapRectToParent(qreal,qreal,qreal,qreal)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToParent(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapRectToParent(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToParent(*cpp_arg0);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapRectToParent_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectToParent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectToParent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapRectToScene(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_mapRectToScene_TypeError;

    if (!PyArg_UnpackTuple(args, "mapRectToScene", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // mapRectToScene(qreal,qreal,qreal,qreal)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToScene(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapRectToScene(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QRectF cpp_result = cppSelf->QGraphicsItem::mapRectToScene(*cpp_arg0);
        py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapRectToScene_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapRectToScene_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapRectToScene", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapToItem(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 4)
        goto SbkQGraphicsItemFunc_mapToItem_TypeError;

    if (!PyArg_UnpackTuple(args, "mapToItem", 2, 5, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3]), &(pyargs[4])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
            qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            if (numArgs == 3) {
                // mapToItem(const QGraphicsItem*,qreal,qreal)const
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                QPointF cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, cpp_arg1, cpp_arg2);
                py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
            } else if (numArgs == 5 && PyNumber_Check(pyargs[3]) && PyNumber_Check(pyargs[4])) {
                qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
                qreal cpp_arg4 = Shiboken::Converter<qreal >::toCpp(pyargs[4]);
                // mapToItem(const QGraphicsItem*,qreal,qreal,qreal,qreal)const
                QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
                QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3, cpp_arg4);
                py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
            } else goto SbkQGraphicsItemFunc_mapToItem_TypeError;
        } else if (numArgs == 2 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QRectF > cpp_arg1_auto_ptr;
            QRectF* cpp_arg1 = Shiboken::Converter<QRectF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QRectF >(cpp_arg1);
            // mapToItem(const QGraphicsItem*,QRectF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QPolygonF > cpp_arg1_auto_ptr;
            QPolygonF* cpp_arg1 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg1);
            // mapToItem(const QGraphicsItem*,QPolygonF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QPointF > cpp_arg1_auto_ptr;
            QPointF* cpp_arg1 = Shiboken::Converter<QPointF* >::toCpp(pyargs[1]);
            if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QPointF >(cpp_arg1);
            // mapToItem(const QGraphicsItem*,QPointF)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 2 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QPainterPath* cpp_arg1 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[1]);
            // mapToItem(const QGraphicsItem*,QPainterPath)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPainterPath cpp_result = cppSelf->QGraphicsItem::mapToItem(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapToItem_TypeError;
    } else goto SbkQGraphicsItemFunc_mapToItem_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapToItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtGui.QPainterPath", "PySide.QtGui.QGraphicsItem, PySide.QtCore.QPointF", "PySide.QtGui.QGraphicsItem, PySide.QtGui.QPolygonF", "PySide.QtGui.QGraphicsItem, PySide.QtCore.QRectF", "PySide.QtGui.QGraphicsItem, float, float", "PySide.QtGui.QGraphicsItem, float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapToItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapToParent(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3)
        goto SbkQGraphicsItemFunc_mapToParent_TypeError;

    if (!PyArg_UnpackTuple(args, "mapToParent", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // mapToParent(qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapToParent(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 4 && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            // mapToParent(qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToParent(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapToParent_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapToParent(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPolygonF > cpp_arg0_auto_ptr;
        QPolygonF* cpp_arg0 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg0);
        // mapToParent(QPolygonF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // mapToParent(QPointF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPointF cpp_result = cppSelf->QGraphicsItem::mapToParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[0]);
        // mapToParent(QPainterPath)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPainterPath cpp_result = cppSelf->QGraphicsItem::mapToParent(*cpp_arg0);
        py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapToParent_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapToParent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath", "PySide.QtCore.QPointF", "PySide.QtGui.QPolygonF", "PySide.QtCore.QRectF", "float, float", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapToParent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mapToScene(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 3)
        goto SbkQGraphicsItemFunc_mapToScene_TypeError;

    if (!PyArg_UnpackTuple(args, "mapToScene", 1, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // mapToScene(qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPointF cpp_result = cppSelf->QGraphicsItem::mapToScene(cpp_arg0, cpp_arg1);
            py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
        } else if (numArgs == 4 && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
            qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
            qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
            // mapToScene(qreal,qreal,qreal,qreal)const
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToScene(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
            py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
        } else goto SbkQGraphicsItemFunc_mapToScene_TypeError;
    } else if (numArgs == 1 && Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // mapToScene(QRectF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPolygonF > cpp_arg0_auto_ptr;
        QPolygonF* cpp_arg0 = Shiboken::Converter<QPolygonF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPolygonF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPolygonF >(cpp_arg0);
        // mapToScene(QPolygonF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPolygonF cpp_result = cppSelf->QGraphicsItem::mapToScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPolygonF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // mapToScene(QPointF)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPointF cpp_result = cppSelf->QGraphicsItem::mapToScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    } else if (numArgs == 1 && Shiboken::Converter<QPainterPath& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainterPath* cpp_arg0 = Shiboken::Converter<QPainterPath* >::toCpp(pyargs[0]);
        // mapToScene(QPainterPath)const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        QPainterPath cpp_result = cppSelf->QGraphicsItem::mapToScene(*cpp_arg0);
        py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_mapToScene_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_mapToScene_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainterPath", "PySide.QtCore.QPointF", "PySide.QtGui.QPolygonF", "PySide.QtCore.QRectF", "float, float", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.mapToScene", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mouseDoubleClickEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneMouseEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneMouseEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneMouseEvent* >::toCpp(arg);
        // mouseDoubleClickEvent(QGraphicsSceneMouseEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::mouseDoubleClickEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_mouseDoubleClickEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_mouseDoubleClickEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneMouseEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.mouseDoubleClickEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mouseMoveEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneMouseEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneMouseEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneMouseEvent* >::toCpp(arg);
        // mouseMoveEvent(QGraphicsSceneMouseEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::mouseMoveEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_mouseMoveEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_mouseMoveEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneMouseEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.mouseMoveEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mousePressEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneMouseEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneMouseEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneMouseEvent* >::toCpp(arg);
        // mousePressEvent(QGraphicsSceneMouseEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::mousePressEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_mousePressEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_mousePressEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneMouseEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.mousePressEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_mouseReleaseEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneMouseEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneMouseEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneMouseEvent* >::toCpp(arg);
        // mouseReleaseEvent(QGraphicsSceneMouseEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::mouseReleaseEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_mouseReleaseEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_mouseReleaseEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneMouseEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.mouseReleaseEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_moveBy(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "moveBy", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // moveBy(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::moveBy(cpp_arg0, cpp_arg1);
    } else goto SbkQGraphicsItemFunc_moveBy_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_moveBy_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.moveBy", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_opacity(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // opacity()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::opacity();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_opaqueArea(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // opaqueArea()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPainterPath cpp_result = cppSelf->QGraphicsItem::opaqueArea();
    py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_paint(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "paint", 2, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (Shiboken::Converter<QPainter* >::isConvertible(pyargs[0]) && Shiboken::Converter<QStyleOptionGraphicsItem* >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainter* cpp_arg0 = Shiboken::Converter<QPainter* >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QStyleOptionGraphicsItem* cpp_arg1 = Shiboken::Converter<QStyleOptionGraphicsItem* >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // paint(QPainter*,const QStyleOptionGraphicsItem*,QWidget*)
            if (SbkBaseWrapper_containsCppWrapper(self)) {
                PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsItem.paint()' not implemented.");
                return 0;
            }
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->paint(cpp_arg0, cpp_arg1);
        } else if (Shiboken::Converter<QWidget* >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            QWidget* cpp_arg2 = Shiboken::Converter<QWidget* >::toCpp(pyargs[2]);
            // paint(QPainter*,const QStyleOptionGraphicsItem*,QWidget*)
            if (SbkBaseWrapper_containsCppWrapper(self)) {
                PyErr_SetString(PyExc_NotImplementedError, "pure virtual method 'QGraphicsItem.paint()' not implemented.");
                return 0;
            }
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->paint(cpp_arg0, cpp_arg1, cpp_arg2);
        } else goto SbkQGraphicsItemFunc_paint_TypeError;
    } else goto SbkQGraphicsItemFunc_paint_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_paint_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainter, PySide.QtGui.QStyleOptionGraphicsItem, PySide.QtGui.QWidget = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.paint", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_panel(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // panel()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::panel();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_panelModality(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // panelModality()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem::PanelModality cpp_result = cppSelf->QGraphicsItem::panelModality();
    py_result = Shiboken::Converter<QGraphicsItem::PanelModality >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_parentItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // parentItem()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::parentItem();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(py_result, self);

    // Ownership transferences.
    SbkBaseWrapper_setOwnership(py_result, true);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_parentObject(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // parentObject()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsObject * cpp_result = cppSelf->QGraphicsItem::parentObject();
    py_result = Shiboken::Converter<QGraphicsObject* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_parentWidget(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // parentWidget()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsWidget * cpp_result = cppSelf->QGraphicsItem::parentWidget();
    py_result = Shiboken::Converter<QGraphicsWidget* >::toPython(cpp_result);
    Shiboken::setParent(py_result, self);

    // Ownership transferences.
    SbkBaseWrapper_setOwnership(py_result, true);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_pos(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // pos()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPointF cpp_result = cppSelf->QGraphicsItem::pos();
    py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_prepareGeometryChange(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // prepareGeometryChange()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::prepareGeometryChange();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_removeFromIndex(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // removeFromIndex()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::removeFromIndex();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_removeSceneEventFilter(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // removeSceneEventFilter(QGraphicsItem*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::removeSceneEventFilter(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_removeSceneEventFilter_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_removeSceneEventFilter_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.removeSceneEventFilter", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_resetTransform(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // resetTransform()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::resetTransform();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_rotate(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // rotate(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::rotate(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_rotate_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_rotate_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.rotate", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_rotation(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // rotation()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::rotation();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_scale(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    // invalid argument lengths
    if (numArgs == 1)
        goto SbkQGraphicsItemFunc_scale_TypeError;

    if (!PyArg_UnpackTuple(args, "scale", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 0) {
        // scale()const
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        qreal cpp_result = cppSelf->QGraphicsItem::scale();
        py_result = Shiboken::Converter<double >::toPython(cpp_result);
    } else if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // scale(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::scale(cpp_arg0, cpp_arg1);
        py_result = Py_None;
        Py_INCREF(Py_None);
    } else goto SbkQGraphicsItemFunc_scale_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_scale_TypeError:
        const char* overloads[] = {"", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.scale", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_scene(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // scene()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsScene * cpp_result = cppSelf->QGraphicsItem::scene();
    py_result = Shiboken::Converter<QGraphicsScene* >::toPython(cpp_result);
    Shiboken::setParent(py_result, self);

    // Ownership transferences.
    SbkBaseWrapper_setOwnership(py_result, true);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_sceneBoundingRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // sceneBoundingRect()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QGraphicsItem::sceneBoundingRect();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_sceneEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QEvent* cpp_arg0 = Shiboken::Converter<QEvent* >::toCpp(arg);
        // sceneEvent(QEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::sceneEvent(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_sceneEvent_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_sceneEvent_TypeError:
        const char* overloads[] = {"PySide.QtCore.QEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.sceneEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_sceneEventFilter(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "sceneEventFilter", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QGraphicsItem* >::isConvertible(pyargs[0]) && Shiboken::Converter<QEvent* >::isConvertible(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QEvent* cpp_arg1 = Shiboken::Converter<QEvent* >::toCpp(pyargs[1]);
        // sceneEventFilter(QGraphicsItem*,QEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsItem::sceneEventFilter(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQGraphicsItemFunc_sceneEventFilter_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQGraphicsItemFunc_sceneEventFilter_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem, PySide.QtCore.QEvent", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.sceneEventFilter", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_scenePos(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // scenePos()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPointF cpp_result = cppSelf->QGraphicsItem::scenePos();
    py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_sceneTransform(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // sceneTransform()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QTransform cpp_result = cppSelf->QGraphicsItem::sceneTransform();
    py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_scroll(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "scroll", 2, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        if (numArgs == 2) {
            // scroll(qreal,qreal,QRectF)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::scroll(cpp_arg0, cpp_arg1);
        } else if (Shiboken::Converter<QRectF& >::isConvertible(pyargs[2])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                return 0;
            std::auto_ptr<QRectF > cpp_arg2_auto_ptr;
            QRectF* cpp_arg2 = Shiboken::Converter<QRectF* >::toCpp(pyargs[2]);
            if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[2]))
                cpp_arg2_auto_ptr = std::auto_ptr<QRectF >(cpp_arg2);
            // scroll(qreal,qreal,QRectF)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::scroll(cpp_arg0, cpp_arg1, *cpp_arg2);
        } else goto SbkQGraphicsItemFunc_scroll_TypeError;
    } else goto SbkQGraphicsItemFunc_scroll_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_scroll_TypeError:
        const char* overloads[] = {"float, float, PySide.QtCore.QRectF = QRectF()", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.scroll", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setAcceptDrops(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setAcceptDrops(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setAcceptDrops(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setAcceptDrops_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setAcceptDrops_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setAcceptDrops", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setAcceptHoverEvents(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setAcceptHoverEvents(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setAcceptHoverEvents(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setAcceptHoverEvents_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setAcceptHoverEvents_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setAcceptHoverEvents", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setAcceptTouchEvents(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setAcceptTouchEvents(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setAcceptTouchEvents(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setAcceptTouchEvents_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setAcceptTouchEvents_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setAcceptTouchEvents", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setAcceptedMouseButtons(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (SbkPySide_QtCore_Qt_MouseButton_Check(arg)) {
        QFlags<Qt::MouseButton> cpp_arg0 = Shiboken::Converter<QFlags<Qt::MouseButton> >::toCpp(arg);
        // setAcceptedMouseButtons(QFlags<Qt::MouseButton>)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setAcceptedMouseButtons(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setAcceptedMouseButtons_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setAcceptedMouseButtons_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.MouseButtons", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setAcceptedMouseButtons", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setAcceptsHoverEvents(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setAcceptsHoverEvents(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setAcceptsHoverEvents(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setAcceptsHoverEvents_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setAcceptsHoverEvents_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setAcceptsHoverEvents", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setActive(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setActive(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setActive(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setActive_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setActive_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setActive", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setBoundingRegionGranularity(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setBoundingRegionGranularity(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setBoundingRegionGranularity(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setBoundingRegionGranularity_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setBoundingRegionGranularity_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setBoundingRegionGranularity", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setCacheMode(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setCacheMode", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem::CacheMode >::isConvertible(pyargs[0])) {
        QGraphicsItem::CacheMode cpp_arg0 = Shiboken::Converter<QGraphicsItem::CacheMode >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setCacheMode(QGraphicsItem::CacheMode,QSize)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setCacheMode(cpp_arg0);
        } else if (Shiboken::Converter<QSize& >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            QSize* cpp_arg1 = Shiboken::Converter<QSize* >::toCpp(pyargs[1]);
            // setCacheMode(QGraphicsItem::CacheMode,QSize)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setCacheMode(cpp_arg0, *cpp_arg1);
        } else goto SbkQGraphicsItemFunc_setCacheMode_TypeError;
    } else goto SbkQGraphicsItemFunc_setCacheMode_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setCacheMode_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.CacheMode, PySide.QtCore.QSize = QSize()", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setCacheMode", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setCursor(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QCursor& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QCursor > cpp_arg0_auto_ptr;
        QCursor* cpp_arg0 = Shiboken::Converter<QCursor* >::toCpp(arg);
        if (!Shiboken::Converter<QCursor& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QCursor >(cpp_arg0);
        // setCursor(QCursor)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setCursor(*cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setCursor_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setCursor_TypeError:
        const char* overloads[] = {"PySide.QtGui.QCursor", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setCursor", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setData(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setData", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && Shiboken::Converter<QVariant& >::isConvertible(pyargs[1])) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
        QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QVariant& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
        // setData(int,QVariant)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setData(cpp_arg0, *cpp_arg1);
    } else goto SbkQGraphicsItemFunc_setData_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setData_TypeError:
        const char* overloads[] = {"int, PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setData", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setEnabled(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setEnabled(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setEnabled(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setEnabled_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setEnabled_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setEnabled", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setFiltersChildEvents(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setFiltersChildEvents(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setFiltersChildEvents(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setFiltersChildEvents_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setFiltersChildEvents_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setFiltersChildEvents", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setFlag(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setFlag", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QGraphicsItem::GraphicsItemFlag >::isConvertible(pyargs[0])) {
        QGraphicsItem::GraphicsItemFlag cpp_arg0 = Shiboken::Converter<QGraphicsItem::GraphicsItemFlag >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setFlag(QGraphicsItem::GraphicsItemFlag,bool)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setFlag(cpp_arg0);
        } else if (PyNumber_Check(pyargs[1])) {
            bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
            // setFlag(QGraphicsItem::GraphicsItemFlag,bool)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setFlag(cpp_arg0, cpp_arg1);
        } else goto SbkQGraphicsItemFunc_setFlag_TypeError;
    } else goto SbkQGraphicsItemFunc_setFlag_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setFlag_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.GraphicsItemFlag, bool = true", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setFlag", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setFlags(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Check(arg)) {
        QFlags<QGraphicsItem::GraphicsItemFlag> cpp_arg0 = Shiboken::Converter<QFlags<QGraphicsItem::GraphicsItemFlag> >::toCpp(arg);
        // setFlags(QFlags<QGraphicsItem::GraphicsItemFlag>)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setFlags(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setFlags_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setFlags_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.GraphicsItemFlags", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setFlags", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setFocus(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // setFocus(Qt::FocusReason)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setFocus();
    } else if (Shiboken::Converter<Qt::FocusReason >::isConvertible(arg)) {
        Qt::FocusReason cpp_arg0 = Shiboken::Converter<Qt::FocusReason >::toCpp(arg);
        // setFocus(Qt::FocusReason)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setFocus(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setFocus_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setFocus_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.FocusReason = Qt.OtherFocusReason", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setFocus", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setFocusProxy(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // setFocusProxy(QGraphicsItem*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setFocusProxy(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setFocusProxy_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setFocusProxy_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setFocusProxy", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setGraphicsEffect(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsEffect* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsEffect* cpp_arg0 = Shiboken::Converter<QGraphicsEffect* >::toCpp(arg);
        // setGraphicsEffect(QGraphicsEffect*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setGraphicsEffect(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setGraphicsEffect_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setGraphicsEffect_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsEffect", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setGraphicsEffect", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setGroup(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItemGroup* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItemGroup* cpp_arg0 = Shiboken::Converter<QGraphicsItemGroup* >::toCpp(arg);
        // setGroup(QGraphicsItemGroup*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setGroup(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setGroup_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setGroup_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItemGroup", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setGroup", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setHandlesChildEvents(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setHandlesChildEvents(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setHandlesChildEvents(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setHandlesChildEvents_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setHandlesChildEvents_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setHandlesChildEvents", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setInputMethodHints(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (SbkPySide_QtCore_Qt_InputMethodHint_Check(arg)) {
        QFlags<Qt::InputMethodHint> cpp_arg0 = Shiboken::Converter<QFlags<Qt::InputMethodHint> >::toCpp(arg);
        // setInputMethodHints(QFlags<Qt::InputMethodHint>)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setInputMethodHints(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setInputMethodHints_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setInputMethodHints_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.InputMethodHints", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setInputMethodHints", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setOpacity(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setOpacity(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setOpacity(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setOpacity_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setOpacity_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setOpacity", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setPanelModality(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem::PanelModality >::isConvertible(arg)) {
        QGraphicsItem::PanelModality cpp_arg0 = Shiboken::Converter<QGraphicsItem::PanelModality >::toCpp(arg);
        // setPanelModality(QGraphicsItem::PanelModality)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setPanelModality(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setPanelModality_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setPanelModality_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.PanelModality", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setPanelModality", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setParentItem(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // setParentItem(QGraphicsItem*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setParentItem(cpp_arg0);
        Shiboken::setParent(arg, self);
    } else goto SbkQGraphicsItemFunc_setParentItem_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setParentItem_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setParentItem", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setPos(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setPos", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setPos(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setPos(cpp_arg0, cpp_arg1);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // setPos(QPointF)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setPos(*cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setPos_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setPos_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setPos", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setRotation(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setRotation(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setRotation(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setRotation_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setRotation_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setRotation", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setScale(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setScale(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setScale(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setScale_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setScale_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setScale", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setSelected(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setSelected(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setSelected(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setSelected_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setSelected_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setSelected", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setToolTip(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // setToolTip(QString)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setToolTip(*cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setToolTip_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setToolTip_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setToolTip", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setTransform(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setTransform", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QTransform& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QTransform* cpp_arg0 = Shiboken::Converter<QTransform* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // setTransform(QTransform,bool)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setTransform(*cpp_arg0);
        } else if (PyNumber_Check(pyargs[1])) {
            bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
            // setTransform(QTransform,bool)
            QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
            cppSelf->QGraphicsItem::setTransform(*cpp_arg0, cpp_arg1);
        } else goto SbkQGraphicsItemFunc_setTransform_TypeError;
    } else goto SbkQGraphicsItemFunc_setTransform_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setTransform_TypeError:
        const char* overloads[] = {"PySide.QtGui.QTransform, bool = false", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setTransform", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setTransformOriginPoint(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setTransformOriginPoint", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setTransformOriginPoint(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setTransformOriginPoint(cpp_arg0, cpp_arg1);
    } else if (numArgs == 1 && Shiboken::Converter<QPointF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QPointF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // setTransformOriginPoint(QPointF)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setTransformOriginPoint(*cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setTransformOriginPoint_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setTransformOriginPoint_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.setTransformOriginPoint", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setTransformations(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QList<QGraphicsTransform * >  >::isConvertible(arg)) {
        QList<QGraphicsTransform * >  cpp_arg0 = Shiboken::Converter<QList<QGraphicsTransform * >  >::toCpp(arg);
        // setTransformations(QList<QGraphicsTransform*>)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setTransformations(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setTransformations_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setTransformations_TypeError:
        const char* overloads[] = {"list", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setTransformations", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setVisible(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        bool cpp_arg0 = Shiboken::Converter<bool >::toCpp(arg);
        // setVisible(bool)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setVisible(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setVisible_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setVisible_TypeError:
        const char* overloads[] = {"bool", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setVisible", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setX(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setX(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setX(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setX_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setX_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setX", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setY(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setY(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setY(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setY_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setY_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setY", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_setZValue(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(arg);
        // setZValue(qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::setZValue(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_setZValue_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_setZValue_TypeError:
        const char* overloads[] = {"float", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.setZValue", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_shape(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // shape()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPainterPath cpp_result = cppSelf->QGraphicsItem::shape();
    py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_shear(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "shear", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // shear(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::shear(cpp_arg0, cpp_arg1);
    } else goto SbkQGraphicsItemFunc_shear_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_shear_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.shear", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_show(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // show()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::show();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_stackBefore(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsItem* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // stackBefore(const QGraphicsItem*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::stackBefore(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_stackBefore_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_stackBefore_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.stackBefore", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_toGraphicsObject(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toGraphicsObject()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsObject * cpp_result = cppSelf->QGraphicsItem::toGraphicsObject();
    py_result = Shiboken::Converter<QGraphicsObject* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_toolTip(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toolTip()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QGraphicsItem::toolTip();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_topLevelItem(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // topLevelItem()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsItem * cpp_result = cppSelf->QGraphicsItem::topLevelItem();
    py_result = Shiboken::Converter<QGraphicsItem* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_topLevelWidget(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // topLevelWidget()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsWidget * cpp_result = cppSelf->QGraphicsItem::topLevelWidget();
    py_result = Shiboken::Converter<QGraphicsWidget* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_transform(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // transform()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QTransform cpp_result = cppSelf->QGraphicsItem::transform();
    py_result = Shiboken::Converter<QTransform >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_transformOriginPoint(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // transformOriginPoint()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QPointF cpp_result = cppSelf->QGraphicsItem::transformOriginPoint();
    py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_transformations(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // transformations()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QList<QGraphicsTransform * > cpp_result = cppSelf->QGraphicsItem::transformations();
    py_result = Shiboken::Converter<QList<QGraphicsTransform * > >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_translate(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "translate", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // translate(qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::translate(cpp_arg0, cpp_arg1);
    } else goto SbkQGraphicsItemFunc_translate_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_translate_TypeError:
        const char* overloads[] = {"float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.translate", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_type(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // type()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QGraphicsItem::type();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_ungrabKeyboard(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // ungrabKeyboard()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::ungrabKeyboard();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_ungrabMouse(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // ungrabMouse()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::ungrabMouse();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_unsetCursor(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // unsetCursor()
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    cppSelf->QGraphicsItem::unsetCursor();

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject*
SbkQGraphicsItemFunc_update(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    // invalid argument lengths
    if (numArgs == 2 || numArgs == 3)
        goto SbkQGraphicsItemFunc_update_TypeError;

    if (!PyArg_UnpackTuple(args, "update", 0, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 0) {
        // update(QRectF)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::update();
    } else if (numArgs == 4 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        qreal cpp_arg2 = Shiboken::Converter<qreal >::toCpp(pyargs[2]);
        qreal cpp_arg3 = Shiboken::Converter<qreal >::toCpp(pyargs[3]);
        // update(qreal,qreal,qreal,qreal)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::update(cpp_arg0, cpp_arg1, cpp_arg2, cpp_arg3);
    } else if (Shiboken::Converter<QRectF& >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QRectF > cpp_arg0_auto_ptr;
        QRectF* cpp_arg0 = Shiboken::Converter<QRectF* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QRectF& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QRectF >(cpp_arg0);
        // update(QRectF)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::update(*cpp_arg0);
    } else goto SbkQGraphicsItemFunc_update_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_update_TypeError:
        const char* overloads[] = {"PySide.QtCore.QRectF = QRectF()", "float, float, float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsItem.update", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_wheelEvent(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsSceneWheelEvent* >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsSceneWheelEvent* cpp_arg0 = Shiboken::Converter<QGraphicsSceneWheelEvent* >::toCpp(arg);
        // wheelEvent(QGraphicsSceneWheelEvent*)
        QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsItem::wheelEvent(cpp_arg0);
    } else goto SbkQGraphicsItemFunc_wheelEvent_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsItemFunc_wheelEvent_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsSceneWheelEvent", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsItem.wheelEvent", overloads);
        return 0;
}

static PyObject*
SbkQGraphicsItemFunc_window(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // window()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    QGraphicsWidget * cpp_result = cppSelf->QGraphicsItem::window();
    py_result = Shiboken::Converter<QGraphicsWidget* >::toPython(cpp_result);
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_x(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // x()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::x();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_y(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // y()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::y();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQGraphicsItemFunc_zValue(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // zValue()const
    QGraphicsItem* cppSelf = Shiboken::Converter<QGraphicsItem* >::toCpp((PyObject*)self);
    qreal cpp_result = cppSelf->QGraphicsItem::zValue();
    py_result = Shiboken::Converter<double >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyMethodDef SbkQGraphicsItem_methods[] = {
    {"acceptDrops", (PyCFunction)SbkQGraphicsItemFunc_acceptDrops, METH_NOARGS},
    {"acceptHoverEvents", (PyCFunction)SbkQGraphicsItemFunc_acceptHoverEvents, METH_NOARGS},
    {"acceptTouchEvents", (PyCFunction)SbkQGraphicsItemFunc_acceptTouchEvents, METH_NOARGS},
    {"acceptedMouseButtons", (PyCFunction)SbkQGraphicsItemFunc_acceptedMouseButtons, METH_NOARGS},
    {"acceptsHoverEvents", (PyCFunction)SbkQGraphicsItemFunc_acceptsHoverEvents, METH_NOARGS},
    {"addToIndex", (PyCFunction)SbkQGraphicsItemFunc_addToIndex, METH_NOARGS},
    {"advance", (PyCFunction)SbkQGraphicsItemFunc_advance, METH_O},
    {"boundingRect", (PyCFunction)SbkQGraphicsItemFunc_boundingRect, METH_NOARGS},
    {"boundingRegion", (PyCFunction)SbkQGraphicsItemFunc_boundingRegion, METH_O},
    {"boundingRegionGranularity", (PyCFunction)SbkQGraphicsItemFunc_boundingRegionGranularity, METH_NOARGS},
    {"cacheMode", (PyCFunction)SbkQGraphicsItemFunc_cacheMode, METH_NOARGS},
    {"childItems", (PyCFunction)SbkQGraphicsItemFunc_childItems, METH_NOARGS},
    {"childrenBoundingRect", (PyCFunction)SbkQGraphicsItemFunc_childrenBoundingRect, METH_NOARGS},
    {"clearFocus", (PyCFunction)SbkQGraphicsItemFunc_clearFocus, METH_NOARGS},
    {"clipPath", (PyCFunction)SbkQGraphicsItemFunc_clipPath, METH_NOARGS},
    {"collidesWithItem", (PyCFunction)SbkQGraphicsItemFunc_collidesWithItem, METH_VARARGS},
    {"collidesWithPath", (PyCFunction)SbkQGraphicsItemFunc_collidesWithPath, METH_VARARGS},
    {"collidingItems", (PyCFunction)SbkQGraphicsItemFunc_collidingItems, METH_NOARGS|METH_O},
    {"commonAncestorItem", (PyCFunction)SbkQGraphicsItemFunc_commonAncestorItem, METH_O},
    {"contains", (PyCFunction)SbkQGraphicsItemFunc_contains, METH_O},
    {"contextMenuEvent", (PyCFunction)SbkQGraphicsItemFunc_contextMenuEvent, METH_O},
    {"cursor", (PyCFunction)SbkQGraphicsItemFunc_cursor, METH_NOARGS},
    {"data", (PyCFunction)SbkQGraphicsItemFunc_data, METH_O},
    {"deviceTransform", (PyCFunction)SbkQGraphicsItemFunc_deviceTransform, METH_O},
    {"dragEnterEvent", (PyCFunction)SbkQGraphicsItemFunc_dragEnterEvent, METH_O},
    {"dragLeaveEvent", (PyCFunction)SbkQGraphicsItemFunc_dragLeaveEvent, METH_O},
    {"dragMoveEvent", (PyCFunction)SbkQGraphicsItemFunc_dragMoveEvent, METH_O},
    {"dropEvent", (PyCFunction)SbkQGraphicsItemFunc_dropEvent, METH_O},
    {"effectiveOpacity", (PyCFunction)SbkQGraphicsItemFunc_effectiveOpacity, METH_NOARGS},
    {"ensureVisible", (PyCFunction)SbkQGraphicsItemFunc_ensureVisible, METH_VARARGS},
    {"extension", (PyCFunction)SbkQGraphicsItemFunc_extension, METH_O},
    {"filtersChildEvents", (PyCFunction)SbkQGraphicsItemFunc_filtersChildEvents, METH_NOARGS},
    {"flags", (PyCFunction)SbkQGraphicsItemFunc_flags, METH_NOARGS},
    {"focusInEvent", (PyCFunction)SbkQGraphicsItemFunc_focusInEvent, METH_O},
    {"focusItem", (PyCFunction)SbkQGraphicsItemFunc_focusItem, METH_NOARGS},
    {"focusOutEvent", (PyCFunction)SbkQGraphicsItemFunc_focusOutEvent, METH_O},
    {"focusProxy", (PyCFunction)SbkQGraphicsItemFunc_focusProxy, METH_NOARGS},
    {"focusScopeItem", (PyCFunction)SbkQGraphicsItemFunc_focusScopeItem, METH_NOARGS},
    {"grabKeyboard", (PyCFunction)SbkQGraphicsItemFunc_grabKeyboard, METH_NOARGS},
    {"grabMouse", (PyCFunction)SbkQGraphicsItemFunc_grabMouse, METH_NOARGS},
    {"graphicsEffect", (PyCFunction)SbkQGraphicsItemFunc_graphicsEffect, METH_NOARGS},
    {"group", (PyCFunction)SbkQGraphicsItemFunc_group, METH_NOARGS},
    {"handlesChildEvents", (PyCFunction)SbkQGraphicsItemFunc_handlesChildEvents, METH_NOARGS},
    {"hasCursor", (PyCFunction)SbkQGraphicsItemFunc_hasCursor, METH_NOARGS},
    {"hasFocus", (PyCFunction)SbkQGraphicsItemFunc_hasFocus, METH_NOARGS},
    {"hide", (PyCFunction)SbkQGraphicsItemFunc_hide, METH_NOARGS},
    {"hoverEnterEvent", (PyCFunction)SbkQGraphicsItemFunc_hoverEnterEvent, METH_O},
    {"hoverLeaveEvent", (PyCFunction)SbkQGraphicsItemFunc_hoverLeaveEvent, METH_O},
    {"hoverMoveEvent", (PyCFunction)SbkQGraphicsItemFunc_hoverMoveEvent, METH_O},
    {"inputMethodEvent", (PyCFunction)SbkQGraphicsItemFunc_inputMethodEvent, METH_O},
    {"inputMethodHints", (PyCFunction)SbkQGraphicsItemFunc_inputMethodHints, METH_NOARGS},
    {"inputMethodQuery", (PyCFunction)SbkQGraphicsItemFunc_inputMethodQuery, METH_O},
    {"installSceneEventFilter", (PyCFunction)SbkQGraphicsItemFunc_installSceneEventFilter, METH_O},
    {"isActive", (PyCFunction)SbkQGraphicsItemFunc_isActive, METH_NOARGS},
    {"isAncestorOf", (PyCFunction)SbkQGraphicsItemFunc_isAncestorOf, METH_O},
    {"isBlockedByModalPanel", (PyCFunction)SbkQGraphicsItemFunc_isBlockedByModalPanel, METH_NOARGS},
    {"isClipped", (PyCFunction)SbkQGraphicsItemFunc_isClipped, METH_NOARGS},
    {"isEnabled", (PyCFunction)SbkQGraphicsItemFunc_isEnabled, METH_NOARGS},
    {"isObscured", (PyCFunction)SbkQGraphicsItemFunc_isObscured, METH_VARARGS},
    {"isObscuredBy", (PyCFunction)SbkQGraphicsItemFunc_isObscuredBy, METH_O},
    {"isPanel", (PyCFunction)SbkQGraphicsItemFunc_isPanel, METH_NOARGS},
    {"isSelected", (PyCFunction)SbkQGraphicsItemFunc_isSelected, METH_NOARGS},
    {"isUnderMouse", (PyCFunction)SbkQGraphicsItemFunc_isUnderMouse, METH_NOARGS},
    {"isVisible", (PyCFunction)SbkQGraphicsItemFunc_isVisible, METH_NOARGS},
    {"isVisibleTo", (PyCFunction)SbkQGraphicsItemFunc_isVisibleTo, METH_O},
    {"isWidget", (PyCFunction)SbkQGraphicsItemFunc_isWidget, METH_NOARGS},
    {"isWindow", (PyCFunction)SbkQGraphicsItemFunc_isWindow, METH_NOARGS},
    {"itemChange", (PyCFunction)SbkQGraphicsItemFunc_itemChange, METH_VARARGS},
    {"itemTransform", (PyCFunction)SbkQGraphicsItemFunc_itemTransform, METH_VARARGS},
    {"keyPressEvent", (PyCFunction)SbkQGraphicsItemFunc_keyPressEvent, METH_O},
    {"keyReleaseEvent", (PyCFunction)SbkQGraphicsItemFunc_keyReleaseEvent, METH_O},
    {"mapFromItem", (PyCFunction)SbkQGraphicsItemFunc_mapFromItem, METH_VARARGS},
    {"mapFromParent", (PyCFunction)SbkQGraphicsItemFunc_mapFromParent, METH_VARARGS},
    {"mapFromScene", (PyCFunction)SbkQGraphicsItemFunc_mapFromScene, METH_VARARGS},
    {"mapRectFromItem", (PyCFunction)SbkQGraphicsItemFunc_mapRectFromItem, METH_VARARGS},
    {"mapRectFromParent", (PyCFunction)SbkQGraphicsItemFunc_mapRectFromParent, METH_VARARGS},
    {"mapRectFromScene", (PyCFunction)SbkQGraphicsItemFunc_mapRectFromScene, METH_VARARGS},
    {"mapRectToItem", (PyCFunction)SbkQGraphicsItemFunc_mapRectToItem, METH_VARARGS},
    {"mapRectToParent", (PyCFunction)SbkQGraphicsItemFunc_mapRectToParent, METH_VARARGS},
    {"mapRectToScene", (PyCFunction)SbkQGraphicsItemFunc_mapRectToScene, METH_VARARGS},
    {"mapToItem", (PyCFunction)SbkQGraphicsItemFunc_mapToItem, METH_VARARGS},
    {"mapToParent", (PyCFunction)SbkQGraphicsItemFunc_mapToParent, METH_VARARGS},
    {"mapToScene", (PyCFunction)SbkQGraphicsItemFunc_mapToScene, METH_VARARGS},
    {"mouseDoubleClickEvent", (PyCFunction)SbkQGraphicsItemFunc_mouseDoubleClickEvent, METH_O},
    {"mouseMoveEvent", (PyCFunction)SbkQGraphicsItemFunc_mouseMoveEvent, METH_O},
    {"mousePressEvent", (PyCFunction)SbkQGraphicsItemFunc_mousePressEvent, METH_O},
    {"mouseReleaseEvent", (PyCFunction)SbkQGraphicsItemFunc_mouseReleaseEvent, METH_O},
    {"moveBy", (PyCFunction)SbkQGraphicsItemFunc_moveBy, METH_VARARGS},
    {"opacity", (PyCFunction)SbkQGraphicsItemFunc_opacity, METH_NOARGS},
    {"opaqueArea", (PyCFunction)SbkQGraphicsItemFunc_opaqueArea, METH_NOARGS},
    {"paint", (PyCFunction)SbkQGraphicsItemFunc_paint, METH_VARARGS},
    {"panel", (PyCFunction)SbkQGraphicsItemFunc_panel, METH_NOARGS},
    {"panelModality", (PyCFunction)SbkQGraphicsItemFunc_panelModality, METH_NOARGS},
    {"parentItem", (PyCFunction)SbkQGraphicsItemFunc_parentItem, METH_NOARGS},
    {"parentObject", (PyCFunction)SbkQGraphicsItemFunc_parentObject, METH_NOARGS},
    {"parentWidget", (PyCFunction)SbkQGraphicsItemFunc_parentWidget, METH_NOARGS},
    {"pos", (PyCFunction)SbkQGraphicsItemFunc_pos, METH_NOARGS},
    {"prepareGeometryChange", (PyCFunction)SbkQGraphicsItemFunc_prepareGeometryChange, METH_NOARGS},
    {"removeFromIndex", (PyCFunction)SbkQGraphicsItemFunc_removeFromIndex, METH_NOARGS},
    {"removeSceneEventFilter", (PyCFunction)SbkQGraphicsItemFunc_removeSceneEventFilter, METH_O},
    {"resetTransform", (PyCFunction)SbkQGraphicsItemFunc_resetTransform, METH_NOARGS},
    {"rotate", (PyCFunction)SbkQGraphicsItemFunc_rotate, METH_O},
    {"rotation", (PyCFunction)SbkQGraphicsItemFunc_rotation, METH_NOARGS},
    {"scale", (PyCFunction)SbkQGraphicsItemFunc_scale, METH_VARARGS},
    {"scene", (PyCFunction)SbkQGraphicsItemFunc_scene, METH_NOARGS},
    {"sceneBoundingRect", (PyCFunction)SbkQGraphicsItemFunc_sceneBoundingRect, METH_NOARGS},
    {"sceneEvent", (PyCFunction)SbkQGraphicsItemFunc_sceneEvent, METH_O},
    {"sceneEventFilter", (PyCFunction)SbkQGraphicsItemFunc_sceneEventFilter, METH_VARARGS},
    {"scenePos", (PyCFunction)SbkQGraphicsItemFunc_scenePos, METH_NOARGS},
    {"sceneTransform", (PyCFunction)SbkQGraphicsItemFunc_sceneTransform, METH_NOARGS},
    {"scroll", (PyCFunction)SbkQGraphicsItemFunc_scroll, METH_VARARGS},
    {"setAcceptDrops", (PyCFunction)SbkQGraphicsItemFunc_setAcceptDrops, METH_O},
    {"setAcceptHoverEvents", (PyCFunction)SbkQGraphicsItemFunc_setAcceptHoverEvents, METH_O},
    {"setAcceptTouchEvents", (PyCFunction)SbkQGraphicsItemFunc_setAcceptTouchEvents, METH_O},
    {"setAcceptedMouseButtons", (PyCFunction)SbkQGraphicsItemFunc_setAcceptedMouseButtons, METH_O},
    {"setAcceptsHoverEvents", (PyCFunction)SbkQGraphicsItemFunc_setAcceptsHoverEvents, METH_O},
    {"setActive", (PyCFunction)SbkQGraphicsItemFunc_setActive, METH_O},
    {"setBoundingRegionGranularity", (PyCFunction)SbkQGraphicsItemFunc_setBoundingRegionGranularity, METH_O},
    {"setCacheMode", (PyCFunction)SbkQGraphicsItemFunc_setCacheMode, METH_VARARGS},
    {"setCursor", (PyCFunction)SbkQGraphicsItemFunc_setCursor, METH_O},
    {"setData", (PyCFunction)SbkQGraphicsItemFunc_setData, METH_VARARGS},
    {"setEnabled", (PyCFunction)SbkQGraphicsItemFunc_setEnabled, METH_O},
    {"setFiltersChildEvents", (PyCFunction)SbkQGraphicsItemFunc_setFiltersChildEvents, METH_O},
    {"setFlag", (PyCFunction)SbkQGraphicsItemFunc_setFlag, METH_VARARGS},
    {"setFlags", (PyCFunction)SbkQGraphicsItemFunc_setFlags, METH_O},
    {"setFocus", (PyCFunction)SbkQGraphicsItemFunc_setFocus, METH_NOARGS|METH_O},
    {"setFocusProxy", (PyCFunction)SbkQGraphicsItemFunc_setFocusProxy, METH_O},
    {"setGraphicsEffect", (PyCFunction)SbkQGraphicsItemFunc_setGraphicsEffect, METH_O},
    {"setGroup", (PyCFunction)SbkQGraphicsItemFunc_setGroup, METH_O},
    {"setHandlesChildEvents", (PyCFunction)SbkQGraphicsItemFunc_setHandlesChildEvents, METH_O},
    {"setInputMethodHints", (PyCFunction)SbkQGraphicsItemFunc_setInputMethodHints, METH_O},
    {"setOpacity", (PyCFunction)SbkQGraphicsItemFunc_setOpacity, METH_O},
    {"setPanelModality", (PyCFunction)SbkQGraphicsItemFunc_setPanelModality, METH_O},
    {"setParentItem", (PyCFunction)SbkQGraphicsItemFunc_setParentItem, METH_O},
    {"setPos", (PyCFunction)SbkQGraphicsItemFunc_setPos, METH_VARARGS},
    {"setRotation", (PyCFunction)SbkQGraphicsItemFunc_setRotation, METH_O},
    {"setScale", (PyCFunction)SbkQGraphicsItemFunc_setScale, METH_O},
    {"setSelected", (PyCFunction)SbkQGraphicsItemFunc_setSelected, METH_O},
    {"setToolTip", (PyCFunction)SbkQGraphicsItemFunc_setToolTip, METH_O},
    {"setTransform", (PyCFunction)SbkQGraphicsItemFunc_setTransform, METH_VARARGS},
    {"setTransformOriginPoint", (PyCFunction)SbkQGraphicsItemFunc_setTransformOriginPoint, METH_VARARGS},
    {"setTransformations", (PyCFunction)SbkQGraphicsItemFunc_setTransformations, METH_O},
    {"setVisible", (PyCFunction)SbkQGraphicsItemFunc_setVisible, METH_O},
    {"setX", (PyCFunction)SbkQGraphicsItemFunc_setX, METH_O},
    {"setY", (PyCFunction)SbkQGraphicsItemFunc_setY, METH_O},
    {"setZValue", (PyCFunction)SbkQGraphicsItemFunc_setZValue, METH_O},
    {"shape", (PyCFunction)SbkQGraphicsItemFunc_shape, METH_NOARGS},
    {"shear", (PyCFunction)SbkQGraphicsItemFunc_shear, METH_VARARGS},
    {"show", (PyCFunction)SbkQGraphicsItemFunc_show, METH_NOARGS},
    {"stackBefore", (PyCFunction)SbkQGraphicsItemFunc_stackBefore, METH_O},
    {"toGraphicsObject", (PyCFunction)SbkQGraphicsItemFunc_toGraphicsObject, METH_NOARGS},
    {"toolTip", (PyCFunction)SbkQGraphicsItemFunc_toolTip, METH_NOARGS},
    {"topLevelItem", (PyCFunction)SbkQGraphicsItemFunc_topLevelItem, METH_NOARGS},
    {"topLevelWidget", (PyCFunction)SbkQGraphicsItemFunc_topLevelWidget, METH_NOARGS},
    {"transform", (PyCFunction)SbkQGraphicsItemFunc_transform, METH_NOARGS},
    {"transformOriginPoint", (PyCFunction)SbkQGraphicsItemFunc_transformOriginPoint, METH_NOARGS},
    {"transformations", (PyCFunction)SbkQGraphicsItemFunc_transformations, METH_NOARGS},
    {"translate", (PyCFunction)SbkQGraphicsItemFunc_translate, METH_VARARGS},
    {"type", (PyCFunction)SbkQGraphicsItemFunc_type, METH_NOARGS},
    {"ungrabKeyboard", (PyCFunction)SbkQGraphicsItemFunc_ungrabKeyboard, METH_NOARGS},
    {"ungrabMouse", (PyCFunction)SbkQGraphicsItemFunc_ungrabMouse, METH_NOARGS},
    {"unsetCursor", (PyCFunction)SbkQGraphicsItemFunc_unsetCursor, METH_NOARGS},
    {"update", (PyCFunction)SbkQGraphicsItemFunc_update, METH_VARARGS},
    {"wheelEvent", (PyCFunction)SbkQGraphicsItemFunc_wheelEvent, METH_O},
    {"window", (PyCFunction)SbkQGraphicsItemFunc_window, METH_NOARGS},
    {"x", (PyCFunction)SbkQGraphicsItemFunc_x, METH_NOARGS},
    {"y", (PyCFunction)SbkQGraphicsItemFunc_y, METH_NOARGS},
    {"zValue", (PyCFunction)SbkQGraphicsItemFunc_zValue, METH_NOARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQGraphicsItem_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QGraphicsItem",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::SbkBaseWrapper_Dealloc< QGraphicsItemWrapper >,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQGraphicsItem_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQGraphicsItem_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0
};

static SbkBaseWrapperType* SbkQGraphicsItem_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQGraphicsItem_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QGraphicsItem*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

static PyGetSetDef SbkPySide_QtGui_QGraphicsItem_CacheMode_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_CacheMode_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_CacheMode_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "CacheMode",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsItem_CacheMode_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_CacheMode_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_CacheMode_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtGui_QGraphicsItem_PanelModality_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_PanelModality_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_PanelModality_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PanelModality",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsItem_PanelModality_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_PanelModality_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_PanelModality_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___and__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toPython(
        ((QGraphicsItem::GraphicsItemFlags) ((SbkEnumObject*)self)->ob_ival)
        & Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___or__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toPython(
        ((QGraphicsItem::GraphicsItemFlags) ((SbkEnumObject*)self)->ob_ival)
        | Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___xor__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toPython(
        ((QGraphicsItem::GraphicsItemFlags) ((SbkEnumObject*)self)->ob_ival)
        ^ Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___invert__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toPython(
        ~ Shiboken::Converter< QGraphicsItem::GraphicsItemFlags >::toCpp(self)
    );
}


static PyNumberMethods SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_as_number = {
    /*nb_add*/                  0,
    /*nb_subtract*/             0,
    /*nb_multiply*/             0,
    /*nb_divide*/               0,
    /*nb_remainder*/            0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             0,
    /*nb_positive*/             0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___invert__,
    /*nb_lshift*/               0,
    /*nb_rshift*/               0,
    /*nb_and*/                  (binaryfunc)SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___and__,
    /*nb_xor*/                  (binaryfunc)SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___xor__,
    /*nb_or*/                   (binaryfunc)SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag___or__,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          0,
    /*nb_inplace_subtract*/     0,
    /*nb_inplace_multiply*/     0,
    /*nb_inplace_divide*/       0,
    /*nb_inplace_remainder*/    0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       0,
    /*nb_inplace_rshift*/       0,
    /*nb_inplace_and*/          0,
    /*nb_inplace_xor*/          0,
    /*nb_inplace_or*/           0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};


static PyGetSetDef SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "GraphicsItemFlag",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        &SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "GraphicsItemFlags",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             &SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    if (!PyType_IsSubtype(type, &SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type))
        return 0;

    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


static PyGetSetDef SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "GraphicsItemChange",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtGui_QGraphicsItem_Extension_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsItem_Extension_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsItem_Extension_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Extension",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsItem_Extension_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsItem_Extension_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtGui_QGraphicsItem_Extension_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void)
init_QGraphicsItem(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQGraphicsItem_Type);


    // Fill type discovery information
    SbkQGraphicsItem_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQGraphicsItem_Type.type_discovery->addTypeDiscoveryFunction(&SbkQGraphicsItem_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQGraphicsItem_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQGraphicsItem_Type));
    PyModule_AddObject(module, "QGraphicsItem",
        ((PyObject*)&SbkQGraphicsItem_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: CacheMode
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_CACHEMODE_IDX] = &SbkPySide_QtGui_QGraphicsItem_CacheMode_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "CacheMode",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type,
        (long) QGraphicsItem::NoCache, "NoCache");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "NoCache", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_CacheMode_Type.tp_dict,
        "NoCache", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type,
        (long) QGraphicsItem::ItemCoordinateCache, "ItemCoordinateCache");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemCoordinateCache", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_CacheMode_Type.tp_dict,
        "ItemCoordinateCache", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_CacheMode_Type,
        (long) QGraphicsItem::DeviceCoordinateCache, "DeviceCoordinateCache");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "DeviceCoordinateCache", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_CacheMode_Type.tp_dict,
        "DeviceCoordinateCache", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsItem::CacheMode");

    // init enum class: PanelModality
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_PANELMODALITY_IDX] = &SbkPySide_QtGui_QGraphicsItem_PanelModality_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "PanelModality",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type,
        (long) QGraphicsItem::NonModal, "NonModal");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "NonModal", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_PanelModality_Type.tp_dict,
        "NonModal", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type,
        (long) QGraphicsItem::PanelModal, "PanelModal");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "PanelModal", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_PanelModality_Type.tp_dict,
        "PanelModal", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_PanelModality_Type,
        (long) QGraphicsItem::SceneModal, "SceneModal");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "SceneModal", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_PanelModality_Type.tp_dict,
        "SceneModal", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsItem::PanelModality");

    // init enum class: GraphicsItemFlag
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_GRAPHICSITEMFLAG_IDX] = &SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "GraphicsItemFlag",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type));

    // init flags class: QFlags<QGraphicsItem::GraphicsItemFlag>
    SbkPySide_QtGuiTypes[SBK_QFLAGS_QGRAPHICSITEM_GRAPHICSITEMFLAG__IDX] = &SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "GraphicsItemFlags",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlags_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIsMovable, "ItemIsMovable");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIsMovable", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIsMovable", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIsSelectable, "ItemIsSelectable");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIsSelectable", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIsSelectable", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIsFocusable, "ItemIsFocusable");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIsFocusable", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIsFocusable", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemClipsToShape, "ItemClipsToShape");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemClipsToShape", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemClipsToShape", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemClipsChildrenToShape, "ItemClipsChildrenToShape");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemClipsChildrenToShape", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemClipsChildrenToShape", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIgnoresTransformations, "ItemIgnoresTransformations");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIgnoresTransformations", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIgnoresTransformations", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIgnoresParentOpacity, "ItemIgnoresParentOpacity");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIgnoresParentOpacity", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIgnoresParentOpacity", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemDoesntPropagateOpacityToChildren, "ItemDoesntPropagateOpacityToChildren");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemDoesntPropagateOpacityToChildren", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemDoesntPropagateOpacityToChildren", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemStacksBehindParent, "ItemStacksBehindParent");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemStacksBehindParent", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemStacksBehindParent", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemUsesExtendedStyleOption, "ItemUsesExtendedStyleOption");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemUsesExtendedStyleOption", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemUsesExtendedStyleOption", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemHasNoContents, "ItemHasNoContents");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemHasNoContents", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemHasNoContents", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemSendsGeometryChanges, "ItemSendsGeometryChanges");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSendsGeometryChanges", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemSendsGeometryChanges", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemAcceptsInputMethod, "ItemAcceptsInputMethod");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemAcceptsInputMethod", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemAcceptsInputMethod", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemNegativeZStacksBehindParent, "ItemNegativeZStacksBehindParent");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemNegativeZStacksBehindParent", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemNegativeZStacksBehindParent", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIsPanel, "ItemIsPanel");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIsPanel", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIsPanel", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemIsFocusScope, "ItemIsFocusScope");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemIsFocusScope", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemIsFocusScope", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type,
        (long) QGraphicsItem::ItemSendsScenePositionChanges, "ItemSendsScenePositionChanges");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSendsScenePositionChanges", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemFlag_Type.tp_dict,
        "ItemSendsScenePositionChanges", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsItem::GraphicsItemFlag");

    // init enum class: GraphicsItemChange
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_GRAPHICSITEMCHANGE_IDX] = &SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "GraphicsItemChange",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemPositionChange, "ItemPositionChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemPositionChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemPositionChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemMatrixChange, "ItemMatrixChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemMatrixChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemMatrixChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemVisibleChange, "ItemVisibleChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemVisibleChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemVisibleChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemEnabledChange, "ItemEnabledChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemEnabledChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemEnabledChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemSelectedChange, "ItemSelectedChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSelectedChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemSelectedChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemParentChange, "ItemParentChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemParentChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemParentChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemChildAddedChange, "ItemChildAddedChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemChildAddedChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemChildAddedChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemChildRemovedChange, "ItemChildRemovedChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemChildRemovedChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemChildRemovedChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemTransformChange, "ItemTransformChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemTransformChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemTransformChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemPositionHasChanged, "ItemPositionHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemPositionHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemPositionHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemTransformHasChanged, "ItemTransformHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemTransformHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemTransformHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemSceneChange, "ItemSceneChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSceneChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemSceneChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemVisibleHasChanged, "ItemVisibleHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemVisibleHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemVisibleHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemEnabledHasChanged, "ItemEnabledHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemEnabledHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemEnabledHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemSelectedHasChanged, "ItemSelectedHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSelectedHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemSelectedHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemParentHasChanged, "ItemParentHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemParentHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemParentHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemSceneHasChanged, "ItemSceneHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemSceneHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemSceneHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemCursorChange, "ItemCursorChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemCursorChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemCursorChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemCursorHasChanged, "ItemCursorHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemCursorHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemCursorHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemToolTipChange, "ItemToolTipChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemToolTipChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemToolTipChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemToolTipHasChanged, "ItemToolTipHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemToolTipHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemToolTipHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemFlagsChange, "ItemFlagsChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemFlagsChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemFlagsChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemFlagsHaveChanged, "ItemFlagsHaveChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemFlagsHaveChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemFlagsHaveChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemZValueChange, "ItemZValueChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemZValueChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemZValueChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemZValueHasChanged, "ItemZValueHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemZValueHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemZValueHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemOpacityChange, "ItemOpacityChange");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemOpacityChange", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemOpacityChange", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemOpacityHasChanged, "ItemOpacityHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemOpacityHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemOpacityHasChanged", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type,
        (long) QGraphicsItem::ItemScenePositionHasChanged, "ItemScenePositionHasChanged");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "ItemScenePositionHasChanged", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_GraphicsItemChange_Type.tp_dict,
        "ItemScenePositionHasChanged", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsItem::GraphicsItemChange");

    // init enum class: Extension
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_EXTENSION_IDX] = &SbkPySide_QtGui_QGraphicsItem_Extension_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsItem_Extension_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsItem_Extension_Type);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
            "Extension",((PyObject*)&SbkPySide_QtGui_QGraphicsItem_Extension_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsItem_Extension_Type,
        (long) QGraphicsItem::UserExtension, "UserExtension");
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict,
        "UserExtension", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsItem_Extension_Type.tp_dict,
        "UserExtension", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsItem::Extension");


    // Begin code injection
    
    PyObject* userTypeConstant =  PyInt_FromLong(QGraphicsItem::UserType);
    PyDict_SetItemString(SbkQGraphicsItem_Type.super.ht_type.tp_dict, "UserType", userTypeConstant);
    
    // End of code injection
    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsItem >("QGraphicsItem*");
    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsItem >(typeid(QGraphicsItem).name());
}


} // extern "C"

namespace Shiboken
{
// Copy C++ object implementation
// Converter implementations
} // namespace Shiboken


