/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtcore_python.h"

#include "qabstractfileengine_wrapper.h"

// Extra includes
#include <QDateTime>
#include <qabstractfileengine.h>
#include <qdatetime.h>
#include <qstring.h>
#include <qstringlist.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QAbstractFileEngineWrapper::QAbstractFileEngineWrapper() : QAbstractFileEngine() {
    // ... middle
}

bool QAbstractFileEngineWrapper::caseSensitive() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "caseSensitive"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::caseSensitive();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.caseSensitive");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::close()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "close"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::close();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.close");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::copy(const QString & newName)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "copy"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::copy(newName);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QString& >::toPython(newName)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.copy");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QStringList QAbstractFileEngineWrapper::entryList(QFlags<QDir::Filter> filters, const QStringList & filterNames) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "entryList"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::entryList(filters, filterNames);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QFlags<QDir::Filter> >::toPython(filters),
        Shiboken::Converter<QStringList& >::toPython(filterNames)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QStringList();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QStringList >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.entryList");
        return QStringList()        ;
    }
    QStringList cpp_result(Shiboken::Converter<QStringList >::toCpp(py_result));
    return cpp_result;
}

QFlags<QAbstractFileEngine::FileFlag> QAbstractFileEngineWrapper::fileFlags(QFlags<QAbstractFileEngine::FileFlag> type) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "fileFlags"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::fileFlags(type);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::toPython(type)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QFlags<QAbstractFileEngine::FileFlag>();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.fileFlags");
        return QFlags<QAbstractFileEngine::FileFlag>()        ;
    }
    QFlags<QAbstractFileEngine::FileFlag> cpp_result(Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::toCpp(py_result));
    return cpp_result;
}

QString QAbstractFileEngineWrapper::fileName(QAbstractFileEngine::FileName file) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "fileName"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::fileName(file);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QAbstractFileEngine::FileName >::toPython(file)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QString();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QString >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.fileName");
        return QString()        ;
    }
    QString cpp_result(Shiboken::Converter<QString >::toCpp(py_result));
    return cpp_result;
}

QDateTime QAbstractFileEngineWrapper::fileTime(QAbstractFileEngine::FileTime time) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "fileTime"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::fileTime(time);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QAbstractFileEngine::FileTime >::toPython(time)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QDateTime();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QDateTime >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.fileTime");
        return QDateTime()        ;
    }
    QDateTime cpp_result(Shiboken::Converter<QDateTime >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::flush()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "flush"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::flush();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.flush");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

int QAbstractFileEngineWrapper::handle() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "handle"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::handle();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.handle");
        return int(0)        ;
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::isRelativePath() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isRelativePath"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::isRelativePath();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.isRelativePath");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::isSequential() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isSequential"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::isSequential();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.isSequential");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::link(const QString & newName)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "link"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::link(newName);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QString& >::toPython(newName)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.link");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::mkdir(const QString & dirName, bool createParentDirectories) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mkdir"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::mkdir(dirName, createParentDirectories);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QString& >::toPython(dirName),
        Shiboken::Converter<bool >::toPython(createParentDirectories)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.mkdir");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::open(QFlags<QIODevice::OpenModeFlag> openMode)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "open"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::open(openMode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QFlags<QIODevice::OpenModeFlag> >::toPython(openMode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.open");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QString QAbstractFileEngineWrapper::owner(QAbstractFileEngine::FileOwner arg__1) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "owner"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::owner(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QAbstractFileEngine::FileOwner >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QString();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QString >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.owner");
        return QString()        ;
    }
    QString cpp_result(Shiboken::Converter<QString >::toCpp(py_result));
    return cpp_result;
}

uint QAbstractFileEngineWrapper::ownerId(QAbstractFileEngine::FileOwner arg__1) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "ownerId"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::ownerId(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QAbstractFileEngine::FileOwner >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return uint(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.ownerId");
        return uint(0)        ;
    }
    uint cpp_result(Shiboken::Converter<unsigned int >::toCpp(py_result));
    return cpp_result;
}

qint64 QAbstractFileEngineWrapper::pos() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "pos"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::pos();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.pos");
        return qint64(0)        ;
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::remove()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "remove"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::remove();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.remove");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::rename(const QString & newName)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "rename"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::rename(newName);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QString& >::toPython(newName)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.rename");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::rmdir(const QString & dirName, bool recurseParentDirectories) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "rmdir"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::rmdir(dirName, recurseParentDirectories);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(OO)",
        Shiboken::Converter<QString& >::toPython(dirName),
        Shiboken::Converter<bool >::toPython(recurseParentDirectories)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.rmdir");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::seek(qint64 pos)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "seek"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::seek(pos);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(L)",
        Shiboken::Converter<long long >::toPython(pos)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.seek");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

void QAbstractFileEngineWrapper::setFileName(const QString & file)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "setFileName"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::setFileName(file);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QString& >::toPython(file)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QAbstractFileEngineWrapper::setPermissions(uint perms)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "setPermissions"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::setPermissions(perms);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(I)",
        Shiboken::Converter<unsigned int >::toPython(perms)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.setPermissions");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::setSize(qint64 size)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "setSize"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::setSize(size);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(L)",
        Shiboken::Converter<long long >::toPython(size)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.setSize");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QAbstractFileEngineWrapper::size() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "size"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::size();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.size");
        return qint64(0)        ;
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

bool QAbstractFileEngineWrapper::supportsExtension(QAbstractFileEngine::Extension extension) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "supportsExtension"));
    if (py_override.isNull()) {
        return this->QAbstractFileEngine::supportsExtension(extension);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(O)",
        Shiboken::Converter<QAbstractFileEngine::Extension >::toPython(extension)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.supportsExtension");
        return bool(0)        ;
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QAbstractFileEngineWrapper::write(const char * data, qint64 len)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "write"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QAbstractFileEngine::write(data, len);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(zL)",
        data,
        Shiboken::Converter<long long >::toPython(len)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_SetString(PyExc_TypeError, "Invalid return value in function QAbstractFileEngine.write");
        return qint64(0)        ;
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

QAbstractFileEngineWrapper::~QAbstractFileEngineWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQAbstractFileEngine_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QAbstractFileEngineWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);
    assert(!sbkSelf->cptr);
    // QAbstractFileEngine()
    cptr = new QAbstractFileEngineWrapper();

    sbkSelf->cptr = cptr;
    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf);

    return 1;
}

static PyObject*
SbkQAbstractFileEngineFunc_atEnd(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // atEnd()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QAbstractFileEngine::atEnd();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_caseSensitive(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // caseSensitive()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QAbstractFileEngine::caseSensitive();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_close(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    // close()
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    thread_state_saver.save();
    bool cpp_result = cppSelf->QAbstractFileEngine::close();
    thread_state_saver.restore();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_copy(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // copy(QString)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::copy(*cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_copy_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_copy_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.copy", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_create(PyObject* self, PyObject* arg)
{
    PyObject* py_result = 0;

    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // create(QString)
        QAbstractFileEngine * cpp_result = QAbstractFileEngine::create(*cpp_arg0);
        py_result = Shiboken::Converter<QAbstractFileEngine* >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_create_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_create_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.create", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_entryList(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "entryList", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && SbkPySide_QtCore_QDir_Filter_Check(pyargs[0]) && Shiboken::Converter<QStringList& >::isConvertible(pyargs[1])) {
        QFlags<QDir::Filter> cpp_arg0 = Shiboken::Converter<QFlags<QDir::Filter> >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QStringList > cpp_arg1_auto_ptr;
        QStringList* cpp_arg1 = Shiboken::Converter<QStringList* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QStringList& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QStringList >(cpp_arg1);
        // entryList(QFlags<QDir::Filter>,QStringList)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QStringList cpp_result = cppSelf->QAbstractFileEngine::entryList(cpp_arg0, *cpp_arg1);
        py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_entryList_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_entryList_TypeError:
        const char* overloads[] = {"PySide.QtCore.QDir.Filters, PySide.QtCore.QStringList", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QAbstractFileEngine.entryList", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_error(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // error()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    QFile::FileError cpp_result = cppSelf->QAbstractFileEngine::error();
    py_result = Shiboken::Converter<QFile::FileError >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_errorString(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // errorString()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QAbstractFileEngine::errorString();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_fileFlags(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // fileFlags(QFlags<QAbstractFileEngine::FileFlag>)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QFlags<QAbstractFileEngine::FileFlag> cpp_result = cppSelf->QAbstractFileEngine::fileFlags();
        py_result = Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::toPython(cpp_result);
    } else if (SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Check(arg)) {
        QFlags<QAbstractFileEngine::FileFlag> cpp_arg0 = Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::toCpp(arg);
        // fileFlags(QFlags<QAbstractFileEngine::FileFlag>)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QFlags<QAbstractFileEngine::FileFlag> cpp_result = cppSelf->QAbstractFileEngine::fileFlags(cpp_arg0);
        py_result = Shiboken::Converter<QFlags<QAbstractFileEngine::FileFlag> >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_fileFlags_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_fileFlags_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.FileFlags = QAbstractFileEngine.FileInfoAll", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.fileFlags", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_fileName(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // fileName(QAbstractFileEngine::FileName)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QAbstractFileEngine::fileName();
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    } else if (Shiboken::Converter<QAbstractFileEngine::FileName >::isConvertible(arg)) {
        QAbstractFileEngine::FileName cpp_arg0 = Shiboken::Converter<QAbstractFileEngine::FileName >::toCpp(arg);
        // fileName(QAbstractFileEngine::FileName)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QAbstractFileEngine::fileName(cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_fileName_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_fileName_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.FileName = DefaultName", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.fileName", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_fileTime(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QAbstractFileEngine::FileTime >::isConvertible(arg)) {
        QAbstractFileEngine::FileTime cpp_arg0 = Shiboken::Converter<QAbstractFileEngine::FileTime >::toCpp(arg);
        // fileTime(QAbstractFileEngine::FileTime)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QDateTime cpp_result = cppSelf->QAbstractFileEngine::fileTime(cpp_arg0);
        py_result = Shiboken::Converter<QDateTime >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_fileTime_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_fileTime_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.FileTime", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.fileTime", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_flush(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    // flush()
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    thread_state_saver.save();
    bool cpp_result = cppSelf->QAbstractFileEngine::flush();
    thread_state_saver.restore();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_handle(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // handle()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QAbstractFileEngine::handle();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_isRelativePath(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isRelativePath()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QAbstractFileEngine::isRelativePath();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_isSequential(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isSequential()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QAbstractFileEngine::isSequential();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_link(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // link(QString)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::link(*cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_link_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_link_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.link", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_mkdir(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "mkdir", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QString& >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QString& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
        // mkdir(QString,bool)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::mkdir(*cpp_arg0, cpp_arg1);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_mkdir_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_mkdir_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString, bool", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QAbstractFileEngine.mkdir", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_open(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (SbkPySide_QtCore_QIODevice_OpenModeFlag_Check(arg)) {
        QFlags<QIODevice::OpenModeFlag> cpp_arg0 = Shiboken::Converter<QFlags<QIODevice::OpenModeFlag> >::toCpp(arg);
        // open(QFlags<QIODevice::OpenModeFlag>)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::open(cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_open_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_open_TypeError:
        const char* overloads[] = {"PySide.QtCore.QIODevice.OpenMode", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.open", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_owner(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QAbstractFileEngine::FileOwner >::isConvertible(arg)) {
        QAbstractFileEngine::FileOwner cpp_arg0 = Shiboken::Converter<QAbstractFileEngine::FileOwner >::toCpp(arg);
        // owner(QAbstractFileEngine::FileOwner)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        QString cpp_result = cppSelf->QAbstractFileEngine::owner(cpp_arg0);
        py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_owner_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_owner_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.FileOwner", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.owner", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_ownerId(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QAbstractFileEngine::FileOwner >::isConvertible(arg)) {
        QAbstractFileEngine::FileOwner cpp_arg0 = Shiboken::Converter<QAbstractFileEngine::FileOwner >::toCpp(arg);
        // ownerId(QAbstractFileEngine::FileOwner)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        uint cpp_result = cppSelf->QAbstractFileEngine::ownerId(cpp_arg0);
        py_result = Shiboken::Converter<unsigned int >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_ownerId_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_ownerId_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.FileOwner", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.ownerId", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_pos(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // pos()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    qint64 cpp_result = cppSelf->QAbstractFileEngine::pos();
    py_result = Shiboken::Converter<long long >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_remove(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    // remove()
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    thread_state_saver.save();
    bool cpp_result = cppSelf->QAbstractFileEngine::remove();
    thread_state_saver.restore();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_rename(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // rename(QString)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::rename(*cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_rename_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_rename_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.rename", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_rmdir(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "rmdir", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QString& >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!Shiboken::Converter<QString& >::isConvertible(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        bool cpp_arg1 = Shiboken::Converter<bool >::toCpp(pyargs[1]);
        // rmdir(QString,bool)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::rmdir(*cpp_arg0, cpp_arg1);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_rmdir_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_rmdir_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString, bool", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QAbstractFileEngine.rmdir", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_seek(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (PyNumber_Check(arg)) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(arg);
        // seek(qint64)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QAbstractFileEngine::seek(cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_seek_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_seek_TypeError:
        const char* overloads[] = {"long long", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.seek", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_setError(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setError", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QFile::FileError >::isConvertible(pyargs[0]) && Shiboken::Converter<QString& >::isConvertible(pyargs[1])) {
        QFile::FileError cpp_arg0 = Shiboken::Converter<QFile::FileError >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!Shiboken::Converter<QString& >::isConvertible(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        // setError(QFile::FileError,QString)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        cppSelf->QAbstractFileEngine::setError(cpp_arg0, *cpp_arg1);
    } else goto SbkQAbstractFileEngineFunc_setError_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQAbstractFileEngineFunc_setError_TypeError:
        const char* overloads[] = {"PySide.QtCore.QFile.FileError, PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QAbstractFileEngine.setError", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_setFileName(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QString& >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!Shiboken::Converter<QString& >::isConvertible(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // setFileName(QString)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        cppSelf->QAbstractFileEngine::setFileName(*cpp_arg0);
    } else goto SbkQAbstractFileEngineFunc_setFileName_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQAbstractFileEngineFunc_setFileName_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.setFileName", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_setPermissions(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        uint cpp_arg0 = Shiboken::Converter<uint >::toCpp(arg);
        // setPermissions(uint)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QAbstractFileEngine::setPermissions(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_setPermissions_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_setPermissions_TypeError:
        const char* overloads[] = {"unsigned int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.setPermissions", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_setSize(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(arg);
        // setSize(qint64)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QAbstractFileEngine::setSize(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_setSize_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_setSize_TypeError:
        const char* overloads[] = {"long long", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.setSize", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_size(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // size()const
    QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
    qint64 cpp_result = cppSelf->QAbstractFileEngine::size();
    py_result = Shiboken::Converter<long long >::toPython(cpp_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;
}

static PyObject*
SbkQAbstractFileEngineFunc_supportsExtension(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QAbstractFileEngine::Extension >::isConvertible(arg)) {
        QAbstractFileEngine::Extension cpp_arg0 = Shiboken::Converter<QAbstractFileEngine::Extension >::toCpp(arg);
        // supportsExtension(QAbstractFileEngine::Extension)const
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QAbstractFileEngine::supportsExtension(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_supportsExtension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_supportsExtension_TypeError:
        const char* overloads[] = {"PySide.QtCore.QAbstractFileEngine.Extension", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QAbstractFileEngine.supportsExtension", overloads);
        return 0;
}

static PyObject*
SbkQAbstractFileEngineFunc_write(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "write", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<const char * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        qint64 cpp_arg1 = Shiboken::Converter<qint64 >::toCpp(pyargs[1]);
        // write(const char*,qint64)
        QAbstractFileEngine* cppSelf = Shiboken::Converter<QAbstractFileEngine* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        qint64 cpp_result = cppSelf->QAbstractFileEngine::write(cpp_arg0, cpp_arg1);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<long long >::toPython(cpp_result);
    } else goto SbkQAbstractFileEngineFunc_write_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
        return py_result;
;

    SbkQAbstractFileEngineFunc_write_TypeError:
        const char* overloads[] = {"str, long long", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QAbstractFileEngine.write", overloads);
        return 0;
}

static PyMethodDef SbkQAbstractFileEngine_methods[] = {
    {"atEnd", (PyCFunction)SbkQAbstractFileEngineFunc_atEnd, METH_NOARGS},
    {"caseSensitive", (PyCFunction)SbkQAbstractFileEngineFunc_caseSensitive, METH_NOARGS},
    {"close", (PyCFunction)SbkQAbstractFileEngineFunc_close, METH_NOARGS},
    {"copy", (PyCFunction)SbkQAbstractFileEngineFunc_copy, METH_O},
    {"create", (PyCFunction)SbkQAbstractFileEngineFunc_create, METH_O|METH_STATIC},
    {"entryList", (PyCFunction)SbkQAbstractFileEngineFunc_entryList, METH_VARARGS},
    {"error", (PyCFunction)SbkQAbstractFileEngineFunc_error, METH_NOARGS},
    {"errorString", (PyCFunction)SbkQAbstractFileEngineFunc_errorString, METH_NOARGS},
    {"fileFlags", (PyCFunction)SbkQAbstractFileEngineFunc_fileFlags, METH_NOARGS|METH_O},
    {"fileName", (PyCFunction)SbkQAbstractFileEngineFunc_fileName, METH_NOARGS|METH_O},
    {"fileTime", (PyCFunction)SbkQAbstractFileEngineFunc_fileTime, METH_O},
    {"flush", (PyCFunction)SbkQAbstractFileEngineFunc_flush, METH_NOARGS},
    {"handle", (PyCFunction)SbkQAbstractFileEngineFunc_handle, METH_NOARGS},
    {"isRelativePath", (PyCFunction)SbkQAbstractFileEngineFunc_isRelativePath, METH_NOARGS},
    {"isSequential", (PyCFunction)SbkQAbstractFileEngineFunc_isSequential, METH_NOARGS},
    {"link", (PyCFunction)SbkQAbstractFileEngineFunc_link, METH_O},
    {"mkdir", (PyCFunction)SbkQAbstractFileEngineFunc_mkdir, METH_VARARGS},
    {"open", (PyCFunction)SbkQAbstractFileEngineFunc_open, METH_O},
    {"owner", (PyCFunction)SbkQAbstractFileEngineFunc_owner, METH_O},
    {"ownerId", (PyCFunction)SbkQAbstractFileEngineFunc_ownerId, METH_O},
    {"pos", (PyCFunction)SbkQAbstractFileEngineFunc_pos, METH_NOARGS},
    {"remove", (PyCFunction)SbkQAbstractFileEngineFunc_remove, METH_NOARGS},
    {"rename", (PyCFunction)SbkQAbstractFileEngineFunc_rename, METH_O},
    {"rmdir", (PyCFunction)SbkQAbstractFileEngineFunc_rmdir, METH_VARARGS},
    {"seek", (PyCFunction)SbkQAbstractFileEngineFunc_seek, METH_O},
    {"setError", (PyCFunction)SbkQAbstractFileEngineFunc_setError, METH_VARARGS},
    {"setFileName", (PyCFunction)SbkQAbstractFileEngineFunc_setFileName, METH_O},
    {"setPermissions", (PyCFunction)SbkQAbstractFileEngineFunc_setPermissions, METH_O},
    {"setSize", (PyCFunction)SbkQAbstractFileEngineFunc_setSize, METH_O},
    {"size", (PyCFunction)SbkQAbstractFileEngineFunc_size, METH_NOARGS},
    {"supportsExtension", (PyCFunction)SbkQAbstractFileEngineFunc_supportsExtension, METH_O},
    {"write", (PyCFunction)SbkQAbstractFileEngineFunc_write, METH_VARARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQAbstractFileEngine_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QAbstractFileEngine",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::SbkBaseWrapper_Dealloc< QAbstractFileEngineWrapper >,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQAbstractFileEngine_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQAbstractFileEngine_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0
};

static SbkBaseWrapperType* SbkQAbstractFileEngine_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQAbstractFileEngine_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QAbstractFileEngine*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

static PyGetSetDef SbkPySide_QtCore_QAbstractFileEngine_FileName_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_FileName_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_FileName_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "FileName",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QAbstractFileEngine_FileName_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_FileName_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileName_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtCore_QAbstractFileEngine_FileTime_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_FileTime_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "FileTime",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QAbstractFileEngine_FileTime_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_FileTime_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileTime_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtCore_QAbstractFileEngine_FileOwner_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_FileOwner_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "FileOwner",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QAbstractFileEngine_FileOwner_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_FileOwner_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileOwner_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

static PyGetSetDef SbkPySide_QtCore_QAbstractFileEngine_Extension_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_Extension_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_Extension_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Extension",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QAbstractFileEngine_Extension_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_Extension_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_Extension_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlag___and__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QAbstractFileEngine::FileFlags >::toPython(
        ((QAbstractFileEngine::FileFlags) ((SbkEnumObject*)self)->ob_ival)
        & Shiboken::Converter< QAbstractFileEngine::FileFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlag___or__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QAbstractFileEngine::FileFlags >::toPython(
        ((QAbstractFileEngine::FileFlags) ((SbkEnumObject*)self)->ob_ival)
        | Shiboken::Converter< QAbstractFileEngine::FileFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlag___xor__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QAbstractFileEngine::FileFlags >::toPython(
        ((QAbstractFileEngine::FileFlags) ((SbkEnumObject*)self)->ob_ival)
        ^ Shiboken::Converter< QAbstractFileEngine::FileFlags >::toCpp(arg)
    );
}

PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlag___invert__(PyObject* self, PyObject* arg)
{
    return Shiboken::Converter< QAbstractFileEngine::FileFlags >::toPython(
        ~ Shiboken::Converter< QAbstractFileEngine::FileFlags >::toCpp(self)
    );
}


static PyNumberMethods SbkPySide_QtCore_QAbstractFileEngine_FileFlag_as_number = {
    /*nb_add*/                  0,
    /*nb_subtract*/             0,
    /*nb_multiply*/             0,
    /*nb_divide*/               0,
    /*nb_remainder*/            0,
    /*nb_divmod*/               0,
    /*nb_power*/                0,
    /*nb_negative*/             0,
    /*nb_positive*/             0,
    /*nb_absolute*/             0,
    /*nb_nonzero*/              0,
    /*nb_invert*/               (unaryfunc)SbkPySide_QtCore_QAbstractFileEngine_FileFlag___invert__,
    /*nb_lshift*/               0,
    /*nb_rshift*/               0,
    /*nb_and*/                  (binaryfunc)SbkPySide_QtCore_QAbstractFileEngine_FileFlag___and__,
    /*nb_xor*/                  (binaryfunc)SbkPySide_QtCore_QAbstractFileEngine_FileFlag___xor__,
    /*nb_or*/                   (binaryfunc)SbkPySide_QtCore_QAbstractFileEngine_FileFlag___or__,
    /*nb_coerce*/               0,
    /*nb_int*/                  0,
    /*nb_long*/                 0,
    /*nb_float*/                0,
    /*nb_oct*/                  0,
    /*nb_hex*/                  0,
    /*nb_inplace_add*/          0,
    /*nb_inplace_subtract*/     0,
    /*nb_inplace_multiply*/     0,
    /*nb_inplace_divide*/       0,
    /*nb_inplace_remainder*/    0,
    /*nb_inplace_power*/        0,
    /*nb_inplace_lshift*/       0,
    /*nb_inplace_rshift*/       0,
    /*nb_inplace_and*/          0,
    /*nb_inplace_xor*/          0,
    /*nb_inplace_or*/           0,
    /*nb_floor_divide*/         0,
    /*nb_true_divide*/          0,
    /*nb_inplace_floor_divide*/ 0,
    /*nb_inplace_true_divide*/  0,
    /*nb_index*/                0
};


static PyGetSetDef SbkPySide_QtCore_QAbstractFileEngine_FileFlag_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_FileFlag_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "FileFlag",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        &SbkPySide_QtCore_QAbstractFileEngine_FileFlag_as_number,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QAbstractFileEngine_FileFlag_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_FileFlag_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlag_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QAbstractFileEngine_FileFlags_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type = {
    PyObject_HEAD_INIT(&PyType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "FileFlags",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             &SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QAbstractFileEngine_FileFlags_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject*
SbkPySide_QtCore_QAbstractFileEngine_FileFlags_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    if (!PyType_IsSubtype(type, &SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type))
        return 0;

    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}



PyAPI_FUNC(void)
init_QAbstractFileEngine(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQAbstractFileEngine_Type);

    // Fill type discovery information
    SbkQAbstractFileEngine_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQAbstractFileEngine_Type.type_discovery->addTypeDiscoveryFunction(&SbkQAbstractFileEngine_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQAbstractFileEngine_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQAbstractFileEngine_Type));
    PyModule_AddObject(module, "QAbstractFileEngine",
        ((PyObject*)&SbkQAbstractFileEngine_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: FileName
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_FILENAME_IDX] = &SbkPySide_QtCore_QAbstractFileEngine_FileName_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "FileName",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::DefaultName, "DefaultName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "DefaultName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "DefaultName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::BaseName, "BaseName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "BaseName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "BaseName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::PathName, "PathName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "PathName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "PathName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::AbsoluteName, "AbsoluteName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "AbsoluteName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "AbsoluteName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::AbsolutePathName, "AbsolutePathName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "AbsolutePathName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "AbsolutePathName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::LinkName, "LinkName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "LinkName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "LinkName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::CanonicalName, "CanonicalName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "CanonicalName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "CanonicalName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::CanonicalPathName, "CanonicalPathName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "CanonicalPathName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "CanonicalPathName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::BundleName, "BundleName");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "BundleName", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "BundleName", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileName_Type,
        (long) QAbstractFileEngine::NFileNames, "NFileNames");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "NFileNames", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileName_Type.tp_dict,
        "NFileNames", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QAbstractFileEngine::FileName");

    // init enum class: FileTime
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_FILETIME_IDX] = &SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "FileTime",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type,
        (long) QAbstractFileEngine::CreationTime, "CreationTime");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "CreationTime", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type.tp_dict,
        "CreationTime", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type,
        (long) QAbstractFileEngine::ModificationTime, "ModificationTime");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ModificationTime", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type.tp_dict,
        "ModificationTime", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type,
        (long) QAbstractFileEngine::AccessTime, "AccessTime");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "AccessTime", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileTime_Type.tp_dict,
        "AccessTime", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QAbstractFileEngine::FileTime");

    // init enum class: FileOwner
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_FILEOWNER_IDX] = &SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "FileOwner",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type,
        (long) QAbstractFileEngine::OwnerUser, "OwnerUser");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "OwnerUser", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type.tp_dict,
        "OwnerUser", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type,
        (long) QAbstractFileEngine::OwnerGroup, "OwnerGroup");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "OwnerGroup", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileOwner_Type.tp_dict,
        "OwnerGroup", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QAbstractFileEngine::FileOwner");

    // init enum class: Extension
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_EXTENSION_IDX] = &SbkPySide_QtCore_QAbstractFileEngine_Extension_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "Extension",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type,
        (long) QAbstractFileEngine::AtEndExtension, "AtEndExtension");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "AtEndExtension", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_Extension_Type.tp_dict,
        "AtEndExtension", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type,
        (long) QAbstractFileEngine::FastReadLineExtension, "FastReadLineExtension");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "FastReadLineExtension", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_Extension_Type.tp_dict,
        "FastReadLineExtension", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type,
        (long) QAbstractFileEngine::MapExtension, "MapExtension");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "MapExtension", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_Extension_Type.tp_dict,
        "MapExtension", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_Extension_Type,
        (long) QAbstractFileEngine::UnMapExtension, "UnMapExtension");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "UnMapExtension", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_Extension_Type.tp_dict,
        "UnMapExtension", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QAbstractFileEngine::Extension");

    // init enum class: FileFlag
    SbkPySide_QtCoreTypes[SBK_QABSTRACTFILEENGINE_FILEFLAG_IDX] = &SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "FileFlag",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type));

    // init flags class: QFlags<QAbstractFileEngine::FileFlag>
    SbkPySide_QtCoreTypes[SBK_QFLAGS_QABSTRACTFILEENGINE_FILEFLAG__IDX] = &SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type);
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
            "FileFlags",((PyObject*)&SbkPySide_QtCore_QAbstractFileEngine_FileFlags_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ReadOwnerPerm, "ReadOwnerPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ReadOwnerPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ReadOwnerPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::WriteOwnerPerm, "WriteOwnerPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "WriteOwnerPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "WriteOwnerPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ExeOwnerPerm, "ExeOwnerPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ExeOwnerPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ExeOwnerPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ReadUserPerm, "ReadUserPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ReadUserPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ReadUserPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::WriteUserPerm, "WriteUserPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "WriteUserPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "WriteUserPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ExeUserPerm, "ExeUserPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ExeUserPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ExeUserPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ReadGroupPerm, "ReadGroupPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ReadGroupPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ReadGroupPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::WriteGroupPerm, "WriteGroupPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "WriteGroupPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "WriteGroupPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ExeGroupPerm, "ExeGroupPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ExeGroupPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ExeGroupPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ReadOtherPerm, "ReadOtherPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ReadOtherPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ReadOtherPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::WriteOtherPerm, "WriteOtherPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "WriteOtherPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "WriteOtherPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ExeOtherPerm, "ExeOtherPerm");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ExeOtherPerm", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ExeOtherPerm", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::LinkType, "LinkType");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "LinkType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "LinkType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::FileType, "FileType");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "FileType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "FileType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::DirectoryType, "DirectoryType");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "DirectoryType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "DirectoryType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::BundleType, "BundleType");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "BundleType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "BundleType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::HiddenFlag, "HiddenFlag");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "HiddenFlag", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "HiddenFlag", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::LocalDiskFlag, "LocalDiskFlag");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "LocalDiskFlag", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "LocalDiskFlag", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::ExistsFlag, "ExistsFlag");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "ExistsFlag", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "ExistsFlag", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::RootFlag, "RootFlag");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "RootFlag", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "RootFlag", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::Refresh, "Refresh");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "Refresh", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "Refresh", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::PermsMask, "PermsMask");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "PermsMask", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "PermsMask", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::TypesMask, "TypesMask");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "TypesMask", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "TypesMask", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::FlagsMask, "FlagsMask");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "FlagsMask", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "FlagsMask", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type,
        (long) QAbstractFileEngine::FileInfoAll, "FileInfoAll");
    PyDict_SetItemString(SbkQAbstractFileEngine_Type.super.ht_type.tp_dict,
        "FileInfoAll", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QAbstractFileEngine_FileFlag_Type.tp_dict,
        "FileInfoAll", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QAbstractFileEngine::FileFlag");

    Shiboken::TypeResolver::createObjectTypeResolver<QAbstractFileEngine >("QAbstractFileEngine*");
    Shiboken::TypeResolver::createObjectTypeResolver<QAbstractFileEngine >(typeid(QAbstractFileEngine).name());
}


} // extern "C"

namespace Shiboken
{
// Copy C++ object implementation
// Converter implementations
} // namespace Shiboken

