/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __TRIGGER_H__
#define __TRIGGER_H__

#include <QObject>
#include <QStringList>
#include <QHash>

#include <boost_headers.hpp>
#include <pyqt_signal_slot.hpp>
#include <pyside_global.hpp>

namespace PySide
{

class abstract_qobject_connection;
struct trigger_data;

int callable_arg_count(PyObject *callable);
inline int
callable_arg_count(boost::python::object callable)
{
    return callable_arg_count(callable.ptr());
}

/**
*   The problem:
*       When Qt emits a signal, it need to call the python implementation 
*       of the slot, by Qt can't do it because it's does not know about python.
*   The solution:
*       Catch the signal emitted by Qt then redirect it to the python
*       implementation, to do this, the signals are not connected to the real
*       QObject, but to the trigger (this class).
*
*   To do this, all methods implemmented by the usual QOBJECT macro are
*   implemmented by us, so we can do what we want when we receive a signal.
*
*   For more info: \link http://doc.trolltech.com/qq/qq16-dynamicqobject.html
*/
class PYSIDE_LOCAL trigger : public QObject
{
public:
    /// Constructs the trigger for the object \p parent
    trigger(QObject *parent);
    ~trigger();

    const QMetaObject *metaObject() const;

    /// Create a dynamic slot associeted with a calleble object,
    /// to extenal use in other library
    pyqt_slot register_dynamic_slot(const boost::python::object &callback);

    void invalidate_metaobject();

    /// Call a list of slots from the qobject handled by
    /// this trigger with \p args as arguments.
    void fire(const QList<int> slot_indexes,
              const boost::python::object &args);

    /// Connect a signal to a python callable object
    bool connect(QObject* src, const pyqt_signal& signal,
                 const boost::python::object& callback, int num_slot_args,
                 Qt::ConnectionType type = Qt::AutoConnection);
    /// Connect a signal to a qobject slot
    bool connect(QObject* src, const pyqt_signal& signal,
                 QObject* receiver, const pyqt_slot& slot,
                 Qt::ConnectionType type = Qt::AutoConnection);
    /// disconnect a signal from a python callable object
    bool disconnect(const QObject* src, const pyqt_signal& signal,
                    const boost::python::object &callback,
                    int* slot_index_found = 0);
    /// disconnect a signal from a qobject slot
    bool disconnect(const QObject* src, const pyqt_signal& signal,
                    const QObject* receiver, const pyqt_slot& slot,
                    int* slot_index_found = 0);
protected:
    /**
    *   Qt's meta-object system uses the qt_metacall() function to access the
    *   meta-information for a particular QObject object (its signals, slots,
    *   properties, etc.).
    *
    *   About the implemmentation
    *
    *   The call may indicate access to the meta-object of the QObject base
    *   class, we need to take care of this use case.
    *   If the QObject::qt_metacall() call returns -1, this means that the
    *   metacall has been handled by QObject and that there is nothing to do.
    *   In that case, we return immediately. Similarly, if the metacall isn't
    *   a slot invocation, we follow QObject's convention and return an
    *   identifier that can be handled by a subclass.
    *
    *   In the case of a method invocation, we use the identifier to find the
    *   connection_info, connection_info is a class that store some info about
    *   the signal/slot connection, as it's name suggests. connection_info's
    *   are created when a signal is connected to a slot and every
    *   connection_info has a identifier, we use the slot_index parameter to
    *   pass the connection_info index, so the connection_info index MUST not
    *   be a valid slot_index, we take care of this by creating connection_info
    *   indexes greater than QObject::metaObject()->methodCount().
    *
    *   If all goes well, we invoke the specified slot (using py_fire()) and
    *   return -1 to indicate that the metacall has been processed.
    *
    *   \see \link http://doc.trolltech.com/qq/qq16-dynamicqobject.html
    */
    int qt_metacall(QMetaObject::Call c, int id, void **args);

    //used to call mehtos inside of Qt mail loop
    void timerEvent(QTimerEvent *event);

private:
    trigger_data *m_data;

    void py_fire(const abstract_qobject_connection* ci, void **args);
    int add_connection_info(abstract_qobject_connection *ci,
                            const pyqt_signal& signal = pyqt_signal());
    bool connect(QObject* src, const pyqt_signal& signal,
                 abstract_qobject_connection* ci, Qt::ConnectionType type = Qt::AutoConnection);
    void update_metaobject() const;


    void call_method(const abstract_qobject_connection *ci, const boost::python::object &args);
    void call_method(const abstract_qobject_connection *ci, void **args);

    //thread safe call mehtods
    void idle_call_method(const abstract_qobject_connection *ci, const boost::python::object &args);
};

} // namespace PySide

#endif

