/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <QtCore/QString>
#include <QtCore/QBool>

#include "boost_headers.hpp"
#include "type_converter.hpp"
#include "type_manager.hpp"

using namespace boost;

namespace PySide
{

/************************* QString ****************************/
struct Py2QString
{
    static void*
    convertible(PyObject* py_obj)
    {
#if PY_VERSION_HEX < 0x03000000
        if (!(PyString_Check(py_obj) || PyUnicode_Check(py_obj)) )
#else
        if (!PyUnicode_Check(py_obj))
#endif
            return 0;

        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef python::converter::rvalue_from_python_storage<QString> storage_t;
        storage_t* the_storage = reinterpret_cast<storage_t*>(data);
        void* memory_chunk = the_storage->storage.bytes;

#if PY_VERSION_HEX < 0x03000000
        if (PyString_Check(py_obj)) {
            const char* value = python::extract<const char*>(py_obj);
            new (memory_chunk) QString(value);
        } else {
#else
        {
#endif
            Py_ssize_t size = PyUnicode_GetSize(py_obj);
            wchar_t* ws = new wchar_t[size];
            PyUnicode_AsWideChar((PyUnicodeObject*)py_obj, ws, size);
            new (memory_chunk) QString(QString::fromWCharArray(ws, size));
            delete ws;
        }

        data->convertible = memory_chunk;
    }
};

void
register_qstring_converter()
{
    python::converter::registry::push_back(&Py2QString::convertible,
                                           &Py2QString::construct,
                                           python::type_id<QString>());
}

/************************* QBool ****************************/

struct PYSIDE_LOCAL QBool2Py
{
    static PyObject*
    convert(const QBool& qbool)
    {
        return python::incref(boost::python::object(qbool==true).ptr());
    }
};


void
register_qbool_converter()
{
    python::to_python_converter<QBool, QBool2Py>();
}

/************************* QByteArray ****************************/
struct Py2QByteArray
{
    static void*
    convertible(PyObject* py_obj)
    {
        if (
#if PY_VERSION_HEX < 0x03000000
            !PyString_Check(py_obj) &&
            // old way buffer
            !PyBuffer_Check(py_obj) &&
#else
            // buffer new generation
            !PyObject_CheckBuffer(py_obj) &&
#endif
            !PyUnicode_Check(py_obj)

           )
            return 0;

        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef python::converter::rvalue_from_python_storage<QByteArray> storage_t;
        storage_t* the_storage = reinterpret_cast<storage_t*>(data);
        void* memory_chunk = the_storage->storage.bytes;

#if PY_VERSION_HEX < 0x03000000
        if (PyString_Check(py_obj)) {
#else
        if (PyUnicode_Check(py_obj)) {
#endif
            new (memory_chunk) QByteArray(python::extract<const char*>(py_obj));
            data->convertible = memory_chunk;
        }
#if PY_VERSION_HEX < 0x03000000
        else if (py_obj->ob_type->tp_as_buffer
               && PyType_HasFeature(py_obj->ob_type, Py_TPFLAGS_HAVE_GETCHARBUFFER)
               && py_obj->ob_type->tp_as_buffer->bf_getcharbuffer) {
            QByteArray charData;
            PyBufferProcs* bufferProcs = py_obj->ob_type->tp_as_buffer;
            int segments = bufferProcs->bf_getsegcount(py_obj, 0);
            for (int i = 0; i < segments; ++i) {
                char* segmentData;
                int length = bufferProcs->bf_getcharbuffer(py_obj, i, &segmentData);
                if (length == -1)
                    break;
                charData.append(segmentData, length);
            }
            new (memory_chunk) QByteArray(charData);
            data->convertible = memory_chunk;
        }
#else
        else if (PyObject_CheckBuffer(py_obj)) {
            Py_buffer buf;
            if (PyObject_GetBuffer(py_obj, &buf, PyBUF_SIMPLE) == 0) {
                new (memory_chunk) QByteArray((char *)buf.buf, buf.len);
                data->convertible = memory_chunk;
            }
        }
#endif
    }
};

void
register_qbytearray_converter()
{
    python::converter::registry::push_back (&Py2QByteArray::convertible,
                                            &Py2QByteArray::construct,
                                            python::type_id<QByteArray>());
}

} // namespace PySide

