/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef PYQTSIGNALSLOT_H
#define PYQTSIGNALSLOT_H

#include <QByteArray>
#include <QStringList>

#include <pyside_global.hpp>

namespace PySide
{

class PYSIDE_API pyqt_slot
{
private:
    QByteArray m_data;
protected:
    bool m_is_signal;
public:
    explicit pyqt_slot(const QByteArray& data = QByteArray())
            : m_data(data), m_is_signal(false) {}
    const QByteArray& signature() const { return m_data; }
    bool is_slot() const { return !m_is_signal; }
    bool is_signal() const { return m_is_signal; }
    bool is_null() const { return m_data.isEmpty(); }
    void parse_signature(QStringList* result) const;

    QByteArray
    toQtStyleSignature() const
    {
        #if QSLOT_CODE != 1 || QSIGNAL_CODE != 2
            #error QSLOT_CODE and/or QSIGNAL_CODE changed! \
                    change the hardcoded stuff to the correct value!
        #endif
        char code = is_signal() ? '2' : '1';
        return QByteArray(m_data).prepend(code);
    }

    QByteArray
    function_name() const
    {
        int idx = m_data.indexOf('(');
        return idx < 0 ? QByteArray() : m_data.left(idx);
    }

    bool
    operator==(const pyqt_slot& other) const
    {
        return m_is_signal == other.m_is_signal && m_data == other.m_data;
    }
};

// this class is just for type checking and init some constants for signals.
// dont add any attributes to it!! because we cast it to pyqt_slot by value!
class PYSIDE_API pyqt_signal : public pyqt_slot
{
public:
    explicit
    pyqt_signal(const QByteArray& data = QByteArray())
            : pyqt_slot(data)
    {
        m_is_signal = true;
    }
};

// TODO: this does not need to be inlined
inline PYSIDE_API void
pyqt_slot::parse_signature(QStringList* result) const
{
    QString str_signature(m_data);

    if (str_signature.contains("()") || str_signature.contains("(void)")) {
        result->clear();
        return;
    } else if (str_signature.contains('(')) {
        //get args types
        QString types = str_signature.replace(QRegExp(".+\\((.*)\\)"), "\\1");
        *result = types.split(',');
    } else {
        result->clear();
    }
}

} // namespace PySide

#endif

