/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "qptr.hpp"
#include "wrapper.hpp"

#include <QCoreApplication>
#include <QGraphicsItem>
#include <QLinkedList>
#include <QDebug>

using namespace boost;

namespace PySide
{

class PYSIDE_LOCAL qptr_base::qptr_base_private
{
public:
    int refcount;
    enum qptr_state {
        acquired        = 1,
        invalid         = 2
    };
    unsigned int state;
    void* ptr;
    bool internal_cpp_ref;
    PyObject* py_obj;
    void (*deleter)(void*);
    qptr_base_private* parent;
    bool is_wrapper;

    qptr_base_private(void* cpp_obj, void (*deleter)(void*))
        : refcount(1),
          state(acquired),
          ptr(cpp_obj),
          internal_cpp_ref(false),
          py_obj(0),
          deleter(deleter),
          parent(0),
          is_wrapper(false),
          m_children(0)
    {
    }

    ~qptr_base_private()
    {
        delete m_children;
        m_children = 0;
    }

    QLinkedList<qptr_base>&
    children()
    {
        if (!m_children) {
            m_children = new QLinkedList<qptr_base>();
        }
        return *m_children;
    }

    void
    remove_child(qptr_base& child) {
        if (m_children && m_children->removeOne(child)) {
            PyObject *py_obj = child.get_pyobject();
            if (py_obj) {
                python::decref(py_obj);
            }
            child.m_data->parent = 0;
        }
    }

    void
    remove_children()
    {
        delete m_children;
        m_children = 0;
    }

    bool
    has_children()
    {
        return m_children ? !m_children->empty() : false;
    }

private:
    QLinkedList<qptr_base>* m_children;
    qptr_base_private(qptr_base_private&);
    qptr_base_private& operator=(qptr_base_private&);
};

QHash<const volatile void*, qptr_base::qptr_base_private*> qptr_base::m_cpp_to_qptrs;

qptr_base::qptr_base(void *cpp_obj, PyObject *py_obj, void (*deleter)(void*), int mode)
    : m_data(0)
{
    Q_ASSERT(cpp_obj);
    Q_ASSERT(deleter);
    bool use_cache = mode & check_cache;
    bool is_wrapper = mode & wrapper_pointer;

    if (use_cache && m_cpp_to_qptrs.contains(cpp_obj)) {
        m_data = m_cpp_to_qptrs[cpp_obj];
        m_data->refcount++;
    } else {
        m_data = new qptr_base_private(cpp_obj, deleter);
        m_cpp_to_qptrs[cpp_obj] =  m_data;
        m_data->is_wrapper = is_wrapper;
        m_data->py_obj = py_obj;
    }
}

qptr_base::qptr_base(const qptr_base& other)
{
    m_data = other.m_data;
    Q_ASSERT(m_data);
    m_data->refcount++;
}

qptr_base::qptr_base(qptr_base::qptr_base_private* data) : m_data(data)
{
    Q_ASSERT(m_data);
    m_data->refcount++;
}

void
qptr_base::add_child(qptr_base& child)
{
    Q_ASSERT(m_data);
    if (m_data->children().contains(child)) {
        return;
    } else {
        //inc pyref to avoid delete child
        PyObject *py_obj = child.get_pyobject();
        if (py_obj) {
            python::incref(py_obj);
        }

        //remove old parent
        if (child.m_data->parent) {
            child.remove_parent();
        }
        m_data->children() << child;
        child.m_data->parent = m_data;
    }
}

void
qptr_base::remove_child(qptr_base& child)
{
    Q_ASSERT(m_data);
    m_data->remove_child(child);
}

void
qptr_base::remove_parent()
{
    Q_ASSERT(m_data);
    if (m_data->parent) {
        m_data->parent->remove_child(*this);
    }
}

PyObject*
qptr_base::get_pyobject()
{
    Q_ASSERT(m_data);

    return m_data->py_obj;
}

void
qptr_base::set_pyobject(PyObject* py_obj)
{
    Q_ASSERT(m_data);
    m_data->py_obj = py_obj;
}

qptr_base::~qptr_base()
{
    release();
}

void
qptr_base::invalidate()
{
    invalidate(m_data->state & qptr_base_private::acquired);
}

void
qptr_base::invalidate(bool invalidate_ptr)
{
    Q_ASSERT(m_data);

    if (m_data->state & qptr_base_private::invalid) {
        return;
    }

    if (m_data->has_children()) {
        bool invalidate_child_ptr = m_data->state & qptr_base_private::acquired;
        foreach (qptr_base ptr, m_data->children()) {
            ptr.invalidate(invalidate_ptr && invalidate_child_ptr);
        }
        m_data->remove_children();
    }

    remove_parent();

    if (invalidate_ptr) {
       m_cpp_to_qptrs.remove(m_data->ptr);
       m_data->state |= qptr_base_private::invalid;
    }

}

/**
 * This function is called from wrapper classes to inform
 * that the destructor was called from C++ side.
 */
void
qptr_base::invalidate(void* cpp_obj, bool cleanup)
{
    qptr_base_private *data = m_cpp_to_qptrs.value(cpp_obj);

    if (data) {
        qptr_base ptr(data);
        if (cleanup && ptr.has_cpp_ref()) {
            ptr.invalidate();
            ptr.remove_cpp_ref();
        } else if(!cleanup) {
            ptr.invalidate();
        }
    }
}

bool
qptr_base::exists(void* cpp_obj)
{
    return m_cpp_to_qptrs.contains(cpp_obj);
}

bool
qptr_base::is_null() const
{
    return !m_data || m_data->state & qptr_base_private::invalid;
}

void
qptr_base::destroy()
{
    //remove from global list to avoid recursive call
    m_cpp_to_qptrs.remove(m_data->ptr);

    bool can_delete = (m_data->state & qptr_base_private::acquired)
                        && !(m_data->state & qptr_base_private::invalid);

    if (can_delete && m_data->ptr) {
        //delete pointer
        m_data->deleter(m_data->ptr);
    }

    //invalidate all children
    invalidate();

    //delete shared data
    delete m_data;
    m_data = 0;
}

void
qptr_base::release()
{
    if (m_data) {
        Q_ASSERT(m_data->refcount > 0);
        m_data->refcount--;
        if (m_data->refcount == 0) {
            destroy();
        }
    }
}

int
qptr_base::children_count()
{
    return m_data->has_children() ? m_data->children().count() : 0;
}

bool
qptr_base::has_parent()
{
    return m_data->parent;
}

int
qptr_base::refcount()
{
    return m_data ? m_data->refcount : 0;
}

void*
qptr_base::raw_ptr() const
{
    Q_ASSERT(m_data);
    return m_data->ptr;
}

qptr_base::qptr_base_private*
qptr_base::create_copy() const
{
    if (m_data) {
        m_data->refcount++;
    }
    return m_data;
}


qptr_base&
qptr_base::operator=(const qptr_base& other)
{
    if (m_data != other.m_data) {
        release();
        m_data = other.create_copy();
    }
    return *this;
}

void
qptr_base::acquire_ownership()
{
    Q_ASSERT(m_data);

    m_data->state |= qptr_base_private::acquired;

    //force get ownership of c++ pointer
    if (has_cpp_ref()) {
        remove_cpp_ref();
    }
}

void
qptr_base::release_ownership()
{
    Q_ASSERT(m_data);

    m_data->state &= ~qptr_base_private::acquired;

    if (is_wrapper())
        add_cpp_ref();
}

bool
qptr_base::has_ownership()
{
    return m_data->state & qptr_base_private::acquired;
}

//this will mantain PyObject live if all refs to him was removed
void
qptr_base::add_cpp_ref()
{
    if (!m_data->internal_cpp_ref) {
        PyObject *py_obj = get_pyobject();
        if (py_obj) {
            m_data->internal_cpp_ref = true;
            python::incref(py_obj);
        }
    }
}

// this will remove internal ref to PyOjbect then he can
// rest in peace when all PyObject lost the reference
void
qptr_base::remove_cpp_ref()
{
    if (m_data->internal_cpp_ref) {
        m_data->internal_cpp_ref = false;
        python::decref(get_pyobject());
        release_ownership();
    }
}

bool
qptr_base::has_cpp_ref()
{
    return m_data->internal_cpp_ref;
}

bool
qptr_base::ptr_assert() const
{
    Q_ASSERT(m_data);
    if (m_data->state & qptr_base_private::invalid) {
        PyErr_SetString(PyExc_RuntimeError,
                        "internal C++ object already deleted.");
        python::throw_error_already_set(); // throw cpp exception
    }
    return true;
}

bool
qptr_base::is_wrapper() const
{
    return m_data->is_wrapper;
}

} // namespace PySide


