#!/usr/bin/env python

import errno
import os
import sys

from PySide.QtCore import Qt, QCoreApplication, QDir, QFile, QIODevice, \
                          QString, QStringList, QTextStream, QVariant
from PySide.QtSql  import QSqlDatabase, QSqlQuery

class DBException(Exception): #{{{1
    def __init__(self, errmsg, dberr):
        self.errmsg = errmsg
        self.dberr  = dberr

    def __str__(self):
        return repr("%s: %s" % (self.errmsg, self.dberr))

class DBExceptionPrepare(DBException): #{{{1
    def __init__(self, query):
        DBException.__init("Error preparing query", query.lastError().text())

class DBExceptionExecute(DBException): #{{{1
    def __init__(self, query):
        DBException.__init("Error executing query", query.lastError().text())

class LTDatabase(): #{{{1
    # Class variables {{{2
    _Conn        = None

    # Define the configuration directory, and the database filename #{{{2
    _configDir = '.pybraryThing'
    _dbFile    = 'database.sqlite'

    # Define the SQL queries #{{{2
    SQL_CREATE_DISPLAY_OPTIONS = 'create table display_options ( column text, display_name text, display_order integer, sort_order integer, display_width integer)'

    SQL_DELETE_BOOKS = 'delete from books'

    SQL_INSERT_DISPLAY_OPTION = 'insert into display_options values(:col, :name, :disp, :sort, 0)'

    SQL_SELECT_DISPLAY_ORDER = 'select column, display_name, display_width from display_options where display_order > 0 order by display_order'
    SQL_SELECT_DISPLAY_SORT = 'select column from display_options where sort_order > 0 order by sort_order'

    SQL_SELECT_FIELDS_DISPLAY = 'select display_name from display_options where display_order > 0 order by display_order'
    SQL_SELECT_FIELDS_NONDISPLAY = "select display_name from display_options where display_order = 0 and column != 'id'"
    SQL_SELECT_FIELDS_NONSORT = 'select display_name from display_options where sort_order = 0'
    SQL_SELECT_FIELDS_SORT = 'select display_name from display_options where sort_order > 0 order by sort_order'

    SQL_SELECT_FIELD_NAMES = 'select column, display_name from display_options'
    SQL_SELECT_BOOK_INFO = 'select * from books where id = :id'

    SQL_UPDATE_DISPLAY_ORDER_1 = 'update display_options set display_order = display_order - 1 where display_order > %d and display_order <= %d'
    SQL_UPDATE_DISPLAY_ORDER_2 = 'update display_options set display_order = display_order + 1 where display_order >= %d and display_order < %d'
    SQL_UPDATE_DISPLAY_ORDER_3 = 'update display_options set display_order = :pos where column = :col'
    SQL_UPDATE_DISPLAY_ORDER_4 = 'update display_options set display_order = :pos where display_name = :col'
    SQL_UPDATE_DISPLAY_WIDTH = 'update display_options set display_width = :size where column = :col'
    SQL_UPDATE_SORT_ORDER = 'update display_options set sort_order = :pos where display_name = :col'

    SQL_RESET_DISPLAY_ORDER = 'update display_options set display_order = 0'
    SQL_RESET_SORT_ORDER = 'update display_options set sort_order = 0'

    # Define the fields to be used (mapped from LibraryThing export) #{{{2
    bookFields = [
        [ "id", "integer", "ID" ],
        [ "title", "text", "Title" ],
        [ "author_lf", "text", "Author (Last, First)" ],
        [ "author_fl", "text", "Author (First Last)" ],
        [ "other_auth", "text", "Other authors" ],
        [ "publication", "text", "Publication" ],
        [ "date", "integer", "Publication date" ],
        [ "isbn", "text", "ISBN" ],
        [ "series", "text", "Series" ],
        [ "source", "text", "Source" ],
        [ "language_1", "text", "Primary language" ],
        [ "language_2", "text", "Secondary language" ],
        [ "orig_lang", "text", "Original language" ],
        [ "lcc", "text", "LC classification" ],
        [ "ddc", "text", "Dewey" ],
        [ "bcid", "text", "BCID" ],
        [ "date_entered", "text", "Date entered" ],
        [ "date_acquired", "text", "Date acquired" ],
        [ "date_started", "text", "Date started reading" ],
        [ "date_finished", "text", "Date finished reading" ],
        [ "stars", "integer", "Rating" ],
        [ "tags", "text", "Tags" ],
        [ "review", "text", "Review" ],
        [ "summary", "text", "Summary" ],
        [ "comments", "text", "Comments" ],
        [ "priv_comments", "text", "Private comments" ],
        [ "copies", "text", "Copies" ],
        [ "encoding", "text", "Entry encoding" ],
    ]

    def __init__(self): #{{{2
        configDir = QDir(QDir.homePath() + "/" + self._configDir)

        # Ensure the config directory exists
        if not QDir.home().exists(self._configDir):
            if not QDir.home().mkdir(self._configDir):
                raise IOError(errno.EIO, "Failed to create directory", str(configDir.path()).encode())

        # Check the config directory is actually a directory
        if not configDir.exists():
            raise IOError(errno.ENOTDIR, "Not a directory", str(configDir.path()).encode())

        # Check whether we're creating a new database
        new_db = False
        if not configDir.exists(self._dbFile):
            new_db = True

        # Open/Create the SQLite database file
        self._conn = QSqlDatabase.addDatabase('QSQLITE')
        dbFile = QDir.toNativeSeparators(configDir.path() + "/" + self._dbFile)
        self._conn.setDatabaseName(dbFile)

        if not self._conn.open():
            raise IOError(errno.EIO, "Failed to open database", str(dbFile).encode())

        # If this is a new database, initialise it
        if new_db:
            self._initialise()

    def _initialise(self): #{{{2
        # Create the display options table
        q = QSqlQuery(self.SQL_CREATE_DISPLAY_OPTIONS, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

        # Insert the default display options
        self._conn.transaction()
        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_INSERT_DISPLAY_OPTION):
            raise DBExceptionPrepare(q)

        for bookField in self.bookFields:
            q.bindValue(':col', QVariant(bookField[0]))
            q.bindValue(':name', QVariant(bookField[2]))

            if bookField[0] == 'title':
                q.bindValue(':disp', QVariant(1))
                q.bindValue(':sort', QVariant(2))
            elif bookField[0] == 'author_lf':
                q.bindValue(':disp', QVariant(2))
                q.bindValue(':sort', QVariant(1))
            else:
                q.bindValue(':disp', QVariant(0))
                q.bindValue(':sort', QVariant(0))

            if not q.exec_():
                raise DBExceptionExecute(q)

        self._conn.commit()

        # Create the books table
        createSql = 'create table books('

        # Build up the table definition based on the bookFields array
        for bookField in self.bookFields:
            createSql += "%s %s," % ( bookField[0], bookField[1] )

        createSql = createSql.rstrip(",") + ")"

        # Run the create command
        q = QSqlQuery(createSql, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

    def loadFile(self, filename, progress = None): #{{{2
        file = QFile(filename)

        # Check the file exists
        if not file.exists():
            raise IOException(errno.ENOENT, "File not found", filename)

        # Open the file in read-only text mode
        if not file.open(QIODevice.ReadOnly | QIODevice.Text):
            raise IOException(errno.EIO, "Cannot open file", filename)
        
        # Turn the file into a stream
        stream = QTextStream(file)

        # Load the file contents into the database
        self.loadStream(stream, progress)

    def loadStream(self, stream, progress = None): #{{{2
        # Start a transaction - provides performance benefits
        # as well as allowing rollback on errors
        self._conn.transaction()

        # Clear the old data
        q = QSqlQuery(self.SQL_DELETE_BOOKS, self._conn)
        if not q.isActive():
            self.conn.rollback()
            raise DBExceptionExecute(q)

        # Build the insert command (based on the bookFields array)
        insertSql = "insert into books values("

        for bookField in self.bookFields:
            insertSql += ":%s," % bookField[0]

        insertSql = insertSql.rstrip(',') + ")"
        q = QSqlQuery(self._conn)
        if not q.prepare(insertSql):
            self.conn.rollback()
            raise DBExceptionPrepare(q)

        # Use UTF-16 for reading the stream
        stream.setCodec('UTF-16')

        # Check the number of rows to load
        rowCount = 0
        while not stream.atEnd():
            stream.readLine()
            rowCount += 1
        stream.seek(0)

        # Initialise the progress bar
        if progress != None:
            progress.setMaximum(rowCount)
        else:
            print "Loading %d records" % rowCount

        seenHeader = False
        rowNum = 0

        # Read the export file
        while not stream.atEnd():
            line = stream.readLine()

            if progress != None:
                rowNum += 1
                progress.setValue(rowNum)

            # Skip the header line
            if not seenHeader:
                seenHeader = True
                continue

            # Split into fields
            row = line.split("\t")

            # Make sure there are the correct number of fields
            # Skip processing is there are no fields (an empty line)
            if row.isEmpty():
                continue

            for i, bookField in enumerate(self.bookFields):
                fieldName = bookField[0]
                val = None
                insertTags = False

                if row.size() == 28 and i > 21:
                    i -= 1

                if fieldName == 'isbn':
                    isbn = row.at(i)
                    while isbn.startsWith('['):
                        isbn = isbn.mid(1)
                    while isbn.endsWith(']'):
                        isbn.chop(1)
                    val = QVariant(isbn)
                elif fieldName == 'tags' and row.size() == 28:
                    val = QVariant(QVariant.String)
                elif row[i] == '(blank)' or row[i] == '':
                    val = QVariant(QVariant.String)
                else:
                    val = QVariant(row.at(i))

                q.bindValue(":%s" % fieldName, val)

            # Insert the data row into the table
            if not q.exec_():
                self.conn.rollback()
                raise DBExceptionExecute(q)

        # Commit the changes
        self._conn.commit()

    def getQuery(self, model): #{{{2
        querySql = QString('select id, ')
        colNames = []
        colWidths = []

        # Lookup the columns to display, along with the names and widths
        q = QSqlQuery(self.SQL_SELECT_DISPLAY_ORDER, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

        while q.next():
            res = q.result().data(0).toString()
            querySql += res + ', '
            colNames.append(q.result().data(1))
            colWidths.append(q.result().data(2).toInt()[0])

        querySql.chop(2)
        querySql += ' from books order by '

        # Look up the ordering for the query results
        q = QSqlQuery(self.SQL_SELECT_DISPLAY_SORT, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

        sortSeen = False
        while q.next():
            sortSeen = True
            res = q.result().data(0).toString()
            querySql += 'upper(' + res + '), '

        if sortSeen:
            querySql.chop(2)
        else:
            querySql += 'id'

        # Create the query model
        model.setQuery(querySql)

        # Ensure all data is retrieved
        while model.canFetchMore():
            model.fetchMore()

        # Set the column headers
        for i, colName in enumerate(colNames):
            model.setHeaderData(i + 1, Qt.Horizontal, colName)

        return colWidths

    def changeColumnOrder(self, col, oldPos, newPos): #{{{2
        self._conn.transaction()

        # Move the intervening columns up or down as appropriate
        if newPos > oldPos:
            updateSql = self.SQL_UPDATE_DISPLAY_ORDER_1 % (oldPos, newPos)
        else:
            updateSql = self.SQL_UPDATE_DISPLAY_ORDER_2 % (newPos, oldPos)

        q = QSqlQuery(updateSql, self._conn)
        if not q.isActive():
            self._conn.rollback()
            raise DBExceptionExecute(q)

        # Update the position of the moved column
        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_UPDATE_DISPLAY_ORDER_3):
            self._conn.rollback()
            raise DBExceptionPrepare(q)

        q.bindValue(':pos', QVariant(newPos))
        q.bindValue(':col', QVariant(col))
        if not q.exec_():
            self._conn.rollback()
            raise DBExceptionExecute(q)

        self._conn.commit()

    def changeColumnSize(self, col, newSize): #{{{2
        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_UPDATE_DISPLAY_WIDTH):
            raise DBExceptionPrepare(q)

        q.bindValue(':size', QVariant(newSize))
        q.bindValue(':pos', QVariant(col))
        if not q.exec_():
            raise DBExceptionExecute(q)

    def getNonDisplayColumns(self): #{{{2
        return self.getColumns(self.SQL_SELECT_FIELDS_NONDISPLAY)

    def getDisplayColumns(self): #{{{2
        return self.getColumns(self.SQL_SELECT_FIELDS_DISPLAY)

    def getNonSortColumns(self): #{{{2
        return self.getColumns(self.SQL_SELECT_FIELDS_NONSORT)

    def getSortColumns(self): #{{{2
        return self.getColumns(self.SQL_SELECT_FIELDS_SORT)

    def getColumns(self, query): #{{{2
        q = QSqlQuery(query, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

        cols = QStringList()
        while q.next():
            cols.append(q.result().data(0).toString())

        return cols


    def saveDisplayOrder(self, columns): #{{{2
        self._conn.transaction()

        q = QSqlQuery(self.SQL_RESET_DISPLAY_ORDER, self._conn)
        if not q.isActive():
            self._conn.rollback()
            raise DBExceptionExecute(q)

        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_UPDATE_DISPLAY_ORDER_4):
            self._conn.rollback()
            raise DBExceptionPrepare(q)

        for i, colName in enumerate(columns):
            q.bindValue(':pos', QVariant(i + 1))
            q.bindValue(':col', QVariant(colName))

            if not q.exec_():
                self._conn.rollback()
                raise DBExceptionExecute(q)

        self._conn.commit()

    def saveSortOrder(self, columns): #{{{2
        self._conn.transaction()

        q = QSqlQuery(self.SQL_RESET_SORT_ORDER, self._conn)
        if not q.isActive():
            self._conn.rollback()
            raise DBExceptionExecute(q)

        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_UPDATE_SORT_ORDER):
            self._conn.rollback()
            raise DBExceptionprepare(q)

        for i, colName in enumerate(columns):
            q.bindValue(':pos', QVariant(i + 1))
            q.bindValue(':col', QVariant(colName))

            if not q.exec_():
                self._conn.rollback()
                raise DBExceptionExecute(q)

        self._conn.commit()

    def getFieldNames(self): #{{{2
        # Get the column names
        q = QSqlQuery(self.SQL_SELECT_FIELD_NAMES, self._conn)
        if not q.isActive():
            raise DBExceptionExecute(q)

        fieldNames = {}
        while q.next():
            fieldNames[q.value(0).toString()] = q.value(1).toString()

        return fieldNames

    def getBookInfo(self, bookID): #{{{2
        # Pull back all info for the supplied ID
        q = QSqlQuery(self._conn)
        if not q.prepare(self.SQL_SELECT_BOOK_INFO):
            raise DBExceptionPrepare(q)

        q.bindValue(':id', QVariant(bookID))
        if not q.exec_():
            raise DBExceptionExecute(q)

        if not q.first():
            return

        # Pull the data into a dictionary
        recData = q.record()
        bookData = {}

        for i in range(recData.count()):
            colName = recData.fieldName(i)
            colVal  = recData.value(i).toString()

            bookData[colName] = colVal

        return bookData

# Handle running as a script {{{1
if __name__ == '__main__':
    app = QCoreApplication(sys.argv)
    db = LTDatabase()

    if len(sys.argv) > 1:
        for file in sys.argv[1:]:
            if os.path.isfile(file):
                db.loadFile(file)
