/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Sokoban.h"

#include <list>
#include <stack>
#include <vector>

#include <cassert>
#include <iostream>
Sokoban::Sokoban()
: width(0),height(0)
{
  // no code
}

size_t Sokoban::CellToPos(size_t x, size_t y) const
{
  return x+y*width;
}

void Sokoban::PosToCell(size_t pos, size_t& x, size_t& y) const
{
  x=pos % width;
  y=pos / width;
}

void Sokoban::SetPackage(size_t x, size_t y)
{
  area[CellToPos(x,y)]=area[CellToPos(x,y)] | typePackage;
}
void Sokoban::ClearPackage(size_t x, size_t y)
{
  area[CellToPos(x,y)]=area[CellToPos(x,y)] & ~typePackage;
}

void Sokoban::SetSokoban(size_t x, size_t y)
{
  area[CellToPos(x,y)]=area[CellToPos(x,y)] | typeSokoban;
}

void Sokoban::ClearSokoban(size_t x, size_t y)
{
  area[CellToPos(x,y)]=area[CellToPos(x,y)] & ~typeSokoban;
}

void Sokoban::StartUndo()
{
  undo.push(UndoEntry());
}

void Sokoban::AddCellToUndoEntry(size_t x, size_t y)
{
  UndoCell cell;

  cell.pos=CellToPos(x,y);
  cell.type=area[cell.pos];

  undo.top().cells.push_back(cell);
}

void Sokoban::SetAreaSize(size_t width, size_t height)
{
  this->width=width;
  this->height=height;

  area.resize(width*height);

  for (size_t i=0; i<area.size(); ++i) {
    area[i]=typeFloor;
  }
}

void Sokoban::Set(size_t pos, unsigned char type)
{
  area[pos]=type;
}

void Sokoban::Set(size_t x, size_t y, unsigned char type)
{
  area[CellToPos(x,y)]=type;
}

unsigned char Sokoban::Get(size_t x, size_t y) const
{
  return area[CellToPos(x,y)];
}

size_t Sokoban::GetWidth() const
{
  return width;
}
size_t Sokoban::GetHeight() const
{
  return height;
}

void Sokoban::GetSokoban(size_t &x, size_t &y) const
{
  size_t pos=0;

  while (!(area[pos] & typeSokoban)) {
    pos++;
  }

  PosToCell(pos,x,y);
}

void Sokoban::MoveSokoban(size_t dx, size_t dy)
{
  assert(IsFloor(dx,dy) || IsGoal(dx,dy));

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(dx,dy);
  AddCellToUndoEntry(x,y);

  SetSokoban(dx,dy);
  ClearSokoban(x,y);
}

bool Sokoban::IsIn(size_t x, size_t y) const
{
  return x<width && y << height;
}

bool Sokoban::IsFloor(size_t x, size_t y) const
{
  return area[CellToPos(x,y)] & typeFloor;
}

bool Sokoban::IsWall(size_t x, size_t y) const
{
  return area[CellToPos(x,y)] & typeWall;
}

bool Sokoban::IsPackage(size_t x, size_t y) const
{
  return area[CellToPos(x,y)] & typePackage;
}

bool Sokoban::IsGoal(size_t x, size_t y) const
{
  return area[CellToPos(x,y)] & typeGoal;
}

bool Sokoban::IsSokoban(size_t x, size_t y) const
{
  return area[CellToPos(x,y)] & typeSokoban;
}

bool Sokoban::IsConnectedInternal(std::vector<bool>& visited,
                                  size_t sx, size_t sy,
                                  size_t dx, size_t dy,
                                  size_t& moves) const
{
  if (moves>10) {
    return false;
  }

  if (sx==dx && sy==dy) {
    return true;
  }

  std::vector<size_t> newMoves;

  // Left
  if (sx>0 && !visited[CellToPos(sx-1,sy)]) {
    newMoves.push_back(CellToPos(sx-1,sy));
  }

  // Top
  if (sy>0 && !visited[CellToPos(sx,sy-1)]) {
    newMoves.push_back(CellToPos(sx,sy-1));
  }

  // Right
  if (sx<width-1 && !visited[CellToPos(sx+1,sy)]) {
    newMoves.push_back(CellToPos(sx+1,sy));
  }

  // Bottom
  if (sy<height-1 && !visited[CellToPos(sx,sy+1)]) {
    newMoves.push_back(CellToPos(sx,sy+1));
  }

  while (newMoves.size()>0) {
    size_t costs=0;
    std::vector<size_t>::iterator solution=newMoves.end();

    for (std::vector<size_t>::iterator iter=newMoves.begin();
         iter!=newMoves.end();
         ++iter) {
      size_t nx,ny;
      size_t currentCosts;

      PosToCell(*iter,nx,ny);

      currentCosts=std::max(nx,dx)-std::min(nx,dx)+std::max(ny,dy)-std::min(ny,dy);
      if (currentCosts>=costs) {
        costs=currentCosts;
        solution=iter;
      }
    }

    size_t nx,ny;

    PosToCell(*solution,nx,ny);

    if (!IsPackage(nx,ny) && (IsFloor(nx,ny) || IsGoal(nx,ny))) {
      moves++;
      visited[CellToPos(nx,ny)]=true;

      if (IsConnectedInternal(visited,nx,ny,dx,dy,moves)) {
        return true;
      }

      visited[CellToPos(nx,ny)]=false;
      moves--;
    }

    newMoves.erase(solution);
  }

  return false;
}

bool Sokoban::IsConnected(size_t sx, size_t sy, size_t dx, size_t dy, size_t& moves) const
{
  std::vector<bool> visited(width*height);
  bool              result;

  moves=0;
  visited.resize(width*height,false);
  visited[CellToPos(sx,sy)]=true;

  result=IsConnectedInternal(visited,sx,sy,dx,dy,moves);

  return result;
}

bool Sokoban::CanMoveLeft() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (x==0) {
    return false;
  }

  if (IsWall(x-1,y)) {
    return false;
  }

  if (IsPackage(x-1,y)) {
    if (x-1==0) {
      return false;
    }

    if (IsWall(x-2,y) || IsPackage(x-2,y)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveRight() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (x>=width-1) {
    return false;
  }

  if (IsWall(x+1,y)) {
    return false;
  }

  if (IsPackage(x+1,y)) {
    if (x+1>=width-1) {
      return false;
    }

    if (IsWall(x+2,y) || IsPackage(x+2,y)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveUp() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (y==0) {
    return false;
  }

  if (IsWall(x,y-1)) {
    return false;
  }

  if (IsPackage(x,y-1)) {
    if (y-1==0) {
      return false;
    }

    if (IsWall(x,y-2) || IsPackage(x,y-2)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveDown() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (y>=height-1) {
    return false;
  }

  if (IsWall(x,y+1)) {
    return false;
  }

  if (IsPackage(x,y+1)) {
    if (y+1>=height-1) {
      return false;
    }

    if (IsWall(x,y+2) || IsPackage(x,y+2)) {
      return false;
    }
  }

  return true;
}

void Sokoban::MoveLeft()
{
  assert(CanMoveLeft());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x-1,y);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x-1,y)) {
    AddCellToUndoEntry(x-2,y);
    SetPackage(x-2,y);
    ClearPackage(x-1,y);
  }

  SetSokoban(x-1,y);
  ClearSokoban(x,y);
}

void Sokoban::MoveRight()
{
  assert(CanMoveRight());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x+1,y);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x+1,y)) {
    AddCellToUndoEntry(x+2,y);
    SetPackage(x+2,y);
    ClearPackage(x+1,y);
  }

  SetSokoban(x+1,y);
  ClearSokoban(x,y);
}

void Sokoban::MoveUp()
{
  assert(CanMoveUp());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x,y-1);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x,y-1)) {
    AddCellToUndoEntry(x,y-2);
    SetPackage(x,y-2);
    ClearPackage(x,y-1);
  }

  SetSokoban(x,y-1);
  ClearSokoban(x,y);
}

void Sokoban::MoveDown()
{
  assert(CanMoveDown());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x,y+1);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x,y+1)) {
    AddCellToUndoEntry(x,y+2);
    SetPackage(x,y+2);
    ClearPackage(x,y+1);
  }

  SetSokoban(x,y+1);
  ClearSokoban(x,y);
}

bool Sokoban::IsFinished() const
{
  for (size_t i=0; i<area.size(); ++i) {
    size_t x,y;

    PosToCell(i,x,y);

    if (IsGoal(x,y) && !IsPackage(x,y)) {
      return false;
    }
  }

  return true;
}

void Sokoban::Undo()
{
  if (undo.empty()) {
    return;
  }

  UndoEntry entry=undo.top();

  undo.pop();

  for (size_t i=0; i<entry.cells.size(); ++i) {
    area[entry.cells[i].pos]=entry.cells[i].type;
  }
}

void Sokoban::SetLineAsText(size_t line, const std::wstring& text)
{
  for (size_t i=0; i<text.length(); ++i) {
    unsigned char type;

    switch (text[i]) {
    case L' ':
      type=Sokoban::typeFloor;
      break;
    case L'#':
      type=Sokoban::typeWall;
      break;
    case L'$':
      type=Sokoban::typeFloor|Sokoban::typePackage;
      break;
    case L'.':
      type=Sokoban::typeGoal;
      break;
    case L'@':
      type=Sokoban::typeFloor|Sokoban::typeSokoban;
      break;
    case L'+':
      type=Sokoban::typeGoal|Sokoban::typeSokoban;
      break;
    case L'*':
      type=Sokoban::typeGoal|Sokoban::typePackage;
      break;
    default:
      assert(false);
      break;
    }
    Set(line*GetWidth()+i,type);
  }
}

void Sokoban::Postprocess()
{
  // We reset every floor information besides
  for (size_t y=0; y<height; y++) {
    for (size_t x=0; x<width; x++) {
      if (IsFloor(x,y) && !(IsWall(x,y) || IsPackage(x,y) || IsGoal(x,y) ||  IsSokoban(x,y))) {
        Set(x,y,Get(x,y) & ~typeFloor);
      }
    }
  }

  /*
    Every neighbour if a floor cell that is not a wall is also a floor cell
    Repeat until no further changes are found.
    This aproach detects floor pieces that are unreachable and marks them as background.
  */

  size_t changes;
  do {
    changes=0;

    for (size_t y=0; y<height; y++) {
      for (size_t x=0; x<width; x++) {
        if (IsFloor(x,y)) {
          // Right
          if (x<width-1 && !IsFloor(x+1,y) && !IsWall(x+1,y)) {
            Set(x+1,y,Get(x+1,y) | typeFloor);
            changes++;
          }
          // Left
          if (x>0 && !IsFloor(x-1,y) && !IsWall(x-1,y)) {
            Set(x-1,y,Get(x-1,y) | typeFloor);
            changes++;
          }
          // Top
          if (y>0 && !IsFloor(x,y-1) && !IsWall(x,y-1)) {
            Set(x,y-1,Get(x,y-1) | typeFloor);
            changes++;
          }
          // Bottom
          if (y<height-1 && !IsFloor(x,y+1) && !IsWall(x,y+1)) {
            Set(x,y+1,Get(x,y+1) | typeFloor);
            changes++;
          }
        }
      }
    }
  } while (changes!=0);
}

