/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/View.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Probe.h>

#include "config.h"
#include "Configuration.h"
#include "GameArea.h"

static Lum::Def::AppInfo info;

typedef Lum::Model::StdTable<Collection*>      CollectionsModel;
typedef Lum::Base::Reference<CollectionsModel> CollectionsModelRef;

class CollectionDataProvider : public CollectionsModel::DataProvider
{
public:
  std::wstring GetString(const CollectionsModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return (*iter)->GetTitle();
    case 2:
      if ((*iter)->IsFinished()) {
        return L"finished";
      }
      else {
        return Lum::Base::NumberToWString(collection->GetCurrent()+1)+
               L"/"+
               Lum::Base::NumberToWString(collection->GetLevelCount());
      }
    default:
      assert(false);
    }
  }
};

class SelectCollection : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              okAction;
  CollectionsModelRef                list;
  Lum::Model::SingleLineSelectionRef selection;
  Collection                         *result;

public:
  SelectCollection()
  : okAction(new Lum::Model::Action),
    list(new CollectionsModel(new CollectionDataProvider())),
    selection(new Lum::Model::SingleLineSelection),
    result(NULL)
  {
    okAction->Disable();

    Observe(okAction);
    Observe(selection);

    for (size_t i=0; i<Collection::GetCollectionCount(); ++i) {
      list->Append(Collection::GetCollection(i));
    }
  }

  void PreInit()
  {
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);
    panel->SetWidth(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetWidth()*80)/100);
    panel->SetHeight(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetHeight()*80)/100);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Collection",Lum::Base::Size::stdCharWidth,25);
    headerModel->AddColumn(L"Current level",Lum::Base::Size::stdCharWidth,13);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(okAction);
    panel->Add(table);

    panel->AddSpace();
    panel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

    SetMain(panel);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      if (selection->HasSelection()) {
        result=Collection::GetCollection(selection->GetLine()-1);
      }

      Exit();
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  Collection* GetResult() const
  {
    return result;
  }
};

class SelectLevel : public Lum::Dialog
{
private:
  Collection                         *collection;
  Lum::Model::ActionRef              okAction;
  Lum::Model::StringTableRef         list;
  Lum::Model::SingleLineSelectionRef selection;
  bool                               success;
  size_t                             result;

public:
  SelectLevel(Collection *collection)
  : collection(collection),
    okAction(new Lum::Model::Action),
    list(new Lum::Model::StringTable()),
    selection(new Lum::Model::SingleLineSelection),
    success(false),
    result(0)
  {
    okAction->Disable();

    Observe(okAction);
    Observe(selection);

    for (size_t i=0; i<collection->GetLevelCount() && i<=collection->GetCurrent(); ++i) {
      list->Append(collection->GetLevel(i).GetTitle());
    }
  }

  void PreInit()
  {
    Lum::Panel       *panel;
    Lum::Table       *table;

    panel=Lum::VPanel::Create(true,true);
    panel->SetWidth(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetWidth()*80)/100);
    panel->SetHeight(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetHeight()*80)/100);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetModel(list);
    table->SetSelection(selection);
    table->SetDoubleClickAction(okAction);
    panel->Add(table);

    panel->AddSpace();
    panel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

    SetMain(panel);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      if (selection->HasSelection()) {
        result=selection->GetLine()-1;
        success=true;
      }

      Exit();
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  bool Success() const
  {
    return success;
  }

  size_t GetResult() const
  {
    return result;
  }
};

typedef Lum::Model::StdRefTable<Theme,std::vector<Theme> > ThemesModel;
typedef Lum::Base::Reference<ThemesModel>                  ThemesModelRef;

class ThemeDataProvider : public ThemesModel::DataProvider
{
public:
  std::wstring GetString(const ThemesModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return iter->name;
    case 2:
      return Lum::Base::NumberToWString(iter->boxSize);
    default:
      assert(false);
    }
  }
};

class SelectTheme : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              okAction;
  ThemesModelRef                     list;
  Lum::Model::SingleLineSelectionRef selection;
  std::wstring                       result;

public:
  SelectTheme()
  : okAction(new Lum::Model::Action),
    list(new ThemesModel(themes,new ThemeDataProvider())),
    selection(new Lum::Model::SingleLineSelection),
    result(L"")
  {
    okAction->Disable();

    Observe(okAction);
    Observe(selection);
  }

  void PreInit()
  {
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);
    panel->SetWidth(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetWidth()*80)/100);
    panel->SetHeight(Lum::Base::Size::pixel,(GetWindow()->GetParent()->GetHeight()*80)/100);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Theme name",Lum::Base::Size::stdCharWidth,25);
    headerModel->AddColumn(L"box size",Lum::Base::Size::stdCharWidth,13);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(okAction);
    panel->Add(table);

    panel->AddSpace();
    panel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

    SetMain(panel);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      if (selection->HasSelection()) {
        result=themes[selection->GetLine()-1].name;
      }

      Exit();
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  std::wstring GetResult() const
  {
    return result;
  }
};

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef categoryAction;
  Lum::Model::ActionRef levelAction;
  Lum::Model::ActionRef themeAction;
  Lum::Model::ActionRef restartAction;
  Lum::Model::ActionRef undoAction;
  Lum::Model::ActionRef aboutAction;
  Lum::Model::StringRef titleName;
  Lum::Model::StringRef levelName;
  Lum::Model::StringRef themeName;
  Lum::Model::StringRef moves;
  Lum::Model::StringRef pushes;
  GameArea              *gameArea;
  size_t                level;

public:
  MainDialog()
  : categoryAction(new Lum::Model::Action),
    levelAction(new Lum::Model::Action),
    themeAction(new Lum::Model::Action),
    restartAction(new Lum::Model::Action),
    undoAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    titleName(new Lum::Model::String()),
    levelName(new Lum::Model::String()),
    themeName(new Lum::Model::String()),
    moves(new Lum::Model::String()),
    pushes(new Lum::Model::String()),
    gameArea(new GameArea)
  {
    Observe(categoryAction);
    Observe(levelAction);
    Observe(themeAction);
    Observe(restartAction);
    Observe(undoAction);
    Observe(aboutAction);

    Observe(gameArea->GetFinishedModel());

    moves->Set(L"0");
    pushes->Set(L"0");
  }

  void PreInit()
  {
    Lum::Label       *label;
    Lum::Panel       *horiz;
    Lum::Panel       *horiz2;
    Lum::Panel       *vert;
    Lum::TextValue   *textValue;

    horiz=Lum::HPanel::Create(true,true);

    vert=Lum::VPanel::Create(true,true);

    gameArea->SetFlex(true,true);

    vert->Add(Lum::View::Create(gameArea,true,true));

    horiz->Add(vert);
    horiz->AddSpace();

    vert=Lum::VPanel::Create(false,true);

    label=Lum::Label::Create();

    horiz2=Lum::HPanel::Create(true,false);

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetMinWidth(Lum::Base::Size::stdCharWidth,5);
    textValue->SetModel(levelName);
    horiz2->Add(textValue);

    horiz2->Add(Lum::Button::Create(L"...",levelAction));

    label->AddLabel(L"Level:",horiz2);

    /*
    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetMinWidth(Lum::Base::Size::stdCharWidth,5);
    textValue->SetModel(moves);
    label->AddLabel(L"Moves:",textValue);

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetMinWidth(Lum::Base::Size::stdCharWidth,5);
    textValue->SetModel(pushes);
    label->AddLabel(L"Pushes:",textValue);*/
    label->AddLabel(L"Finished:",new Lum::LED(gameArea->GetFinishedModel()));

    vert->Add(label);
    vert->AddSpace();
    vert->Add(Lum::Button::Create(L"_Restart level",restartAction,true,false));
    vert->AddSpace();
    vert->Add(Lum::Button::Create(L"_Undo step",undoAction,true,false));

    horiz->Add(vert);

    SetMain(horiz);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->Group(L"Game")
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"Choose collection")
                                  .SetShortcut(Lum::OS::qualifierControl,L"c"),
                                  categoryAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"Choose level")
                                  .SetShortcut(Lum::OS::qualifierControl,L"l"),
                                  levelAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"Choose theme")
                                  .SetShortcut(Lum::OS::qualifierControl,L"t"),
                                  themeAction))
        ->Separator()
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"Restart level")
                                  .SetShortcut(Lum::OS::qualifierControl,L"r"),
                                  restartAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(L"Undo step")
                                  .SetShortcut(Lum::OS::qualifierControl,L"u"),
                                  undoAction))
      ->End()
      ->GroupHelp()
        //->ActionHelp()
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void SetLevel(size_t level)
  {
    assert(level<collection->GetLevelCount());
    this->level=level;

    gameArea->SetSokoban(collection->GetLevel(level).sokoban);
    levelName->Set(collection->GetLevel(level).GetTitle());
  }

  void SetCollection(Collection *collection)
  {
    assert(collection!=NULL);

    ::collection=collection;
    titleName->Set(collection->GetTitle());

    if (collection->IsFinished()) {
      SetLevel(0);
    }
    else {
      SetLevel(collection->GetCurrent());
    }
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (!LoadLevels()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Cannot find any levels!",
                              L"The program was not able to find any sokoban levels!\n"
                              L"\n"
                              L"This is normally the sign of an uncomplete or corrupt\n"
                              L"installation. Please check your installation and reinstall\n"
                              L"the program, if in doubt.\n"
                              L"\n"
                              L"You cannot use the program and thus it will now exit!");

        Exit();
        return;
      }

      LoadThemes();

      if (!LoadConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"No configuration file found!",
                              L"The program was not able to find its configuration file.\n"
                              L"\n"
                              L"If this is the first start of your program, this is\n"
                              L"OK. The program will generate a configuration file if\n"
                              L"it quits.\n"
                              L"If this is not your first use of this program, you may not\n"
                              L"have solved any levels up to now, have not correctly quit\n"
                              L"the program, or have deleted the history of solved levels!");
      }

      ChooseTheme();
      ChooseCollection();

      gameArea->EvaluateThemeName();
      themeName->Set(::themeName);
      SetCollection(::collection);
    }
    else if (model==categoryAction && categoryAction->IsFinished()) {
      SelectCollection *dialog=new SelectCollection();

      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (dialog->GetResult()!=NULL) {
        SetCollection(dialog->GetResult());
      }

      delete dialog;
    }
    else if (model==levelAction && levelAction->IsFinished()) {
      SelectLevel *dialog=new SelectLevel(collection);

      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (dialog->Success()) {
        SetLevel(dialog->GetResult());
      }

      delete dialog;
    }
    else if (model==themeAction && themeAction->IsFinished()) {
      SelectTheme *dialog=new SelectTheme();

      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (!dialog->GetResult().empty()) {
        ::themeName=dialog->GetResult();
        themeName->Set(dialog->GetResult());
        gameArea->EvaluateThemeName();
      }

      delete dialog;
    }
    else if (model==restartAction && restartAction->IsFinished()) {
      SetLevel(level);
    }
    else if (model==undoAction && undoAction->IsFinished()) {
      gameArea->Undo();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==gameArea->GetFinishedModel() && gameArea->GetFinishedModel()->Get()) {
      if (collection->GetCurrent()<level+1) {
        collection->SetCurrent(level+1);
      }
      if (!collection->IsFinished()) {
        SetLevel(level+1);
      }
    }

    Dialog::Resync(model,msg);
  }
};

int main(int argc, char* argv[])
{
  info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
  info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
  info.SetDescription(L"Move everything to its place...");
  info.SetAuthor(L"Tim Teulings");
  info.SetContact(L"Tim Teulings <tim@teulings.org>");
  info.SetCopyright(L"(c) 2006, Tim Teulings");
  info.SetLicense(L"GNU Public License");

  if (Lum::OS::prober->Open(L"PushIt",argc,argv)) {
    Lum::Dialog *window;

    window=new MainDialog();

    if (window->Open()) {
      window->SetExitAction(window->GetClosedAction());
      window->EventLoop();
      window->Close();
    }

    delete window;

    SaveConfig();
    FreeConfig();

    Lum::OS::display->Close();
  }
}
