/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Configuration.h"

#include <cassert>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <fstream>

#include <dirent.h>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

#include <Lum/Images/Loader.h>

std::vector<Collection*> Collection::collections;
std::vector<Theme>       themes;
std::wstring             themeName;
Collection               *collection=NULL;

void Level::SetTitle(const std::wstring& title)
{
  this->title=title;
}

std::wstring Level::GetTitle() const
{
  return title;
}

Collection::Collection()
 : current(0)
{
  // no code
}

void Collection::Free()
{
  collections.clear();
}

void Collection::SetTitle(const std::wstring& title)
{
  this->title=title;
}

void Collection::SetDescription(const std::wstring& description)
{
  this->description=description;
}

void Collection::SetEmail(const std::wstring& email)
{
  this->email=email;
}

void Collection::SetUrl(const std::wstring& url)
{
  this->url=url;
}

void Collection::SetCopyright(const std::wstring& copyright)
{
  this->copyright=copyright;
}

void Collection::SetCurrent(size_t current)
{
  this->current=current;
}

std::wstring Collection::GetTitle() const
{
  return title;
}

size_t Collection::GetCurrent() const
{
  return current;
}

void Collection::AddLevel(const Level& level)
{
  levels.push_back(level);
}

size_t Collection::GetLevelCount() const
{
  return levels.size();
}

Level Collection::GetLevel(const size_t pos) const
{
  return levels[pos];
}

Level& Collection::GetLevel(const size_t pos)
{
  return levels[pos];
}

bool Collection::IsFinished() const
{
  return current>=levels.size();
}

void Collection::AddCollection(Collection* collection)
{
  collections.push_back(collection);
}

size_t Collection::GetCollectionCount()
{
  return collections.size();
}

Collection* Collection::GetCollection(size_t pos)
{
  return collections[pos];
}

/*
void SetUserName(const std::wstring& name)
{
  userName=name;
}

std::wstring GetUserName()
{
  return userName;
}
*/
bool LoadConfig()
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(Lum::Base::Path::GetApplicationConfigPath());
  //const char*            tmp;

  /*
  tmp=getenv("LOGNAME");

  if (tmp!=NULL && tmp[0]!='\0') {
    userName=Lum::Base::StringToWString(tmp);
  }*/

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"PushIt") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin(); iter!=top->GetChildren().end(); ++iter) {
    if ((*iter)->GetName()==L"Solved") {
      std::wstring coll;
      size_t       current;

      if ((*iter)->GetAttribute(L"collection",coll) && (*iter)->GetAttribute(L"current",current)) {
        for (size_t i=0; i<Collection::GetCollectionCount(); ++i) {
          Collection *collection=Collection::GetCollection(i);
          bool       active=false;

          if (collection->GetTitle()==coll) {
            (*iter)->GetAttribute(L"active",active);
            if (active) {
              ::collection=collection;
            }

            collection->SetCurrent(current);

            break;
          }
        }
      }

    }
    else if ((*iter)->GetName()==L"Theme") {
      Lum::Config::Node *name;

      name=(*iter)->GetChild(L"name");

      if (name!=NULL) {
        themeName=name->GetValue();
      }
    }
  }

  delete top;

  return true;
}

bool LoadLevels()
{
  DIR             *dir;
  struct dirent   *dirEnt;
  Lum::Base::Path path,globalPath;

  path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
  path.AppendDir(L"levels");
  dir=opendir(Lum::Base::WStringToString(path.GetPath()).c_str());
  if (dir==NULL) {
    globalPath=path;
    path.SetNativeDir(L"levels");

    dir=opendir(Lum::Base::WStringToString(path.GetPath()).c_str());
    if (dir==NULL) {
      std::cerr << "Cannot open level directory as directory 'levels' or '" << Lum::Base::WStringToString(globalPath.GetPath()) << "'" << std::endl;
      return false;
    }
  }

  while ((dirEnt=readdir(dir))!=NULL) {
    std::wstring    filename=Lum::Base::StringToWString(dirEnt->d_name);

    path.SetBaseName(filename);

    if (Lum::Base::MatchWildcard(L"soklvl*.xml",filename,false)) {
      Lum::Config::Node      *top,*node;
      Lum::Config::ErrorList errors;

      top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

      if (top==NULL || top->GetName()!=L"SokobanLevels") {
        std::cerr << "Cannot load level " << Lum::Base::WStringToString(filename) << std::endl;
        continue;
      }

      Collection *collection=new Collection;

      node=top->GetChild(L"Title");
      if (node!=NULL) {
        collection->SetTitle(node->GetValue());
      }

      node=top->GetChild(L"Description");
      if (node!=NULL) {
        collection->SetDescription(node->GetValue());
      }

      node=top->GetChild(L"EMail");
      if (node!=NULL) {
        collection->SetEmail(node->GetValue());
      }

      node=top->GetChild(L"Url");
      if (node!=NULL) {
        collection->SetUrl(node->GetValue());
      }

      node=top->GetChild(L"LevelCollection");
      if (node!=NULL) {
        Lum::Config::Node *sub;

        sub=node->GetChild(L"Copyright");
        if (sub!=NULL) {
          collection->SetCopyright(sub->GetValue());
        }

        for (Lum::Config::Node::NodeList::const_iterator iter=node->GetChildren().begin(); iter!=node->GetChildren().end(); ++iter) {
          Lum::Config::Node* sub=*iter;

          if (sub->GetName()==L"Level") {
            Level             level;
            Lum::Config::Node *subsub;
            size_t             width,height;

            subsub=sub->GetChild(L"Id");
            if (subsub!=NULL) {
              level.SetTitle(subsub->GetValue());
            }

            if (!sub->GetAttribute(L"Width",width) || sub->GetAttribute(L"Height",height) ||
                width==0 || height==0) {
              continue;
            }

            level.sokoban.SetAreaSize(width,height);

            size_t line=0;
            for (Lum::Config::Node::NodeList::const_iterator iter=sub->GetChildren().begin(); iter!=sub->GetChildren().end(); ++iter) {
              if ((*iter)->GetName()==L"L") {
                std::wstring fields=(*iter)->GetValue();

                level.sokoban.SetLineAsText(line,fields);

                line++;
              }
            }
            level.sokoban.Postprocess();

            collection->AddLevel(level);
          }
        }

      }

      if (collection->GetLevelCount()>0) {
        Collection::AddCollection(collection);
      }
      else {
        delete collection;
      }

      delete top;
    }
    else if (Lum::Base::MatchWildcard(L"*.xsb",filename,false) ||
             Lum::Base::MatchWildcard(L"*.sok",filename,false)) {
      std::ifstream             file;
      std::string               tmp;
      std::wstring              line;
      std::vector<std::wstring> gameBuffer;
      size_t                    games=0;

      // We reserve space for 100 lines
      gameBuffer.reserve(100);

      file.open(::Lum::Base::WStringToString(path.GetPath()).c_str(),std::ios::in);

      if (!file) {
        std::cerr << "Cannot open file '" << Lum::Base::WStringToString(filename) << "'!" << std::endl;
        continue;
      }

      Collection *collection=new Collection;

      collection->SetTitle(filename.substr(0,filename.size()-4));

      while (true) {

        if (!std::getline(file,tmp)) {
          line.clear();
        }
        else {
          line=Lum::Base::StringToWString(tmp);
        }

        //std::cerr << line << std::endl;

        // Skip space
        size_t i=0;
        while (i<line.length() && line[i]==' ') {
          i++;
        }

        if (i<line.length() && line[i]=='#')  {
          // This is line of a game
          gameBuffer.push_back(line);
        }
        else {
          // This is a comment or separator line
          if (gameBuffer.size()>0) {
            // This is a comment or separator line after the map
            Level  level;
            size_t width=0;

            for (size_t j=0; j<gameBuffer.size(); j++) {
              width=std::max(width,gameBuffer[j].length());
            }

            level.sokoban.SetAreaSize(width,gameBuffer.size());

            for (size_t j=0; j<gameBuffer.size(); j++) {
              level.sokoban.SetLineAsText(j,gameBuffer[j]);
            }
            level.sokoban.Postprocess();

            level.SetTitle(Lum::Base::NumberToWString(games+1));

            collection->AddLevel(level);
            gameBuffer.clear();

            games++;
          }

          // Preprocess comment line

          // Skip leading colons
          size_t j=0;
          while (j<line.length() && line[j]==L';') {
            ++j;
          }

          if (j>0) {
            line.erase(0,j);
          }

          size_t       pos;
          std::wstring keyword,rest;

          pos=line.find(L':');
          if (pos!=std::wstring::npos) {
            for (size_t j=0; j<pos; ++j) {
              wchar_t tmp=std::toupper(line[j]);
              keyword.append(1,tmp);
            }

            ++pos;
            while (pos<line.length() && line[pos]==L' ') {
              ++pos;
            }

            if (pos<line.length()) {
              rest=line.substr(pos);
            }
          }

          if (keyword==L"TITLE") {
            if (games==0) {
              collection->SetTitle(rest);
            }
            else {
              collection->GetLevel(collection->GetLevelCount()-1).SetTitle(rest);
            }
          }
          else if (keyword==L"EMAIL") {
            if (games==0) {
              collection->SetEmail(rest);
            }
            else {
              // TODO
            }
          }
          else if (keyword==L"COPYRIGHT") {
            if (games==0) {
              collection->SetCopyright(rest);
            }
            else {
              // TODO
            }
          }
          else if (keyword==L"HOMEPAGE" || keyword==L"URL" || keyword==L"WEBSITE") {
            if (games==0) {
              collection->SetUrl(rest);
            }
            else {
              // TODO
            }
          }
        }

        if (!file) {
          break;
        }
      };

      if (collection->GetLevelCount()>0) {
        Collection::AddCollection(collection);
      }
      else {
        delete collection;
      }
    }

  }

  closedir(dir);

  return Collection::GetCollectionCount()>0;
}

bool SaveConfig()
{
  Lum::Config::Node *top;
  Lum::Config::Node *node;
  Lum::Base::Path   path(Lum::Base::Path::GetApplicationConfigPath());
  bool              res;

  top=new Lum::Config::Node();
  top->SetName(L"PushIt");

  node=new Lum::Config::Node();

  node->SetName(L"Theme");
  node->SetAttribute(L"name",themeName);

  top->Add(node);

  for (size_t i=0; i<Collection::GetCollectionCount(); ++i) {
    Collection *collection=Collection::GetCollection(i);

    node=new Lum::Config::Node;

    node->SetName(L"Solved");
    node->SetAttribute(L"collection",collection->GetTitle());
    node->SetAttribute(L"current",collection->GetCurrent());
    if (collection==::collection) {
      node->SetAttribute(L"active",true);
    }

    top->Add(node);
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}

static const wchar_t* imageNames[] = {
  L"floor.png",
  L"goal.png",
  L"package.png",
  L"sokoban.png",
  L"wall.png",
  L"left.png",
  L"up.png",
  L"right.png",
  L"down.png"
};

bool LoadThemes()
{
  Theme theme;

  theme.name=L"Internal small";
  theme.boxSize=16;
  themes.push_back(theme);

  theme.name=L"Internal Normal";
  theme.boxSize=24;
  themes.push_back(theme);

  theme.name=L"Internal Big";
  theme.boxSize=32;
  themes.push_back(theme);

  // Find sub directories in the global or local theme directory that contain
  // at least one of the required images

  if (Lum::OS::display->GetType()==Lum::OS::Display::typeGraphical) {
    DIR             *dir;
    struct dirent   *dirEnt;
    Lum::Base::Path path;

    path.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
    path.AppendDir(L"themes");
    dir=opendir(Lum::Base::WStringToString(path.GetPath()).c_str());
    if (dir==NULL) {
      path.SetNativeDir(L"themes");

      dir=opendir(Lum::Base::WStringToString(path.GetPath()).c_str());
      if (dir==NULL) {
        std::cerr << "Cannot open theme directory as directory 'themes' or '" << Lum::Base::WStringToString(path.GetPath()) << "'" << std::endl;
        return false;
      }
    }

    while ((dirEnt=readdir(dir))!=NULL) {
      std::wstring    dirname=Lum::Base::StringToWString(dirEnt->d_name);
      Lum::Base::Path dirPath(path);

      dirPath.AppendDir(dirname);

      if (strcmp(dirEnt->d_name,".")!=0 &&
          strcmp(dirEnt->d_name,"..")!=0 &&
          dirEnt->d_name[0]!='.' &&
          dirPath.IsDir()) {
        Theme  theme;
        size_t imageCount=0;

        theme.name=Lum::Base::StringToWString(dirEnt->d_name);

        for (size_t i=0; i<countImage; i++) {
          Lum::Base::Path path;

          dirPath.SetBaseName(imageNames[i]);

          theme.images[i]=Lum::Images::Factory::factory->CreateImage();

          if (Lum::Images::loader->Load(dirPath.GetPath(),theme.images[i])) {
            imageCount++;
            theme.boxSize=theme.images[i]->GetWidth();
          }
          else {
            std::cerr << "Cannot load image '" << Lum::Base::WStringToString(dirPath.GetPath()) << "'" << std::endl;
            theme.images[i]=NULL;
          }
        }

        if (imageCount==0) {
          std::cerr << "No theming image found => Skipping theme" << std::endl;
          continue;
        }

        themes.push_back(theme);
      }
    }
  }

  return true;
}

void ChooseTheme()
{
  if (themeName.empty()) {
    for (size_t i=0; i<themes.size(); i++) {
      if (themes[i].name==L"Normal") {
        themeName=themes[i].name;
        return;
      }
    }
  }

  if (themeName.empty()) {
    themeName=L"Internal Normal";
  }
}

void ChooseCollection()
{
  if (::collection==NULL) {
    for (size_t i=0; i<Collection::GetCollectionCount(); i++) {
      if (Collection::GetCollection(i)->GetTitle()==L"Classic levels") {
        ::collection=Collection::GetCollection(i);
        return;
      }
    }
  }

  if (::collection==NULL) {
    ::collection=Collection::GetCollection(0);
  }
}

void FreeConfig()
{
  Collection::Free();
}
