/*
* ==============================================================================
*  Name        : game.h
*  Part of     : OpenC / OpenCOpenglEx
*  Interface   : 
*  Description : Contains the class declaration of CGame.
*  Version     : 
*
*  Copyright (c) 2007 Nokia Corporation.
*  This material, including documentation and any related 
*  computer programs, is protected by copyright controlled by 
*  Nokia Corporation.
* ==============================================================================
*/

#ifndef __GAME_CLASS__
#define __GAME_CLASS__

#include <e32base.h> // for CBase
#ifdef __NANOGL__
#include <gl/gl.h>
#else
#ifdef SOFT_LINKAGE
#pragma softfp_linkage
#endif
#include <gles/gl.h> // for opengl es types 
#ifdef SOFT_LINKAGE
#pragma no_softfp_linkage
#endif
#endif
#include <e32keys.h> // for TStdScanCode
//#include "padPlugin.h"
// CONSTANTS

// The number of microseconds that can be spent on one frame.
// 10000000 microseconds = 1 second
// Frame rate = 25 fps.
const TInt timeForOneFrame = 1000000 / 25;

// Vertice coordinates for the cube. 
static const GLbyte vertices[8 * 3] =
     {
     -1,  1,  1,
      1,  1,  1,
      1, -1,  1,
     -1, -1,  1,

     -1,  1, -1,
      1,  1, -1,
      1, -1, -1,
     -1, -1, -1
     };

// Colors for vertices (Red, Green, Blue, Alpha). 
static const GLubyte colors[8 * 4] =
    {
    0  ,255,  0,255,
    0  ,  0,255,255,
    0  ,255,  0,255,
    255,  0,  0,255,

    0  ,  0,255,255,
    255,  0,  0,255,
    0  ,  0,255,255,
    0  ,255,  0,255
    };

// Indices for drawing the triangles.
static const GLubyte triangles[12 * 3] =
    {
    // front 
    1,0,3,
    1,3,2,

    // right 
    2,6,5,
    2,5,1,
    
    // back 
    7,4,5,
    7,5,6,
    
    // left 
    0,4,7,
    0,7,3,
    
    // top 
    5,4,0,
    5,0,1,
    
    // bottom 
    3,7,6,
    3,6,2
    };
    
// MACROS
// Used for specifying the viewing frustum.
#define FRUSTUM_LEFT   -1.f     //left vertical clipping plane
#define FRUSTUM_RIGHT   1.f     //right vertical clipping plane
#define FRUSTUM_BOTTOM -1.f     //bottom horizontal clipping plane
#define FRUSTUM_TOP     1.f     //top horizontal clipping plane
#define FRUSTUM_NEAR    3.f     //near depth clipping plane
#define FRUSTUM_FAR  1000.f     //far depth clipping plane

// CLASS DECLARATION

/**
 * Class that does the actual OpenGL ES rendering.
 */
class CGame : public CBase
{
public: // Static Constructor and destructor.
    /**
     * Factory method for creating a new CGame object.
     */
    static CGame* NewLC();

    /**
     * Destructor.
     */    
    ~CGame();
protected: // Constructors
    /**
     * Standard constructor that must never Leave.
     */ 
    CGame();
    
    /**
     * Second phase contructor. 
     */    
    void ConstructL();
public:
    /**
     * Intializes the Game.
     *
     * @param aScreenWidth Width of the screen.
     * @param aScreenHeight Height of the screen.
     */
    void Initialize( TUint aScreenWidth, TUint aScreenHeight );
    
    /**
     * Renders the next frame using opengl es calls.
     *
     * @param aDurationSinceLastVisit The time (in microseconds) since
     *        the function was last visited.
     */    
    TBool RenderFrame( TInt64 aDurationSinceLastVisit );
    
    /**
     * Cleanups any resources.
     */        
    void Cleanup();
    
    /**
     * Handles a Key up / down event.
     *
     * @param aKey The TStdScanCode of the key.
     * @param aIsKeyUp ETrue if the Key is released, else EFalse.
     */    
    void HandleKeyEvent( TUint32 aKey, TBool aIsKeyUp );
    
    /**
     * Handle screen size changes.
     *
     * @param aScreenWidth Width of the screen.
     * @param aScreenHeight Height of the screen.
     */
    void SetScreenSize( TUint aScreenWidth, TUint aScreenHeight );

    /**
     * Get the current frame number.
     *
     */
    TUint GetCurrentFrame() const
    {
        return iFrame;
    }
    
public: // Data
    /** Width of the screen */
    TInt iScreenWidth;    
    
    /** Height of the screen */
    TInt iScreenHeight;
    
    /** Flags if the game needs to exit */
    TBool iGameOver;
    
    /** The z co-ordinate of the cube */
    TInt iZPos;
    
    /** Flags if the up key is pressed */
    TBool iIsUpPressed;
    
    /** Flags if the down key is pressed */
    TBool iIsDownPressed;
    
    /** The current frame being rendered */
    TUint iFrame;    
}; // CGame

#endif //__GAME_CLASS__
