/*
 *  presence notifier statusbar plugin.
 *  Copyright (C) 2011 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libosso-abook/osso-abook.h>
#include <libosso-abook/osso-abook-roster.h>
#include <libosso-abook/osso-abook-roster-manager.h>
#include <libosso-abook/osso-abook-contact.h>
#include <libosso-abook/osso-abook-aggregator.h>
#include <libosso-abook/osso-abook-account-manager.h>
#include <gconf/gconf-client.h>

#include <libmcclient/mc-account.h>
#include <libmcclient/mc-profile.h>
#include <libebook/e-book-view.h>
#include <libebook/e-book-query.h>
#include "presence-notifier-sp.h"

#define PRESENCE_NOTIFIER_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
										      TYPE_PRESENCE_NOTIFIER_STATUS_PLUGIN, PresenceNotifierStatusPluginPrivate))

#define GC_ROOT "/apps/maemo/presence-notifier"
#define NOTIFIER_DISABLED_GCONF_PATH GC_ROOT "/disabled"
#define NOTIFIER_NOTIFY_ONLINE_ONLY_GCONF_PATH GC_ROOT "/online_only"
#define NOTIFIER_NOTIFY_STATUS_MESSAGE_GCONF_PATH GC_ROOT "/status_message"
#define NOTIFIER_AUTO_CLOSE_TIME_GCONF_PATH GC_ROOT "/notification_timeout"
#define NOTIFIER_AUTO_CLOSE_NOTIFICATION_GCONF_PATH GC_ROOT "/auto_notification_close"

struct _PresenceNotifierStatusPluginPrivate
{
  gboolean disabled;
  guint gconf_notify_handler;
  gboolean notify_message_change;
  gboolean notify_status_change;
  gboolean notify_online_status_only;
  int notification_time_out;
  GHashTable* contacts;
  OssoABookAggregator* abook_aggregator;
  OssoABookAccountManager* abook_account_manager;
  DBusGConnection* dbus_connection;
};

typedef struct _presence_data_t
{
  gchar* status_message;
  gchar* status;
  TpConnectionPresenceType presence_type;
}presence_data_t;

HD_DEFINE_PLUGIN_MODULE(PresenceNotifierStatusPlugin, presence_notifier_status_plugin, HD_TYPE_STATUS_MENU_ITEM);

static void
_print_presence_data(presence_data_t* presence_data)
{
  g_warning("presence data\n");
  g_warning("message %s\n", presence_data->status_message);
  g_warning("type %d\n", presence_data->presence_type);
}

static void
_print_contact_presence(OssoABookPresence* presence)
{
  g_warning("contact presence\n");
  g_warning("message %s\n", osso_abook_presence_get_presence_status_message(presence));
  g_warning("type %d\n", osso_abook_presence_get_presence_type(presence));
}

static void presence_data_free(presence_data_t* data)
{
  g_free(data->status_message);
  g_free(data->status);
  g_free(data);
}

static void
_show_presence_message(PresenceNotifierStatusPlugin* plugin,
		       OssoABookPresence* presence)
{
  const char* status_message = osso_abook_presence_get_presence_status_message(presence);
  if(!status_message)
    status_message = osso_abook_presence_get_display_status(presence);
  
  gchar* contact_name = g_strdup_printf("%s",osso_abook_contact_get_name(OSSO_ABOOK_CONTACT(presence)));
  /*,
    osso_abook_contact_get_bound_name(OSSO_ABOOK_CONTACT(presence))
  */
  
  DBusGProxy* dbus_proxy = NULL;
  DBusGConnection* dbus_conn = NULL;

  if(plugin->priv->dbus_connection)
  {
    dbus_proxy = dbus_g_proxy_new_for_name(plugin->priv->dbus_connection,
					   "org.freedesktop.Notifications",
					   "/org/freedesktop/Notifications",
					   "org.freedesktop.Notifications");
    dbus_g_proxy_call_no_reply(dbus_proxy, 
			       "Notify", 
			       G_TYPE_STRING, "Presence Notifier",
			       G_TYPE_UINT, 0,
			       G_TYPE_STRING, osso_abook_presence_get_icon_name(presence),
			       G_TYPE_STRING, status_message,
			       G_TYPE_STRING, contact_name,
			       G_TYPE_STRV, 0,
			       dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_VALUE), NULL,
			       G_TYPE_INT, plugin->priv->notification_time_out*1000,
			       G_TYPE_INVALID,
			       G_TYPE_INT,
			       G_TYPE_INVALID);
    g_object_unref(dbus_proxy);
    g_free(contact_name);
  }
}

static void
_notify_on_change(PresenceNotifierStatusPlugin* plugin,
		  OssoABookPresence* presence,
		  presence_data_t* presence_data)
{
  if(!presence_data)
    return;

  TpConnectionPresenceType contact_status_type = osso_abook_presence_get_presence_type(presence);
  TpConnectionStatus connection_status = mc_account_get_connection_status(osso_abook_contact_get_account(OSSO_ABOOK_CONTACT(presence)));

  if(contact_status_type == TP_CONNECTION_PRESENCE_TYPE_UNKNOWN ||
     connection_status != TP_CONNECTION_STATUS_CONNECTED)
    return;

  gchar* contact_status = g_strdup(osso_abook_presence_get_presence_status(presence));
  gchar* contact_status_message = g_strdup(osso_abook_presence_get_presence_status_message(presence));
  gboolean message_changed = g_strcmp0(presence_data->status_message, contact_status_message) != 0;
  gboolean status_changed = g_strcmp0(presence_data->status, contact_status) != 0 ||
    presence_data->presence_type != contact_status_type;
  if(message_changed || status_changed)
  {
    if(presence_data->status_message != NULL)
      g_free(presence_data->status_message);
    if(presence_data->status != NULL)
      g_free(presence_data->status);

    presence_data->status_message = contact_status_message;
    presence_data->status = contact_status;
    TpConnectionPresenceType prior_status_type = presence_data->presence_type;
    
    presence_data->presence_type = contact_status_type;
    
    if(!plugin->priv->disabled &&
       (
	 (plugin->priv->notify_online_status_only &&
	  contact_status_type == TP_CONNECTION_PRESENCE_TYPE_AVAILABLE &&
	  (message_changed && plugin->priv->notify_message_change ||
	   status_changed)) ||
	 (!plugin->priv->notify_online_status_only && 
	  (status_changed || (plugin->priv->notify_message_change && message_changed))
	   )
	 ) &&
       prior_status_type != TP_CONNECTION_PRESENCE_TYPE_UNKNOWN
      )
    {
      _show_presence_message(plugin, presence);
    }
  }
  else
  {
    g_free(contact_status);
    g_free(contact_status_message);
  }
}

static void
_roster_contact_presence_status_message(OssoABookPresence* presence,
					GParamSpec* psp,
					gpointer user)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(user);
  /* g_warning("notify status message\n"); */
  /* g_warning("contact %s\n", osso_abook_contact_get_name(OSSO_ABOOK_CONTACT(presence))); */
  /* _print_contact_presence(presence); */
  presence_data_t* contact_presence = g_hash_table_lookup(plugin->priv->contacts, 
							  osso_abook_contact_get_persistent_uid(OSSO_ABOOK_CONTACT(presence)));
  _print_presence_data(contact_presence);
  _notify_on_change(plugin, presence, contact_presence);
}

static void
_roster_contact_presence_type(OssoABookPresence* presence,
			      GParamSpec* psp,
			      gpointer user)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(user);
  /* g_warning("notify presence type\n"); */
  /* g_warning("contact %s\n", osso_abook_contact_get_name(OSSO_ABOOK_CONTACT(presence))); */
  /* _print_contact_presence(presence); */

  presence_data_t* contact_presence = g_hash_table_lookup(plugin->priv->contacts, 
							  osso_abook_contact_get_persistent_uid(OSSO_ABOOK_CONTACT(presence)));
  _print_presence_data(contact_presence);

  _notify_on_change(plugin, presence, contact_presence);
}
static void
_roster_contact_presence_status(OssoABookPresence* presence,
				GParamSpec* psp,
				gpointer user)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(user);
  /* g_warning("notify status\n"); */
  /* g_warning("contact %s\n", osso_abook_contact_get_name(OSSO_ABOOK_CONTACT(presence))); */
  /* _print_contact_presence(presence); */
  presence_data_t* contact_presence = g_hash_table_lookup(plugin->priv->contacts, 
							    osso_abook_contact_get_persistent_uid(OSSO_ABOOK_CONTACT(presence)));
  _print_presence_data(contact_presence);
  _notify_on_change(plugin, presence, contact_presence);
}

static void
account_ready_cb(OssoABookAccountManager *manager,
		 const GError *error,
		 gpointer data)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(data);
}

static presence_data_t*
presence_data_for_contact(OssoABookContact* contact)
{
  presence_data_t* presence_data = g_new0(presence_data_t, 1);
  OssoABookPresence* presence = OSSO_ABOOK_PRESENCE(contact);
  const gchar* status_message = osso_abook_presence_get_presence_status_message(presence);
  const gchar* status = osso_abook_presence_get_presence_status(presence);
  TpConnectionPresenceType presence_type =  osso_abook_presence_get_presence_type(presence);
  presence_data->status_message = g_strdup(status_message);
  presence_data->status = g_strdup(status);
  presence_data->presence_type = presence_type;
  return presence_data;
}

static void
_list_roster_contacts(PresenceNotifierStatusPlugin* plugin)
{
  GList* roster_contacts = osso_abook_aggregator_list_roster_contacts(plugin->priv->abook_aggregator);
  GList* roster;
  for(roster = roster_contacts; roster; roster = roster->next)
  {
    OssoABookContact* roster_contact = (OssoABookContact*) roster->data;
    gchar* name = g_strdup(osso_abook_contact_get_persistent_uid(roster_contact));
    presence_data_t* presence_data = presence_data_for_contact(roster_contact);
    /* g_warning("list roster\n"); */
    /* g_warning("contact %s\n", osso_abook_contact_get_name(roster_contact)); */
    /* _print_contact_presence(OSSO_ABOOK_PRESENCE(roster_contact)); */
    /* _print_presence_data(presence_data); */

    g_hash_table_insert(plugin->priv->contacts,
			name,
			presence_data);
    g_signal_connect(OSSO_ABOOK_PRESENCE(roster_contact), "notify::presence-status",
		     G_CALLBACK(_roster_contact_presence_status), plugin);
    g_signal_connect(OSSO_ABOOK_PRESENCE(roster_contact), "notify::presence-type",
		     G_CALLBACK(_roster_contact_presence_type), plugin);
    g_signal_connect(OSSO_ABOOK_PRESENCE(roster_contact), "notify::presence-status-message", 
		     G_CALLBACK(_roster_contact_presence_status_message), plugin); 
  }
  g_list_free(roster_contacts);
}

static void
aggregator_ready_cb(OssoABookWaitable *waitable,
		    const GError *error,
		    gpointer data)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(data);
  _list_roster_contacts(plugin);
}

static void
_read_notifier_setting(PresenceNotifierStatusPlugin* plugin)
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return;
  plugin->priv->disabled = gconf_client_get_bool(client, NOTIFIER_DISABLED_GCONF_PATH, NULL);
  plugin->priv->notify_online_status_only = gconf_client_get_bool(client, NOTIFIER_NOTIFY_ONLINE_ONLY_GCONF_PATH, NULL);
  plugin->priv->notify_message_change = gconf_client_get_bool(client, NOTIFIER_NOTIFY_STATUS_MESSAGE_GCONF_PATH, NULL);
  plugin->priv->notification_time_out = gconf_client_get_int(client, NOTIFIER_AUTO_CLOSE_TIME_GCONF_PATH, NULL);
  if(gconf_client_get_int(client, NOTIFIER_AUTO_CLOSE_NOTIFICATION_GCONF_PATH, NULL) &&
     plugin->priv->notification_time_out == 0)
    plugin->priv->notification_time_out = 6;
    
  g_object_unref(client);
}

static void
_gconf_notifier_disabled_changed(GConfClient* client,
				 guint cnxn_id,
				 GConfEntry* entry,
				 gpointer user_data)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(user_data);
  _read_notifier_setting(plugin);
}

static void
_register_gconf_changes(PresenceNotifierStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return;
  }
  
  gconf_client_add_dir(client,
		       GC_ROOT,
		       GCONF_CLIENT_PRELOAD_NONE,
		       &error);
  if(error != NULL)
  {
    g_warning("can not listen on gconf client changes %s\n", error->message);
    g_error_free(error);
    error = NULL;
    g_object_unref(client);
    return;
  }
  plugin->priv->gconf_notify_handler = 
    gconf_client_notify_add(client, GC_ROOT, _gconf_notifier_disabled_changed, plugin, NULL, &error);
  if(error != NULL)
  {
    g_warning("can not add notify on gconf client %s\n", error->message);
    g_error_free(error);
    error = NULL;
    g_object_unref(client);
    return;
  }
  g_object_unref(client);
}

static void
_unregister_gconf_changes(PresenceNotifierStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  if(GCONF_IS_CLIENT(client))
  {
    gconf_client_notify_remove(client, 
			       plugin->priv->gconf_notify_handler);
    plugin->priv->gconf_notify_handler = 0;
    gconf_client_remove_dir(client, GC_ROOT,
			    NULL);
    g_object_unref(client);
  }
}

static void
presence_notifier_status_plugin_init(PresenceNotifierStatusPlugin* plugin)
{
  plugin->priv = PRESENCE_NOTIFIER_STATUS_PLUGIN_GET_PRIVATE(plugin);
  plugin->priv->gconf_notify_handler = 0;
  plugin->priv->notify_online_status_only = TRUE;
  plugin->priv->notify_status_change = TRUE;
  plugin->priv->notify_message_change = TRUE;
  plugin->priv->notification_time_out = 6;
  plugin->priv->contacts = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, (GDestroyNotify)presence_data_free);
  plugin->priv->dbus_connection = hd_status_plugin_item_get_dbus_g_connection(HD_STATUS_PLUGIN_ITEM(&plugin->parent),
									      DBUS_BUS_SESSION, 
									      NULL);
  plugin->priv->abook_account_manager = osso_abook_account_manager_new();
  EBookQuery* ebook_query = e_book_query_from_string("(not (is_vcard \"X-TELEPATHY-BLOCKED\"  \"yes\"))");
  osso_abook_account_manager_set_roster_query(plugin->priv->abook_account_manager, ebook_query);
  plugin->priv->abook_aggregator = OSSO_ABOOK_AGGREGATOR(osso_abook_aggregator_new_with_query(NULL, 
											      ebook_query, 
											      NULL, 
											      0, 
											      NULL));
  osso_abook_aggregator_set_roster_manager(OSSO_ABOOK_AGGREGATOR(plugin->priv->abook_aggregator),
					   OSSO_ABOOK_ROSTER_MANAGER(plugin->priv->abook_account_manager));
  e_book_query_unref(ebook_query);

  osso_abook_roster_start(OSSO_ABOOK_ROSTER(plugin->priv->abook_aggregator));
  osso_abook_account_manager_call_when_ready(plugin->priv->abook_account_manager,
					     account_ready_cb, plugin, NULL);
  osso_abook_waitable_call_when_ready(OSSO_ABOOK_WAITABLE(plugin->priv->abook_aggregator),
				      aggregator_ready_cb, plugin, NULL);
  _register_gconf_changes(plugin);
  _read_notifier_setting(plugin);
}

static void
presence_notifier_status_plugin_finalize(GObject* object)
{
  PresenceNotifierStatusPlugin* plugin = PRESENCE_NOTIFIER_STATUS_PLUGIN(object);
  if(plugin->priv->contacts)
    g_hash_table_destroy(plugin->priv->contacts);
  if(plugin->priv->gconf_notify_handler != 0)
    _unregister_gconf_changes(plugin);
  if(plugin->priv->abook_account_manager)
  {
    g_object_unref(plugin->priv->abook_account_manager);
  }
  if(plugin->priv->abook_aggregator)
  {
    g_object_unref(plugin->priv->abook_aggregator);
  }
  G_OBJECT_CLASS(presence_notifier_status_plugin_parent_class)->finalize(object);
}

static void
presence_notifier_status_plugin_class_finalize(PresenceNotifierStatusPluginClass* klass)
{
}

static void
presence_notifier_status_plugin_class_init(PresenceNotifierStatusPluginClass* klass)
{
  g_type_class_add_private(klass, sizeof(PresenceNotifierStatusPluginPrivate));
  G_OBJECT_CLASS(klass)->finalize = (GObjectFinalizeFunc)presence_notifier_status_plugin_finalize;
}
