/*
 * Pidgin Extended Preferences Plugin
 *
 * Copyright 2004-05 Kevin Stange <extprefs@simguy.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define PURPLE_PLUGINS

#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "plugin.h"
#include "version.h"
#include "gtkplugin.h"
#include "gtkdocklet.h"
#include "gtkblist.h"
#include "pidginstock.h"
#include "core.h"

#include "dbus-service-info.h"

static DBusConnection *dbus_connection(void);

static void
pidgin_status_menu_set_tooltip(gchar *str)
{
	DBusConnection *connection = dbus_connection();
	DBusMessage *message = dbus_message_new_method_call(
		PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, 
		PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, 
		NULL,
		PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_SET_TOOLTIP);
	DBusMessage *reply;

	if (message) {
		dbus_message_append_args(message, DBUS_TYPE_STRING, &str, DBUS_TYPE_INVALID);
		g_print("|--- *** pidgin_status_menu_set_tooltip: Making dbus call\n");
		reply = dbus_connection_send_with_reply_and_block(connection, message, -1, NULL);
		dbus_message_unref(message);
	}

	if (reply)
		dbus_message_unref(reply);
}

static void
pidgin_status_menu_plugin_update_icon(GdkPixbuf *pb)
{
	gchar *buffer;
	gsize buffer_size;

	if (gdk_pixbuf_save_to_buffer(pb, &buffer, &buffer_size, "png", NULL, NULL)) {
		DBusConnection *conn = dbus_connection();
		DBusMessage *message = dbus_message_new_method_call(
			PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, 
			PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, 
			NULL,
			PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_UPDATE_ICON);
		DBusMessage *reply;

		dbus_message_append_args(message, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &buffer, buffer_size, DBUS_TYPE_INVALID);
		g_print("|--- *** pidgin_status_menu_plugin_update_icon: Making dbus call: buffer size is %d\n", buffer_size);
		reply = dbus_connection_send_with_reply_and_block(conn, message, -1, NULL);
		dbus_message_unref(message);
		if (reply)
			dbus_message_unref(reply);
	}
}

static void
pidgin_status_menu_plugin_disconnect()
{
	DBusConnection *connection = dbus_connection();
	DBusMessage *message = dbus_message_new_method_call(
		PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, 
		PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, 
		NULL,
		PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_DISCONNECT);
	DBusMessage *reply;

	if (message) {
		reply = dbus_connection_send_with_reply_and_block(connection, message, -1, NULL);
		dbus_message_unref(message);
	}

	if (reply)
		dbus_message_unref(reply);
}

static void
pidgin_maemo_docklet_ui_ops_create()
{
	g_print("|--- *** pidgin_maemo_docklet_ui_ops_create\n");
}

static void
pidgin_maemo_docklet_ui_ops_destroy()
{
	g_print("|--- *** pidgin_maemo_docklet_ui_ops_destroy\n");
	pidgin_status_menu_plugin_disconnect();
}

static GdkPixbuf *
stock_icon_to_pixbuf(const gchar *stock_id, GtkIconSize icon_size)
{
	GdkPixbuf *pb = NULL;
	PidginBuddyList *blist = pidgin_blist_get_default_gtk_blist();

	g_print("|--- *** stock_icon_to_pixbuf: blist = %p, blist->window = %p\n", blist, blist ? blist->window : (gpointer)0xdeadbeef);

	if (blist) {
		if (blist->window)
			pb = gtk_widget_render_icon(blist->window, stock_id, icon_size, NULL);
	}

	return pb;
}

static void
pidgin_maemo_docklet_ui_ops_update_icon(PurpleStatusPrimitive status, gboolean connecting, gboolean pending)
{
	const gchar *icon_name = NULL;

	switch (status) {
		case PURPLE_STATUS_OFFLINE:
			icon_name = PIDGIN_STOCK_TRAY_OFFLINE;
			break;
		case PURPLE_STATUS_AWAY:
			icon_name = PIDGIN_STOCK_TRAY_AWAY;
			break;
		case PURPLE_STATUS_UNAVAILABLE:
			icon_name = PIDGIN_STOCK_TRAY_BUSY;
			break;
		case PURPLE_STATUS_EXTENDED_AWAY:
			icon_name = PIDGIN_STOCK_TRAY_XA;
			break;
		case PURPLE_STATUS_INVISIBLE:
			icon_name = PIDGIN_STOCK_TRAY_INVISIBLE;
			break;
		default:
			icon_name = PIDGIN_STOCK_TRAY_AVAILABLE;
			break;
	}

	if (pending)
		icon_name = PIDGIN_STOCK_TRAY_PENDING;
	if (connecting)
		icon_name = PIDGIN_STOCK_TRAY_CONNECT;

	if(icon_name) {
		GdkPixbuf *pb;

		pb = stock_icon_to_pixbuf(icon_name, gtk_icon_size_from_name(PIDGIN_ICON_SIZE_TANGO_LARGE));
		g_print("|--- *** pidgin_maemo_docklet_ui_ops_update_icon: pixbuf for icon named \"%s\" at size \"%s\" is %p\n",
			icon_name, PIDGIN_ICON_SIZE_TANGO_LARGE, pb);
		if (pb) {
			pidgin_status_menu_plugin_update_icon(pb);
			g_object_unref(pb);
		}
	}
}

static void
pidgin_maemo_docklet_ui_ops_blank_icon()
{
	g_print("|--- *** pidgin_maemo_docklet_ui_ops_blank_icon\n");
}

static void
pidgin_maemo_docklet_ui_ops_position_menu(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer null)
{
	g_print("|--- *** pidgin_maemo_docklet_ui_ops_position_menu\n");
}

struct docklet_ui_ops pidgin_maemo_docklet_ui_ops = {
	.create = pidgin_maemo_docklet_ui_ops_create,
	.destroy = pidgin_maemo_docklet_ui_ops_destroy,
	.update_icon = pidgin_maemo_docklet_ui_ops_update_icon,
	.blank_icon = pidgin_maemo_docklet_ui_ops_blank_icon,
	.set_tooltip = pidgin_status_menu_set_tooltip,
	.position_menu = pidgin_maemo_docklet_ui_ops_position_menu
};

static gboolean
pidgin_status_menu_plugin_connect()
{
	DBusConnection *connection = dbus_connection();
	DBusMessage *message = dbus_message_new_method_call(
		PIDGIN_SYSTRAY_DBUS_SERVICE_NAME, 
		PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, 
		NULL,
		PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_CONNECT);
	DBusMessage *reply = NULL;
	gboolean ret = FALSE;
	DBusError error;

	if (message) {
		dbus_error_init(&error);
		g_print("|--- *** pidgin_status_menu_plugin_connect: Calling \"connect\"\n");
		reply = dbus_connection_send_with_reply_and_block(connection, message, -1, &error);
		if (!reply)
			g_print("|--- *** pidgin_status_menu_plugin_connect: Failed to get D-Bus \"connect\" reply: %s:%s\n",
				error.name ? error.name : "NULL", error.message ? error.message : "NULL");
		dbus_error_free(&error);
		dbus_message_unref(message);
	}

	if (reply) {
		dbus_error_init(&error);
		if (!dbus_message_get_args(reply, &error, DBUS_TYPE_BOOLEAN, &ret, DBUS_TYPE_INVALID)) {
			g_print("|--- *** pidgin_status_menu_plugin_connect: Failed to get boolean from D-Bus reply: %s:%s\n",
				error.name ? error.name : "NULL", error.message ? error.message : "NULL");
			ret = FALSE;
		}
		else
			g_print("|--- *** pidgin_status_menu_plugin_connect: Successfully got boolean from D-Bus reply\n");
		dbus_error_free(&error);
		dbus_message_unref(reply);
	}

		g_print("|--- *** pidgin_status_menu_plugin_connect: Returning %s\n", ret ? "TRUE" : "FALSE");
	return ret;
}

guint embed_idle_id = 0;

static gboolean
embed_idle(gpointer null) {
	pidgin_docklet_embedded();
	embed_idle_id = 0;
	return FALSE;
}

static void
status_menu_lost() {
	g_print("|--- *** status_menu_lost\n");
	pidgin_status_menu_plugin_disconnect();
	pidgin_docklet_init();

	if (embed_idle_id) {
		g_source_remove(embed_idle_id);
		embed_idle_id = 0;
	}
}

static void
status_menu_found() {
	g_print("|--- *** status_menu_found\n");
//	if (pidgin_docklet_get_ui_ops() != &pidgin_maemo_docklet_ui_ops) {
		docklet_ui_init(); /* sets the X11 UI ops, making sure it is not us who get disconnected */
		pidgin_docklet_embedded(); /* Make sure pidgin calls destroy on the X11 UI ops */
		pidgin_docklet_uninit(); /* Calls destroy on the X11 UI ops */
//	}
	g_print("|--- *** status_menu_found: Setting maemo docklet UI ops\n");
	pidgin_docklet_set_ui_ops(&pidgin_maemo_docklet_ui_ops); /* Set my UI ops */

	/* Need to embed idly, because at pidgin load time, though pidgin stock is already initialized, 
	   there's no way to get a GdkPixbuf from a stock item, because there's no widget we can use for
		 gtk_widget_render_icon (need a properly initialized PidginBuddyList for that) */
	embed_idle_id = g_idle_add((GSourceFunc)embed_idle, NULL);
}

static DBusHandlerResult
message_filter(DBusConnection *connection, DBusMessage *message, gpointer null)
{
	DBusHandlerResult ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	/* clicked(button) */
	if (DBUS_MESSAGE_TYPE_METHOD_CALL == dbus_message_get_type(message) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_PATH, dbus_message_get_path(message)) &&
		!g_strcmp0(PIDGIN_SYSTRAY_DBUS_SERVICE_METHOD_CLICKED, dbus_message_get_member(message))) {
		DBusMessage *reply = dbus_message_new_method_return(message);
		int button = 0;

		if (!dbus_message_get_args(message, NULL, DBUS_TYPE_INT32, &button, DBUS_TYPE_INVALID))
			button = 0;

		if (reply) {
			dbus_connection_send(connection, reply, NULL);
			dbus_connection_flush(connection);
			dbus_message_unref(reply);
		}
		g_print("|--- *** PidginMaemoDocklet::message_filter: Calling pidgin_docklet_clicked()\n");
		pidgin_docklet_clicked(button);
	}
	else
	/* NameOwnerChanged(name, old, new) */
	if (DBUS_MESSAGE_TYPE_SIGNAL == dbus_message_get_type(message) &&
		!g_strcmp0("org.freedesktop.DBus", dbus_message_get_sender(message)) &&
		!g_strcmp0("org.freedesktop.DBus", dbus_message_get_interface(message)) &&
		!g_strcmp0("/org/freedesktop/DBus", dbus_message_get_path(message)) &&
		!g_strcmp0("NameOwnerChanged", dbus_message_get_member(message))) {
		const char *name, *old_owner, *new_owner;

		g_print("|--- *** PidginMaemoDocklet::message_filter: NameOwnerChanged\n");

		if (dbus_message_get_args(message, NULL, 
			DBUS_TYPE_STRING, &name,
			DBUS_TYPE_STRING, &old_owner,
			DBUS_TYPE_STRING, &new_owner,
			DBUS_TYPE_INVALID)) {
			if (!g_strcmp0(name, PIDGIN_SYSTRAY_DBUS_SERVICE_NAME) &&
			    !g_strcmp0(new_owner, ""))
				status_menu_lost();
			else
			if (!g_strcmp0(name, PIDGIN_SYSTRAY_DBUS_SERVICE_NAME) &&
			    g_strcmp0(new_owner, "") &&
					g_strcmp0(old_owner, new_owner))
				if (pidgin_status_menu_plugin_connect())
					status_menu_found();
		}
	}

	return ret;
}

static DBusConnection *
dbus_connection()
{
	static DBusConnection *conn = NULL;

	if (G_UNLIKELY(NULL == conn)) {
		conn = dbus_bus_get(DBUS_BUS_SESSION, NULL);
		dbus_connection_setup_with_g_main(conn, NULL);
		dbus_connection_add_filter(conn, (DBusHandleMessageFunction)message_filter, NULL, NULL);
		dbus_bus_add_match(conn, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',path='/org/freedesktop/DBus',member='NameOwnerChanged'", NULL);
		g_print("|--- *** dbus_connection: My unique name is %s\n", dbus_bus_get_unique_name(conn));
	}

	return conn;
}

static gboolean
plugin_load(PurplePlugin *plugin) {
	if (pidgin_status_menu_plugin_connect())
		status_menu_found();
	return TRUE;
}

static gboolean
plugin_unload(PurplePlugin *plugin) {
	status_menu_lost();
	return TRUE;
}

static PurplePluginInfo info =
{
	PURPLE_PLUGIN_MAGIC,
	PURPLE_MAJOR_VERSION,
	PURPLE_MINOR_VERSION,
	PURPLE_PLUGIN_STANDARD,
	PIDGIN_PLUGIN_TYPE,
	0,
	NULL,
	PURPLE_PRIORITY_DEFAULT,
	"pidgin-maemo-docklet",
	"Maemo Status Menu Icon",
	PIDGIN_MAEMO_DOCKLET_VERSION,
	"Adds a status menu icon for Pidgin.",
	"With this plugin you can access the buddy list from the status menu and receive notifications about incoming messages there.",
	"Gabriel Schulhof <nix@go-nix.ca>",
	"",
	plugin_load,
	plugin_unload,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static void
init_plugin(PurplePlugin *plugin)
{
}

PURPLE_INIT_PLUGIN(pidgin_maemo_docklet, init_plugin, info)
