#include "Dates.h"

#include "Date.h"

#include "../Database.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Menu.h>

#include <Lum/Model/Selection.h>

namespace Views {

  class DatesEntryModelPainter : public Lum::StringCellPainter
  {
  public:
    std::wstring GetCellData() const
    {
      const DatesEntry* entry=dynamic_cast<const DatesEntryModel*>(GetModel())->GetEntry(GetRow());

      if (GetColumn()==1) {
        std::wstring res;

        res=entry->startDate.GetLocaleDate();

        if (!entry->date->wholeDay) {
          res+=L" "+entry->date->startTime.GetLocaleTime();
        }

        if (entry->startDate!=entry->endDate) {
          res+=L"-"+entry->endDate.GetLocaleDate();

          if (!entry->date->wholeDay) {
            res+=L" "+entry->date->endTime.GetLocaleTime();
          }
        }
        else if (!entry->date->wholeDay) {
          res+=L"-"+entry->date->endTime.GetLocaleTime();
        }

        return res;
      }
      else if (GetColumn()==2) {
        return entry->date->GetAttribute(::Data::Date::DateType::description);
      }

      return L"";
    }
  };

  class DatesEntryComparator : public DatesEntryModel::Comparator
  {
  public:
    bool operator()(const DatesEntryModel::ElementType& a,
                    const DatesEntryModel::ElementType& b,
                    size_t column, bool down) const
    {
      if (a->startDate<b->startDate) {
        return true;
      }
      else if (a->startDate==b->startDate) {
        if (a->date->wholeDay && b->date->wholeDay) {
          return false;
        }
        else if (a->date->wholeDay && !b->date->wholeDay) {
          // Current date must be bigger, if it does not also point to 0:00:00
          return !(a->date->startTime.GetHour()==0 &&
                   a->date->startTime.GetMinute()==0 &&
                   a->date->startTime.GetSecond()==0);
        }
        else if (!a->date->wholeDay && b->date->wholeDay) {
          // b date must be bigger, if it does not also point to 0:00:00
          return !(b->date->startTime.GetHour()==0 &&
                   b->date->startTime.GetMinute()==0 &&
                   b->date->startTime.GetSecond()==0);
        }
        else {
          // Both point to full date and time stamps.
          return a->date->startTime<b->date->startTime;
        }
      }
      else {
        return false;
      }
    }
  };

  Dates::Dates()
  : View(L"Dates"),
    viewAction(new Lum::Model::Action),
    addAction(new Lum::Model::Action),
    deleteAction(new Lum::Model::Action),
    model(new DatesEntryModel(new DatesEntryComparator())),
    table(NULL),top(NULL)
  {
    Observe(viewAction);
    Observe(addAction);
    Observe(deleteAction);
  }

  Dates::~Dates()
  {
    delete top;
  }

  void Dates::AddEntryIfInFuture(::Data::DatePtr date,
                                 const Lum::Base::Calendar& now,
                                 const Lum::Base::Calendar& startDate,
                                 const Lum::Base::Calendar& endDate)
  {
    if (endDate>=now) {
      model->Append(new DatesEntry(date,startDate,endDate));
    }
  }

  void Dates::FillList()
  {
    Lum::Base::Calendar now;

    model->Off();

    model->Clear();

    ::Base::DataConstIterator iter;

    iter=database->database.begin();
    while (iter!=database->database.end()) {
      ::Data::DatePtr date;

      if ((date=dynamic_cast< ::Data::DatePtr>(*iter))!=NULL) {
        Lum::Base::Calendar startDate,endDate;

        switch (date->repeat) {
        case ::Data::Date::DateType::noRepeat:
          startDate=date->startDate;
          endDate=date->endDate;
          AddEntryIfInFuture(date,now,startDate,endDate);
          break;
        case ::Data::Date::DateType::dailyRepeat:
          startDate=date->startDate;
          endDate=date->endDate;
          //startDate.AddDays(1);
          //endDate.AddDays(1);

          while (startDate<date->repeatEndDate) {
            AddEntryIfInFuture(date,now,startDate,endDate);

            startDate.AddDays(1);
            endDate.AddDays(1);
          }
          break;
        case ::Data::Date::DateType::weeklyRepeat:
          startDate=date->startDate;
          endDate=date->endDate;
          //startDate.AddDays(7);
          //endDate.AddDays(7);

          while (startDate<date->repeatEndDate) {
            AddEntryIfInFuture(date,now,startDate,endDate);

            startDate.AddDays(7);
            endDate.AddDays(7);
          }
          break;
        case ::Data::Date::DateType::monthlyRepeat:
          startDate=date->startDate;
          endDate=date->endDate;
          //startDate.AddMonths(1);
          //endDate.AddMonths(1);

          while (startDate<date->repeatEndDate) {
            AddEntryIfInFuture(date,now,startDate,endDate);

            startDate.AddMonths(1);
            endDate.AddMonths(1);
          }
          break;
        case ::Data::Date::DateType::yearlyRepeat:
          startDate=date->startDate;
          endDate=date->endDate;
          //startDate.AddYears(1);
          //endDate.AddYears(1);

          while (startDate<date->repeatEndDate) {
            AddEntryIfInFuture(date,now,startDate,endDate);

            startDate.AddYears(1);
            endDate.AddYears(1);
          }
          break;
        }
      }

      iter++;
    }

    model->Sort(1);

    model->On();
  }

  bool Dates::VisitChildren(Lum::Visitor &visitor, bool onlyVisible)
  {
    if (top!=NULL) {
      return visitor.Visit(top);
    }

    return true;
  }

  void Dates::CalcSize()
  {
    Lum::Menu             *menu;
    Lum::Model::HeaderRef headerModel;

    top=Lum::VPanel::Create(true,true);
    top->SetParent(this);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Date",Lum::Base::Size::stdCharWidth,20);
    headerModel->AddColumn(L"Description",Lum::Base::Size::stdCharWidth,40,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetModel(model);
    table->SetPainter(new DatesEntryModelPainter());
    table->SetHeaderModel(headerModel);
    table->SetSelection(new Lum::Model::SingleLineSelection());
    table->SetDoubleClickAction(viewAction);

    menu=new Lum::Menu();
    menu->SetParent(GetWindow());
    menu->AddActionItem(L"View",viewAction);
    menu->AddActionItem(L"Add",addAction);
    menu->AddActionItem(L"Delete",deleteAction);
    table->SetMenu(menu->GetWindow());

    top->Add(table);

    top->CalcSize();

    width=top->GetOWidth();
    height=top->GetOHeight();
    minWidth=top->GetOMinWidth();
    minHeight=top->GetOMinHeight();
    maxWidth=top->GetOMaxWidth();
    maxHeight=top->GetOMaxHeight();

    View::CalcSize();
  }

  void Dates::Layout()
  {
    top->Resize(width,height);
    top->Move(x,y);

    View::Layout();
  }

  void Dates::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==viewAction && viewAction->IsFinished()) {
      if (table->GetTableView()->GetSelection()->HasSelection()) {
        size_t line;
        Date::Data *data;

        line=dynamic_cast<Lum::Model::SingleLineSelection*>(table->GetTableView()->GetSelection())->GetLine();
        data=new ::Base::View::Data(dynamic_cast< DatesEntry*>(this->model->GetEntry(line))->date);
        GUI::GetInstance()->ShowView(L"Date",data);
      }
    }
    else if (model==addAction && addAction->IsFinished()) {
      Date::Data *data;

      data=new ::Base::View::Data(new ::Data::Date());
      GUI::GetInstance()->ShowView(L"Date",data);
    }
    else if (model==deleteAction && deleteAction->IsFinished()) {
      if (table->GetTableView()->GetSelection()->HasSelection()) {
        size_t       line;
        ::Data::Date *date;

        line=dynamic_cast<Lum::Model::SingleLineSelection*>(table->GetTableView()->GetSelection())->GetLine();
        date=dynamic_cast< DatesEntry*>(this->model->GetEntry(line))->date;
        database->Remove(date);
        FillList();
      }
    }
    else {
      View::Resync(model,msg);
    }
  }

  void Dates::Load(::Base::View::DataPtr data=NULL)
  {
    FillList();
  }

  void Dates::Vanish()
  {
    model->Clear();
  }
}
