/*
 * This file is part of Personal GRPS Monitor.
 *
 * Copyright (C) 2009 Andrew Olmsted. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
*  
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/* personal-grps-mon.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "personal-grps-mon.h"
#include <hildon/hildon.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <gconf/gconf-client.h>

#define HOME_DIR g_get_home_dir()

#define GRPS_GCONF_PATH "/system/osso/connectivity/network_type/GPRS"
#define GRPS_GCONF_RX GRPS_GCONF_PATH  "/gprs_rx_bytes"
#define GRPS_GCONF_TX GRPS_GCONF_PATH  "/gprs_tx_bytes"

#define _(String) dgettext("hildon-libs", String)

#define PERSONAL_GRPS_MON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
							                         PERSONAL_TYPE_GRPS_MON,\
							                         PersonalGrpsMonPrivate))

struct _PersonalGrpsMonPrivate
{
    GtkWidget *homeWidget;
    GtkWidget *contents;
	
	GtkWidget *titleLabel;
	GtkWidget *uploadLabel;
	GtkWidget *uploadContent;
	GtkWidget *downloadLabel;
	GtkWidget *downloadContent;
	
	GConfClient *client;
	
	guint eventID;
};

HD_DEFINE_PLUGIN_MODULE (PersonalGrpsMon, personal_grps_mon, HD_TYPE_HOME_PLUGIN_ITEM);

gboolean personal_grps_mon_update_content (PersonalGrpsMon *self);

void personal_grps_mon_content_create (PersonalGrpsMon *self)
{
	//g_warning ("personal_grps_mon_content_create");
	self->priv->contents = gtk_vbox_new (TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->contents), 0);
	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));

    self->priv->titleLabel = gtk_label_new ("GRPS Monitor");
	gtk_label_set_markup (GTK_LABEL (self->priv->titleLabel), "<b><u>GRPS Monitor</u></b>");
	
	self->priv->uploadLabel = gtk_label_new ("Uploaded:  ");
	gtk_size_group_add_widget (group, self->priv->uploadLabel);
	self->priv->uploadContent = gtk_label_new ("0.000 B");
	GtkWidget *uploadBox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (uploadBox), self->priv->uploadLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (uploadBox), self->priv->uploadContent, TRUE, TRUE, 0);
	hildon_helper_set_logical_font (self->priv->uploadLabel, "SmallSystemFont");
	hildon_helper_set_logical_font (self->priv->uploadContent, "SmallSystemFont");
	gtk_misc_set_alignment (GTK_MISC (self->priv->uploadLabel), 0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (self->priv->uploadContent), 1, 0.5);
	
	self->priv->downloadLabel = gtk_label_new ("Downloaded:  ");
	gtk_size_group_add_widget (group, self->priv->downloadLabel);
	self->priv->downloadContent = gtk_label_new ("0.000 B");
	GtkWidget *downloadBox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (downloadBox), self->priv->downloadLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (downloadBox), self->priv->downloadContent, TRUE, TRUE, 0);	
	hildon_helper_set_logical_font (self->priv->downloadLabel, "SmallSystemFont");
	hildon_helper_set_logical_font (self->priv->downloadContent, "SmallSystemFont");
	gtk_misc_set_alignment (GTK_MISC (self->priv->downloadLabel), 0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (self->priv->downloadContent), 1, 0.5);
	
    gtk_container_add (GTK_CONTAINER (self->priv->contents), self->priv->titleLabel);
	gtk_container_add (GTK_CONTAINER (self->priv->contents), uploadBox);
	gtk_container_add (GTK_CONTAINER (self->priv->contents), downloadBox);
    gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), GTK_WIDGET (self->priv->contents), FALSE, FALSE, 0);
	
    gtk_widget_show_all (self->priv->homeWidget);
}

gboolean personal_grps_mon_update_content (PersonalGrpsMon *self)
{
	//g_warning ("personal_grps_mon_update_content");
	gchar *uploadString, *downloadString;
	gint upload, download;
	GError *error = NULL;

	uploadString = gconf_client_get_string (self->priv->client, GRPS_GCONF_TX, &error);
	if (error) {
		g_warning
			("Error trying to get gconf variable %s", GRPS_GCONF_TX);
		upload = 0;
		g_error_free (error);
		error = NULL;
	}
	downloadString = gconf_client_get_string (self->priv->client, GRPS_GCONF_RX, &error);
	if (error) {
		g_warning
			("Error trying to get gconf variable %s", GRPS_GCONF_RX);
		download = 0;
		g_error_free (error);
		error = NULL;
	}
	
	upload = g_ascii_strtoll (uploadString, NULL, 10);
	download = g_ascii_strtoll (downloadString, NULL, 10);
	
	gdouble uploadSize;
	gdouble downloadSize;
	gchar *uploadUnits, *downloadUnits;
	
	if (upload > 1000 && upload <= 1000000) {
		uploadSize = upload / 1000.0;
		uploadUnits = " KB";
	} else if (upload > 1000000 && upload <= 1000000000) {
		uploadSize = upload / 1000000.0;
		uploadUnits = " MB";
	} else if (upload > 1000000000) {
		uploadSize = upload / 1000000000.0;
		uploadUnits = " GB";
	} else {
		uploadSize = upload /1.0;
		uploadUnits = " B";
	}
	
	if (download > 1000 && download <= 1000000) {
		downloadSize = download / 1000.0;
		downloadUnits = " KB";
	} else if (download > 1000000 && download <= 1000000000) {
		downloadSize = download / 1000000.0;
		downloadUnits = " MB";
	} else if (upload > 1000000000) {
		downloadSize = download / 1000000000.0;
		downloadUnits = " GB";
	} else {
		downloadSize = download / 1.0;
		downloadUnits = " B";
	}
	
	gtk_label_set_text (GTK_LABEL (self->priv->uploadContent), g_strconcat (g_strdup_printf ("%.3f", uploadSize), uploadUnits, NULL));
	gtk_label_set_text (GTK_LABEL (self->priv->downloadContent), g_strconcat (g_strdup_printf ("%.3f", downloadSize), downloadUnits, NULL));
	
	return TRUE;
}

static void personal_grps_mon_check_desktop (GObject *gobject, GParamSpec *pspec, PersonalGrpsMon *self)
{
	//g_warning ("personal_grps_mon_check_desktop");
	gchar *name = pspec->name;
	gboolean status;
	g_object_get (gobject, name, &status, NULL);
	if (status) {
		personal_grps_mon_update_content (self);
		if (self->priv->eventID == 0) {
			self->priv->eventID = g_timeout_add (1*1000*60, (GSourceFunc)personal_grps_mon_update_content, self);
		}
	} else if (self->priv->eventID != 0) {
		g_source_remove (self->priv->eventID);
		self->priv->eventID = 0;
	}
}

static void personal_grps_mon_dispose (GObject *object)
{
	//g_warning ("personal_grps_mon_dispose");
	PersonalGrpsMon *self = PERSONAL_GRPS_MON (object);

	G_OBJECT_CLASS (personal_grps_mon_parent_class)->dispose (object);
}

static void personal_grps_mon_finalize (GObject *object)
{
	//g_warning ("personal_grps_mon_finalize");
	PersonalGrpsMon *self = PERSONAL_GRPS_MON (object);

	if (self->priv->eventID) {
		g_source_remove (self->priv->eventID);
	}
	g_object_unref (self->priv->client);

	G_OBJECT_CLASS (personal_grps_mon_parent_class)->finalize (object);
}

static void personal_grps_mon_realize (GtkWidget *widget)
{
	//g_warning ("personal_grps_mon_realize");
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (personal_grps_mon_parent_class)->realize (widget);
}

static gboolean personal_grps_mon_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	//g_warning ("personal_grps_mon_expose_event");
	PersonalGrpsMon *self = PERSONAL_GRPS_MON (widget);
	cairo_t *cr;

    cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));

    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
    cairo_paint (cr);

    cairo_destroy (cr);
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));

	GdkColor color;
	gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	
	gint width, height, x, y;
	gint radius = 10;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;

	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width - radius, y);
	cairo_curve_to(cr, x + width - radius, y, x + width, y, x + width,
				y + radius);
	cairo_line_to(cr, x + width, y + height - radius);
	cairo_curve_to(cr, x + width, y + height - radius, x + width,
				y + height, x + width - radius, y + height);
	cairo_line_to(cr, x + radius, y + height);
	cairo_curve_to(cr, x + radius, y + height, x, y + height, x,
				y + height - radius);
	cairo_line_to(cr, x, y + radius);
	cairo_curve_to(cr, x, y + radius, x, y, x + radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill(cr);
	cairo_destroy(cr);

	return GTK_WIDGET_CLASS (personal_grps_mon_parent_class)->expose_event (widget, event);
}

static void personal_grps_mon_class_init (PersonalGrpsMonClass *klass)
{
	//g_warning ("personal_grps_mon_class_init");
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = personal_grps_mon_dispose;
	object_class->finalize = personal_grps_mon_finalize;
	
	widget_class->realize = personal_grps_mon_realize;
	widget_class->expose_event = personal_grps_mon_expose_event;

	g_type_class_add_private (klass, sizeof (PersonalGrpsMonPrivate));
}

static void personal_grps_mon_class_finalize (PersonalGrpsMonClass *klass G_GNUC_UNUSED)
{
}

static void personal_grps_mon_init (PersonalGrpsMon *self)
{
	//g_warning ("personal_grps_mon_init");
	self->priv = PERSONAL_GRPS_MON_GET_PRIVATE (self);
	self->priv->eventID = 0;
	
	gtk_window_set_default_size (GTK_WINDOW (self), 300, 100);
	
	self->priv->homeWidget = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->homeWidget), 0);
	gtk_container_add (GTK_CONTAINER (self), self->priv->homeWidget);
	gtk_widget_show (self->priv->homeWidget);

	GdkGeometry hints;
	hints.min_width = 300;
	hints.min_height = 100;
	hints.max_width = 300;
	hints.max_height = 100;
	gtk_window_set_geometry_hints (GTK_WINDOW (self), self->priv->homeWidget, &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	
	g_type_init ();
	self->priv->client = gconf_client_get_default ();
	
	personal_grps_mon_content_create (self);
	
	g_signal_connect (self, "notify::is-on-current-desktop", G_CALLBACK (personal_grps_mon_check_desktop), self);
}

PersonalGrpsMon* personal_grps_mon_new (void)
{
  return g_object_new (PERSONAL_TYPE_GRPS_MON, NULL);
}
