#include "prtopstackedwidget.h"
#include <QEvent>
#include <QKeyEvent>
#include "prfeedview.h"
#include "prtextbody.h"
#include <QListWidget>
#include "prconst.h"
#include <QDebug>
#include <QSettings>
#include <QDesktopServices>
#include <QUrl>
#include <QTextBrowser>
#include <QFileInfo>
#include <QPushButton>

#include "feedinterface.h"
#include "feedrenderer.h"
#include "fingerbuttondialog.h"
#include "prappservices.h"
#include "textedithelper.h"


/*!
    \class PrTopStackedWidget

    \brief Stacked widget that contains PrFeedView and PtTextBody

    Handles the dirty details of navigation between views, url selection
    and menu presentation.
*/


PrTopStackedWidget::PrTopStackedWidget(QWidget *parent, PrAppServices* appsrv) :
    QStackedWidget(parent), m_appSrv(appsrv), m_renderer(NULL)
{    
    m_FeedView = new PrFeedView(m_appSrv, this );
    m_TextBody = new PrTextBody(m_appSrv, this);

    this->addWidget(m_FeedView);
    this->addWidget(m_TextBody);

    connect(m_FeedView->listWidget(), SIGNAL(itemActivated(QListWidgetItem*)),
        this,SLOT(showFeed(QListWidgetItem*)));
    connect(m_FeedView->listWidget(), SIGNAL(itemClicked(QListWidgetItem*)),
        this,SLOT(showFeed(QListWidgetItem*)));
    connect(m_TextBody, SIGNAL(anchorClicked(const QUrl&)), this, SLOT(urlClicked(QUrl)));

    m_linksEnabled=false;
}

PrTopStackedWidget::~PrTopStackedWidget()
{
}

void PrTopStackedWidget::changeEvent(QEvent *e)
{
    QStackedWidget::changeEvent(e);
    switch (e->type()) {
        case QEvent::LanguageChange: {
            break;
        }
        default: {
            break;
        }
    }
}

void PrTopStackedWidget::keyPressEvent(QKeyEvent *e)
{
    int scancode=e->nativeScanCode();
    if (scancode==TOGGLE_TEXT && m_TextBody->toPlainText().length()>0) {
        setCurrentWidget(m_TextBody);
        e->accept();
    } else {
        QStackedWidget::keyPressEvent(e);
    }
}

void PrTopStackedWidget::showFeed(QListWidgetItem* item)
{
    if ( m_currentItem != item ) {
        const IFeed* feed = m_feedmap[item];

        QVector<FeedEntry> entries;
        feed->getEntries(entries);
        m_renderer->renderEntries(entries, *m_TextBody);
        m_currentItem = item;
    }
    setCurrentWidget(m_TextBody);
}

void PrTopStackedWidget::setFeedBundle(IFeedBundle* feeds)
{    
    m_feedbundle = feeds;
    QListWidget* w = m_FeedView->listWidget();
    w->clear();
    QVector<IFeed*> allfeeds;
    feeds->getFeeds(allfeeds);

    foreach (const IFeed* feed, allfeeds) {
        QListWidgetItem* lwi = new QListWidgetItem(feed->name(), w);
        feed->decorate(lwi);
        w->addItem(lwi);
        m_feedmap[lwi] = feed;
    }
}

void PrTopStackedWidget::urlClicked(const QUrl& link )
{
    QString url = link.toString();
    qDebug() << url;

    FingerButtonDialog* dlg = new FingerButtonDialog();
    dlg->setWindowTitle(url);
    QStringList buttons;
    QString linkBtnText;

    if ( m_linksEnabled ) {
        linkBtnText=tr("Disable Links");
    } else {
        linkBtnText=tr("Enable Links");
    }
    buttons << tr("Browser") << tr("Tear") << tr("Stash") << linkBtnText << tr("Back") << tr("Go");

    dlg->setButtons(buttons);

    // only allow "Go" for local files
    QUrl baseurl = renderer()->baseUrl();
    bool goAllowed = false;
    if (!baseurl.isEmpty()) {
        QString lfile = link.toLocalFile();
        if (QFileInfo(lfile).isFile()) {
            goAllowed = true;
        }
    }

    QPushButton* goBtn = qobject_cast<QPushButton*>(dlg->widgetAt(5));
    if (!goAllowed) {
        goBtn->setEnabled(false);
    }

    QFileInfo tearpth("/usr/bin/tear");
    if (!tearpth.isExecutable()) {
        qobject_cast<QPushButton*>(dlg->widgetAt(1))->setEnabled(false);
    }

    int ret = dlg->exec();
    int sel = dlg->selectedIndex();
    delete dlg;
    if (ret == QDialog::Rejected) {
        return;
    }

    switch (sel) {
        case 0: {
            m_appSrv->settings()->setValue("browser/kind", "default");
            m_appSrv->launchUrl(url);
            break;
        }
        case 1: {
            m_appSrv->settings()->setValue("browser/kind", "tear");
            m_appSrv->launchUrl(url);
            // yeah, would have to go through dbus
            break;
        }
        case 2: {
            m_urlList.append(url.toAscii());
            m_FeedView->enableStash();
            break;
        }
        case 3: {
            m_linksEnabled=!m_linksEnabled;
            break;
        }
        case 4: {
            bodyWidget()->backward();
            break;
        }
        case 5: {
            bodyWidget()->setSource(link);
            TextEditHelper(*bodyWidget()).cursorToCenter();
            break;
        }
        default: {
            // pass
        }
    }
}

PrTextBody* PrTopStackedWidget::bodyWidget()
{
    return m_TextBody;
}

PrFeedView* PrTopStackedWidget::feedWidget()
{
    return m_FeedView;
}

QStringList PrTopStackedWidget::urlList()
{
    return m_urlList;
}

bool PrTopStackedWidget::linksEnabled()
{
    return m_linksEnabled;
}

void PrTopStackedWidget::setRenderer(FeedRenderer* renderer)
{
    m_renderer = renderer;
}
