/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Walter Guerra <walter.guerra@signove.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon-banner.h>

#include "nm_internet_setup.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static void nm_internet_setup_create_widget (NmInternetSetup *);
static void nm_internet_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_internet_setup_enable_options (NmInternetSetup *, gboolean);

typedef struct _NmInternetSetupPrivate NmInternetSetupPrivate;

struct _NmInternetSetupPrivate
{
    GtkWidget *internet_combobox;

    GtkWidget *dns_hbox;
    GtkWidget *dns_entry;

    GtkWidget *usb_checkbox;
    GtkWidget *bt_checkbox;
    GtkWidget *wlan_checkbox;

    gchar *env;
};

#define NM_INTERNET_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_INTERNET_SETUP_TYPE, NmInternetSetupPrivate))

static void nm_internet_setup_class_init (NmInternetSetupClass *);
static void nm_internet_setup_init       (NmInternetSetup *);
static void nm_internet_setup_dispose    (GObject *);
static void nm_internet_setup_finalize   (GObject *);

static void nm_internet_setup_combobox_changed (gpointer instance, gpointer self);


G_DEFINE_TYPE (NmInternetSetup, nm_internet_setup, NM_CONN_WIDGET_TYPE)

static void
nm_internet_setup_class_init (NmInternetSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmInternetSetupPrivate));

    object_class->dispose = nm_internet_setup_dispose;
    object_class->finalize = nm_internet_setup_finalize;

    conn_widget_class->apply_changes = nm_internet_setup_apply_changes;
}

static void
nm_internet_setup_init (NmInternetSetup *self)
{
    NmInternetSetupPrivate *priv;

    priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

    nm_internet_setup_create_widget (self);
}

static void
nm_internet_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_internet_setup_parent_class)->dispose (object);
}

static void
nm_internet_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_internet_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_internet_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_INTERNET_SETUP_TYPE, NULL);

    NmInternetSetupPrivate *priv;

    priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

    priv->env = g_strdup(env);

    return self;
}


static void
nm_internet_setup_create_widget(NmInternetSetup *self)
{
    GtkWidget *table;
    NmInternetSetupPrivate *priv;

    priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

    GtkWidget *internet_vbox = gtk_vbox_new(FALSE, 0);

    /* Start Label */
    GtkWidget *connect_hbox = gtk_hbox_new(FALSE, 0);
    GtkWidget *connect_label = gtk_label_new(_("Connect using:"));

    /* Internet combobox */
    priv->internet_combobox = gtk_combo_box_new_text ();
    gtk_combo_box_append_text (GTK_COMBO_BOX(priv->internet_combobox), NM_INTERNET_INTERFACE_WLAN);
    gtk_combo_box_append_text (GTK_COMBO_BOX(priv->internet_combobox), NM_INTERNET_INTERFACE_WLAN_ADHOC);
    gtk_combo_box_append_text (GTK_COMBO_BOX(priv->internet_combobox), NM_INTERNET_INTERFACE_USB);
    gtk_combo_box_append_text (GTK_COMBO_BOX(priv->internet_combobox), NM_INTERNET_INTERFACE_BT);
    gtk_combo_box_set_active(GTK_COMBO_BOX(priv->internet_combobox), 0);

    g_signal_connect(G_OBJECT(priv->internet_combobox), "changed",
                     G_CALLBACK (nm_internet_setup_combobox_changed), self);

    gtk_box_pack_start(GTK_BOX(connect_hbox), connect_label, FALSE, TRUE, 4);
    gtk_box_pack_start(GTK_BOX(connect_hbox), priv->internet_combobox, TRUE, TRUE, 2);

    /* DNS */
    GtkWidget *dns_label = gtk_label_new("DNS:");
    priv->dns_hbox = gtk_hbox_new(FALSE, 0);
    priv->dns_entry = gtk_entry_new();

    gtk_entry_set_width_chars(GTK_ENTRY(priv->dns_entry), 15);
    gtk_misc_set_alignment(GTK_MISC(dns_label), 1.0, 0.5);
    gtk_box_pack_start(GTK_BOX(priv->dns_hbox), dns_label, TRUE, TRUE, 4);
    gtk_box_pack_start(GTK_BOX(priv->dns_hbox), priv->dns_entry, FALSE, TRUE, 0);

    /* If this is a Diablo device, enable the internet sharing options */
#ifdef DIABLO

    /* Share Label */
    GtkWidget *share_label = gtk_label_new(_("Share internet connection with:"));
    gtk_misc_set_alignment(GTK_MISC(share_label), 0.1, 0.5);


    /* connection checkboxes */
    GtkWidget *connections_hbox = gtk_hbox_new(FALSE, 5);
    priv->usb_checkbox = gtk_check_button_new_with_label (_("Usb"));
    priv->bt_checkbox = gtk_check_button_new_with_label (_("Bluetooth"));
    priv->wlan_checkbox = gtk_check_button_new_with_label (_("Wlan"));

    /* Add connection shares to the hbox */
    gtk_box_pack_start(GTK_BOX(connections_hbox), priv->usb_checkbox, FALSE, TRUE, 1);
    gtk_box_pack_start(GTK_BOX(connections_hbox), priv->bt_checkbox, FALSE, TRUE, 1);
    gtk_box_pack_start(GTK_BOX(connections_hbox), priv->wlan_checkbox, FALSE, TRUE, 1);

#endif

    /* Add everything to the vbox */
    gtk_box_pack_start(GTK_BOX(internet_vbox), connect_hbox, FALSE, TRUE, 3);
    gtk_box_pack_start(GTK_BOX(internet_vbox), priv->dns_hbox, FALSE, TRUE, 2);
#ifdef DIABLO
    gtk_box_pack_start(GTK_BOX(internet_vbox), share_label, FALSE, TRUE, 2);
    gtk_box_pack_start(GTK_BOX(internet_vbox), connections_hbox, FALSE, TRUE, 2);
#endif

    gtk_container_add(GTK_CONTAINER(self), internet_vbox);

    gtk_widget_show_all(GTK_WIDGET(self));
}



gboolean
nm_internet_setup_apply_changes (NmConnWidget *self)
{
    NmInternetSetupPrivate *priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();

    gint active_checkbox = gtk_combo_box_get_active ( GTK_COMBO_BOX(priv->internet_combobox) );

    gchar *gateway = NULL;

    gchar *dns = NULL;

    gchar *interface_out = NULL;

    nm_internet_setup_save_changes(NM_INTERNET_SETUP(self));

    dns = nm_settings_get_value(settings, priv->env, NM_INTERNET_SETTINGS_GROUP, NM_INTERNET_DNS);
    if (active_checkbox == 0 ||
    	active_checkbox == 1) { // WLAN
        gateway = nm_settings_get_value(settings, priv->env, NM_WLAN_SETTINGS_GROUP,
                                        NM_SETTINGS_GATEWAY);
        interface_out = "wlan0";
    }
    else if ( active_checkbox == 2 ) { // USB
        gateway = nm_settings_get_value(settings, priv->env, NM_USB_SETTINGS_GROUP,
                                        NM_SETTINGS_GATEWAY);
        interface_out = "usb0";
    }
    else if ( active_checkbox == 3 ) { // BT
        gateway = nm_settings_get_value(settings, priv->env, NM_BT_SETTINGS_GROUP,
                                        NM_SETTINGS_GATEWAY);
        interface_out = "bnep0";
    }



    if (gateway != NULL && dns != NULL) {
        gchar command[100] = {0};
        gchar interface_in[50] = {0};
		gchar *temp = gtk_combo_box_get_active_text(GTK_COMBO_BOX(priv->internet_combobox));
		
		sprintf(command, "sudo /usr/sbin/nm-internet-setup start %s %s %s",
			   (strcmp(temp, NM_INTERNET_INTERFACE_WLAN_ADHOC) == 0) ? "Wlan-adhoc" : temp,
			   gateway, dns);

		g_debug("[%s] - command: %s", __FUNCTION__, command);
		system(command);

#ifdef DIABLO
        /* If this is a diablo device, enable internet sharing*/
         
        if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->usb_checkbox)) == TRUE) {
        	strncat(interface_in, "usb0 ", 50);
        }
        if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->bt_checkbox)) == TRUE) {
        	strncat(interface_in, "bnep0 ", 50);
        }
        if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->wlan_checkbox)) == TRUE) {
        	strncat(interface_in, "wlan0 ", 50);
        }

        memset(command, 0, 100);
        if (strlen(interface_in) > 0) {
            sprintf(command, "sudo /usr/sbin/nm-internet-setup start_share %s %s",
            		interface_out, interface_in);
        }
        else {
            sprintf(command, "sudo /usr/sbin/nm-internet-setup stop_share");
        }

        g_debug("[%s] - command: %s", __FUNCTION__, command);
        system(command);
#endif
        return TRUE;
    }
    else {
        g_debug("[%s] - Error getting the gateway", __FUNCTION__);
        return FALSE;
    }
}

gboolean
nm_internet_setup_activate_environment (gchar *env)
{

}

void
nm_internet_setup_save_changes (NmInternetSetup *self)
{
    NmInternetSetupPrivate *priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();

    nm_settings_set_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                           NM_INTERNET_INTERFACE,
                           gtk_combo_box_get_active_text ( GTK_COMBO_BOX(priv->internet_combobox)));
    nm_settings_set_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                           NM_INTERNET_DNS,
                           gtk_entry_get_text ( GTK_ENTRY (priv->dns_entry)));

    nm_settings_set_boolean(settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
							NM_INTERNET_IN_INTERFACE_USB,
							gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->usb_checkbox)));
    nm_settings_set_boolean(settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
							NM_INTERNET_IN_INTERFACE_WLAN,
							gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->wlan_checkbox)));
    nm_settings_set_boolean(settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
							NM_INTERNET_IN_INTERFACE_BT,
							gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->bt_checkbox)));

    nm_settings_save(settings);
}

void
nm_internet_setup_fill_widget(NmInternetSetup *self)
{
    NmInternetSetupPrivate *priv = NM_INTERNET_SETUP_GET_PRIVATE (self);
    NmSettings *settings = nm_settings_new();

    gchar *interface = nm_settings_get_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                                              NM_INTERNET_INTERFACE);
    gchar *dns = nm_settings_get_value (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
                                              NM_INTERNET_DNS);

    gboolean usb = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
											NM_INTERNET_IN_INTERFACE_USB);
    gboolean wlan = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
											NM_INTERNET_IN_INTERFACE_WLAN);
    gboolean bt = nm_settings_get_boolean (settings, priv->env, NM_INTERNET_SETTINGS_GROUP,
											NM_INTERNET_IN_INTERFACE_BT);

    gtk_entry_set_text (GTK_ENTRY (priv->dns_entry), dns);
    gtk_widget_set_sensitive(priv->dns_hbox, TRUE);

    gtk_widget_set_sensitive(priv->wlan_checkbox, TRUE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->wlan_checkbox), wlan);
    gtk_widget_set_sensitive(priv->bt_checkbox, TRUE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->bt_checkbox), bt);
    gtk_widget_set_sensitive(priv->usb_checkbox, TRUE);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->usb_checkbox), usb);

    if (interface == NULL) {
        gtk_combo_box_set_active ( GTK_COMBO_BOX(priv->internet_combobox), 0 );
        gtk_widget_set_sensitive(priv->wlan_checkbox, FALSE);
        gtk_widget_set_sensitive(priv->dns_hbox, FALSE);
    }
    else {
		if (!strcmp(interface, NM_INTERNET_INTERFACE_BT)) {
			gtk_combo_box_set_active ( GTK_COMBO_BOX(priv->internet_combobox), 3 );
			gtk_widget_set_sensitive(priv->bt_checkbox, FALSE);
		}
		else if (!strcmp(interface, NM_INTERNET_INTERFACE_USB)) {
			gtk_combo_box_set_active ( GTK_COMBO_BOX(priv->internet_combobox), 2 );
			gtk_widget_set_sensitive(priv->usb_checkbox, FALSE);
		}
		else if (!strcmp(interface, NM_INTERNET_INTERFACE_WLAN_ADHOC)) {
			gtk_combo_box_set_active ( GTK_COMBO_BOX(priv->internet_combobox), 1 );
			gtk_widget_set_sensitive(priv->wlan_checkbox, FALSE);
		}
		else {
			gtk_combo_box_set_active ( GTK_COMBO_BOX(priv->internet_combobox), 0 );
			gtk_widget_set_sensitive(priv->wlan_checkbox, FALSE);
			gtk_widget_set_sensitive(priv->dns_hbox, FALSE);
		}
    }
}

static void
nm_internet_setup_combobox_changed (gpointer instance, gpointer self)
{
    NmInternetSetupPrivate *priv = NM_INTERNET_SETUP_GET_PRIVATE (self);

	gint active_checkbox = gtk_combo_box_get_active ( GTK_COMBO_BOX(priv->internet_combobox) );

	gtk_widget_set_sensitive(priv->dns_hbox, (active_checkbox == 0) ? FALSE : TRUE);
    gtk_widget_set_sensitive(priv->wlan_checkbox, TRUE);
    gtk_widget_set_sensitive(priv->bt_checkbox, TRUE);
    gtk_widget_set_sensitive(priv->usb_checkbox, TRUE);
    if (active_checkbox == 0 ||
        active_checkbox == 1) { // WLAN
        gtk_widget_set_sensitive(priv->wlan_checkbox, FALSE);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->wlan_checkbox), FALSE);
    }
    else if ( active_checkbox == 2 ) { // USB
        gtk_widget_set_sensitive(priv->usb_checkbox, FALSE);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->usb_checkbox), FALSE);
    }
    else if ( active_checkbox == 3 ) { // BT
        gtk_widget_set_sensitive(priv->bt_checkbox, FALSE);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->bt_checkbox), FALSE);
    }

}

