/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#include <stdlib.h>
/*#include <bt-dbus.h>*/

#include <glib-object.h>

#include "../nm_marshal.h"
#include "nm_bt_utils.h"

#ifndef DBUS_TYPE_G_DICTIONARY
#define DBUS_TYPE_G_DICTIONARY \
  (dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_VALUE))
#endif

#ifndef DBUS_TYPE_G_ARRAY
#define DBUS_TYPE_G_ARRAY \
  (dbus_g_type_get_collection("GPtrArray", DBUS_TYPE_G_OBJECT_PATH))
#endif

/* NmBtUtils signals */
enum {
    DEVICE_FOUND,
    REQUEST_PIN_CODE,
    DEVICE_PAIRED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

static DBusGConnection* nm_bt_utils_create_connection (NmBtUtils *self);

static void nm_bt_utils_device_found (DBusGProxy *object, gchar *addr,
									  GHashTable *hash, gpointer user_data);
static void nm_bt_utils_disconnect_discovery_signals (NmBtUtils *self);
static gboolean nm_bt_utils_connect_device (NmBtUtils *self, gchar *device_path);

static gchar* nm_bt_utils_get_device_path (NmBtUtils *self);
static gchar* nm_bt_utils_adapter (NmBtUtils *self);
static GHashTable *nm_bt_utils_get_properties (NmBtUtils *self, gchar *path, gchar *interface);
static GPtrArray *nm_bt_utils_list_devices (NmBtUtils *self, gchar *path);
static gchar *nm_bt_utils_find_device (NmBtUtils *self, gchar *path);
static void nm_bt_utils_create_device_callback (DBusGProxy *proxy,
										DBusGProxyCall *call, void *user_data);
static gboolean nm_bt_utils_create_device (NmBtUtils *self, gchar *path);
static gboolean nm_bt_utils_register_agent (NmBtUtils *self, gchar *path);


typedef struct _NmBtUtilsPrivate NmBtUtilsPrivate;

struct _NmBtUtilsPrivate
{
    DBusGConnection *connection;
    DBusGProxy *discovery_proxy;

    gchar *conn_name;
    gchar *agent_path;

    gchar *bt_addr;
    gchar *gw;
};

#define NM_BT_UTILS_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_BT_UTILS_TYPE, NmBtUtilsPrivate))

static void nm_bt_utils_class_init (NmBtUtilsClass *klass);
static void nm_bt_utils_init       (NmBtUtils *self);
static void nm_bt_utils_dispose    (GObject *object);
static void nm_bt_utils_finalize   (GObject *object);

G_DEFINE_TYPE (NmBtUtils, nm_bt_utils, G_TYPE_OBJECT)

static DBusGConnection*
nm_bt_utils_create_connection (NmBtUtils *self)
{
    DBusGConnection *connection;

    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (!connection) {
        g_warning ("D-Bus Connection not created");
    }

    return connection;
}

/************************************************************************
 Tool methods to start d-bus network connection.
************************************************************************/

gboolean
nm_bt_utils_connect (NmBtUtils *self, gchar *bt_addr, const gchar *gw)
{
    NmBtUtilsPrivate *priv;
    gchar *device_path = NULL;
    gboolean ret = FALSE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    g_free (priv->bt_addr);
    priv->bt_addr = g_strdup (bt_addr);
    g_free (priv->gw);
    priv->gw = g_strdup (gw);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if (nm_bt_utils_has_network_connection(self)) {

        g_warning ("[%s] NM already connected", __FUNCTION__);
        return ret;
    }

    if (!(device_path = nm_bt_utils_get_device_path (self))) {

    	g_debug ("[%s] NM pairing device", __FUNCTION__);
        return ret;
    }

    ret = nm_bt_utils_connect_device(self, device_path);
    g_free(device_path);
    return ret;
}

static gboolean
nm_bt_utils_connect_device (NmBtUtils *self, gchar *device_path)
{
    NmBtUtilsPrivate *priv;
    gboolean ret = FALSE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    ret = (system(g_strconcat("sudo /usr/sbin/nm-bt-connect ", device_path, " ",
							 priv->gw, NULL)) == 0);

    return ret;
}

/**
 * Verifies if the bt network connection is already set.
 */
gboolean
nm_bt_utils_has_network_connection (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    gchar *adapter = NULL;
    GPtrArray *list_devices = NULL;
	GHashTable *properties = NULL;
	gboolean connection = FALSE;
	GValue *value = NULL;
	gint i = 0;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if(!(adapter = nm_bt_utils_adapter(self))) {

    	return connection;
    }

	if (!(list_devices = nm_bt_utils_list_devices(self, adapter))) {

		g_free(adapter);
		return connection;
	}

	for (i = 0; i < list_devices->len && connection == FALSE; i++) {

		if (!(properties = nm_bt_utils_get_properties(self,
			g_ptr_array_index(list_devices, i), DBUS_INTERFACE_BLUEZ_NETWORK))) {

			continue;
		}

		value = g_hash_table_lookup(properties, "Connected");
		if (value && g_value_get_boolean(value)) {

			connection = TRUE;
		}
		g_hash_table_destroy(properties);
	}

	g_ptr_array_free(list_devices, TRUE);
    g_free(adapter);
    g_debug ("[%s] NM found the BT connection %d", __FUNCTION__, connection);
    return connection;
}

gchar*
nm_bt_utils_get_network_connection_end_point (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    gchar *adapter = NULL;
    GPtrArray *list_devices = NULL;
	GHashTable *net_properties = NULL;
	GHashTable *dev_properties = NULL;
	gchar *end_point = NULL;
	GValue *value = NULL;
	gint i = 0;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, NULL);
    }

    if(!(adapter = nm_bt_utils_adapter(self))) {

    	return end_point;
    }

    if (!(list_devices = nm_bt_utils_list_devices(self, adapter))) {

    	g_free(adapter);
    	return end_point;
    }

	for (i = 0; i < list_devices->len && end_point == NULL; i++) {

		if (!(net_properties = nm_bt_utils_get_properties(self,
			g_ptr_array_index(list_devices, i), DBUS_INTERFACE_BLUEZ_NETWORK))) {

			continue;
		}

		value = g_hash_table_lookup(net_properties, "Connected");
		if (value && g_value_get_boolean(value)) {

			if (!(dev_properties = nm_bt_utils_get_properties(self,
				g_ptr_array_index(list_devices, i),	DBUS_INTERFACE_BLUEZ_DEVICE))) {

				g_hash_table_destroy(net_properties);
				continue;
			}

			value = g_hash_table_lookup(dev_properties, "Name");
			if (value) {

				end_point = g_strdup(g_value_get_string(value));
			}
			g_hash_table_destroy(dev_properties);
		}
		g_hash_table_destroy(net_properties);
	}

	g_ptr_array_free(list_devices, TRUE);
   	g_free(adapter);
    g_debug ("[%s] Network connection found with %s", __FUNCTION__, end_point);
    return end_point;
}

static GPtrArray *
nm_bt_utils_list_devices (NmBtUtils *self, gchar *path)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    GPtrArray *array = NULL;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
										path, DBUS_INTERFACE_BLUEZ_ADP);

	if (!dbus_g_proxy_call(proxy, BLUEZ_LIST_DEVICES, &error, G_TYPE_INVALID,
						   DBUS_TYPE_G_ARRAY, &array, G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
        g_error_free (error);
	}

	g_object_unref (proxy);

	return array;
}

static gchar *
nm_bt_utils_find_device (NmBtUtils *self, gchar *path)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    gchar *device = NULL;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
									   path, DBUS_INTERFACE_BLUEZ_ADP);

	if (!dbus_g_proxy_call(proxy, BLUEZ_FIND_DEVICE, &error,
			G_TYPE_STRING, priv->bt_addr, G_TYPE_INVALID,
			DBUS_TYPE_G_OBJECT_PATH, &device, G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
        g_error_free (error);
	}

	g_object_unref (proxy);

	return device;
}

static void
nm_bt_utils_create_device_callback (DBusGProxy *proxy,
					DBusGProxyCall *call, void *user_data)
{
	NmBtUtils *self = user_data;
    NmBtUtilsPrivate *priv;
	gboolean device_created = FALSE;
    GError *error = NULL;
	gchar *path = NULL;

	priv = NM_BT_UTILS_GET_PRIVATE (self);
	if (dbus_g_proxy_end_call(proxy, call, &error,
			DBUS_TYPE_G_OBJECT_PATH, &path, G_TYPE_INVALID)) {

		device_created = (path != NULL);
		g_free(path);
	}
	else {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
		g_error_free(error);
	}

    g_signal_emit (user_data, signals[DEVICE_PAIRED], 0, device_created, NULL);
	g_object_unref(proxy);
}

static gboolean
nm_bt_utils_create_device (NmBtUtils *self, gchar *path)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    GError *error = NULL;
    gboolean ret = TRUE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
									   path, DBUS_INTERFACE_BLUEZ_ADP);

	nm_bt_utils_register_agent(self, path);

	if (!dbus_g_proxy_begin_call_with_timeout(proxy,
			BLUEZ_CREATE_PAIRED_DEVICE, nm_bt_utils_create_device_callback,
			self, NULL, 120 * 1000,
			G_TYPE_STRING, priv->bt_addr,
			DBUS_TYPE_G_OBJECT_PATH, priv->agent_path,
			G_TYPE_STRING, "DisplayOnly", G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
		g_error_free (error);
		ret = FALSE;
	}

	return ret;
}

static gboolean
nm_bt_utils_register_agent (NmBtUtils *self, gchar *path)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *adapter = NULL;
    GError *error = NULL;
    GObject *object = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

	adapter = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
									   path, DBUS_INTERFACE_BLUEZ_ADP);

    g_free(priv->agent_path);
    priv->agent_path = g_strdup_printf("/org/bluez/agent/%s",
							g_path_get_basename(dbus_g_proxy_get_path(adapter)));

	object = dbus_g_connection_lookup_g_object(priv->connection, priv->agent_path);

	if (object != NULL) {

		return TRUE;
	}

	dbus_g_connection_register_g_object(priv->connection,
									priv->agent_path, G_OBJECT(self));

	if (!dbus_g_proxy_call(adapter, BLUEZ_REGISTER_AGENT, &error,
			DBUS_TYPE_G_OBJECT_PATH, priv->agent_path,
			G_TYPE_STRING, "DisplayOnly",
			G_TYPE_INVALID, G_TYPE_INVALID)) {

		g_warning ("Agent registration failed: %s\n", error->message);
		g_error_free(error);
	}

	g_object_unref(adapter);
	return TRUE;

}

gboolean
nm_bt_utils_is_paired (NmBtUtils *self, gchar *bt_addr)
{
    NmBtUtilsPrivate *priv;
    gboolean ret = FALSE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
    g_free (priv->bt_addr);
    priv->bt_addr = g_strdup (bt_addr);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

	ret = (nm_bt_utils_get_device_path(self) != NULL);

	return ret;
}

gboolean
nm_bt_utils_pair_device (NmBtUtils *self, gchar *bt_addr)
{
    NmBtUtilsPrivate *priv;
    gchar *adapter = NULL;
    gboolean ret = FALSE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
    g_free (priv->bt_addr);
    priv->bt_addr = g_strdup (bt_addr);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if(!(adapter = nm_bt_utils_adapter(self))) {

    	return ret;
    }

	ret = nm_bt_utils_create_device(self, adapter);
	g_free(adapter);
	return ret;
}

static GHashTable *
nm_bt_utils_get_properties (NmBtUtils *self, gchar *path, gchar *interface)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
	GHashTable *hash = NULL;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
										path, interface);

	if (!dbus_g_proxy_call (proxy, BLUEZ_GET_PROPERTIES, &error, G_TYPE_INVALID,
							DBUS_TYPE_G_DICTIONARY, &hash, G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
		g_error_free (error);
	}
	g_object_unref (proxy);

	return hash;
}

static gchar *
nm_bt_utils_get_device_path (NmBtUtils *self)
{
    gchar *adapter = NULL;
	gchar *device_path = NULL;

    if(!(adapter = nm_bt_utils_adapter(self))) {

    	return device_path;
    }

   	device_path = nm_bt_utils_find_device(self, adapter);

    g_free(adapter);
    return device_path;
}

static gchar *
nm_bt_utils_adapter (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy = NULL;
    gchar *adapter = NULL;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
									   "/", DBUS_INTERFACE_BLUEZ);

	if (!dbus_g_proxy_call(proxy, BLUEZ_DEFAULT_ADAPTER, &error, G_TYPE_INVALID,
						   DBUS_TYPE_G_OBJECT_PATH, &adapter, G_TYPE_INVALID)) {

        if (error->domain == DBUS_GERROR &&
			error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

			g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
						dbus_g_error_get_name (error),
						error->message, __FILE__, __LINE__);
		} else {

			g_warning ("Error: %s\n\t%s: %d\n", error->message,
						__FILE__, __LINE__);
		}
		g_error_free (error);
    }

    return adapter;
}

gboolean
nm_bt_utils_remove_connection (NmBtUtils *self)
{
	system("sudo /usr/bin/killall nm-bt-connect");

	nm_bt_utils_ifdown();
    return TRUE;
}


/************************************************************************
 Tool methods to search for bluetooth devices
************************************************************************/

gboolean
nm_bt_utils_start_device_discovery (NmBtUtils *self)
{
    GError *error = NULL;
    NmBtUtilsPrivate *priv;
    gchar *adapter = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if (!nm_bt_utils_radio_is_enabled(self)) {

        nm_bt_utils_radio_enable (self, TRUE);
    }

    if (!(adapter = nm_bt_utils_adapter(self))) {

        return FALSE;
    }

    priv->discovery_proxy = dbus_g_proxy_new_for_name (priv->connection,
                         DBUS_SERVICE_BLUEZ, adapter, DBUS_INTERFACE_BLUEZ_ADP);

    dbus_g_object_register_marshaller(nm_marshal_VOID__STRING_STRING,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_object_register_marshaller(nm_marshal_VOID__STRING_BOXED,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_BOXED, G_TYPE_INVALID);

    dbus_g_proxy_add_signal(priv->discovery_proxy, BLUEZ_DEVICE_FOUND,
                         G_TYPE_STRING, DBUS_TYPE_G_DICTIONARY, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(priv->discovery_proxy,
								 BLUEZ_DEVICE_FOUND,
                                 G_CALLBACK(nm_bt_utils_device_found),
                                 self, NULL);

    dbus_g_proxy_call(priv->discovery_proxy, BLUEZ_START_DISCOVERY, &error,
                       G_TYPE_INVALID, G_TYPE_INVALID);
    if (error != NULL) {

        g_warning ("Failed to start discovery: %s\n", error->message);

        nm_bt_utils_disconnect_discovery_signals (self);

        g_free(adapter);
        g_error_free(error);
        g_object_unref (priv->discovery_proxy);
        priv->discovery_proxy = NULL;

        return FALSE;
    }

    g_free(adapter);
    dbus_g_connection_flush (priv->connection);

    return TRUE;
}

gboolean
nm_bt_utils_stop_device_discovery (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    GError *error = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);
    g_return_val_if_fail (priv->connection != NULL, FALSE);

    if (priv->discovery_proxy) {

        nm_bt_utils_disconnect_discovery_signals (self);

        dbus_g_proxy_call (priv->discovery_proxy, BLUEZ_STOP_DISCOVERY,
                            &error, G_TYPE_INVALID, G_TYPE_INVALID);

        if (error != NULL) {

            g_warning ("Failed to stop discovery: %s\n", error->message);
            g_error_free(error);
        }

        nm_bt_utils_disconnect_discovery_signals (self);

        g_object_unref (priv->discovery_proxy);
        priv->discovery_proxy = NULL;
    }

    dbus_g_connection_flush (priv->connection);
    return TRUE;
}

gboolean
nm_bt_utils_radio_is_enabled (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;
    gboolean res = FALSE;
	GHashTable *hash = NULL;
	gchar *adapter = NULL;
	GValue *value = NULL;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if (!(adapter = nm_bt_utils_adapter(self))) {

    	return res;
    }

	if ((hash = nm_bt_utils_get_properties(self, adapter,
										   DBUS_INTERFACE_BLUEZ_ADP))) {

		value = g_hash_table_lookup(hash, "Discoverable");
		if (value) {

			res = g_value_get_boolean(value);
		}
		g_hash_table_destroy(hash);
	}

	g_free(adapter);
    g_debug ("[%s] Radio is enabled? %s", __FUNCTION__, res ? "TRUE" : "FALSE");
    return res;
}

gboolean
nm_bt_utils_radio_enable (NmBtUtils *self, gboolean enable)
{
    NmBtUtilsPrivate *priv;
    DBusGProxy *proxy;
    GError *error = NULL;
    GValue discoverable = {0};
	gchar *adapter = NULL;
    gboolean res = FALSE;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if (!(adapter = nm_bt_utils_adapter(self))) {

    	return res;
    }

	proxy = dbus_g_proxy_new_for_name (priv->connection, DBUS_SERVICE_BLUEZ,
									   adapter, DBUS_INTERFACE_BLUEZ_ADP);

	g_value_init(&discoverable, G_TYPE_BOOLEAN);
	if (enable) {

		g_value_set_boolean(&discoverable, TRUE);
	}
	else {

		g_value_set_boolean(&discoverable, FALSE);
	}

	res = dbus_g_proxy_call (proxy, BLUEZ_SET_PROPERTY, &error,
							 G_TYPE_STRING, "Discoverable",
							 G_TYPE_VALUE, &discoverable,
							 G_TYPE_INVALID, G_TYPE_INVALID);
	if (!res) {

		g_warning ("[%s] Failed to enable BT radio", __FUNCTION__);
		g_error_free (error);
	}

	g_free(adapter);
	g_object_unref (proxy);

    return res;
}

gchar *
nm_bt_utils_check_pand (NmBtUtils *self, gchar *addr)
{
    NmBtUtilsPrivate *priv;
    gchar *adapter = NULL;
    gchar *device = NULL;
	GHashTable *properties = NULL;
	GValue *value = NULL;
	gboolean has_pan = FALSE;
	gboolean has_gn = FALSE;
	gint j = 0;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    g_free (priv->bt_addr);
    priv->bt_addr = g_strdup (addr);

    if (!priv->connection) {

        priv->connection = nm_bt_utils_create_connection (self);
        g_return_val_if_fail (priv->connection != NULL, FALSE);
    }

    if(!(adapter = nm_bt_utils_adapter(self))) {

    	return NULL;
    }

	if (!(device = nm_bt_utils_find_device(self, adapter))) {

		g_free(adapter);
		return NULL;
	}

	if (!(properties = nm_bt_utils_get_properties(self,
										device, DBUS_INTERFACE_BLUEZ_DEVICE))) {

		g_free(adapter);
		g_free(device);
		return NULL;
	}

	value = g_hash_table_lookup(properties, "UUIDs");
	if (value) {

		char **uuids = (char **)g_value_get_boxed (value);

		while (uuids[j] && has_pan == FALSE && has_gn == FALSE) {

			has_pan = (has_pan == TRUE) ? has_pan :
				(g_ascii_strcasecmp(BLUEZ_NETWORK_PAN_SERVICE_UUID,	uuids[j]) == 0);
			has_gn = (has_gn == TRUE) ? has_gn :
				(g_ascii_strcasecmp(BLUEZ_NETWORK_GN_SERVICE_UUID, uuids[j]) == 0);
			j++;
		}
	}

	g_hash_table_destroy(properties);
	g_free(device);
   	g_free(adapter);
    return (has_pan == TRUE || has_gn == TRUE) ? addr : NULL;
}

static void
nm_bt_utils_device_found (DBusGProxy *object, gchar *addr, GHashTable *hash,
                           gpointer user_data)
{
    GValue *value = g_hash_table_lookup(hash, "Name");
    gchar *name = (value != NULL) ? (gchar *)g_value_get_string(value) : NULL;

	g_debug ("[%s] Device found: %s, %s", __FUNCTION__, addr, name);
    g_signal_emit (user_data, signals[DEVICE_FOUND], 0, name, addr, NULL);
}

static void
nm_bt_utils_disconnect_discovery_signals (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    g_return_if_fail (priv->discovery_proxy != NULL);

    dbus_g_proxy_disconnect_signal (priv->discovery_proxy,
									 BLUEZ_DEVICE_FOUND,
                                     G_CALLBACK (nm_bt_utils_device_found),
                                     self);
}

/************************************************************************
 Tool methods to up and down the interfaces
************************************************************************/

gboolean
nm_bt_utils_ifup (const gchar *gw)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("sudo /sbin/ifup bnep0");

/*    system (g_strconcat ("sudo /sbin/ip route del default via ", gw,*/
/*                          " dev bnep0", NULL));*/
/*    system (g_strconcat ("sudo /sbin/ip route append default via ", gw,*/
/*                          " dev bnep0", NULL));*/

    return status == 0;
}

gboolean
nm_bt_utils_ifdown (void)
{
    // FIXME: replace bnep0 by the file retrieved from dbus
    gint status = system("sudo /sbin/ifdown bnep0");
    return status == 0;
}

/************************************************************************
 Agent methods
************************************************************************/

static gboolean
nm_bt_utils_request_pin_code(NmBtUtils *self,
			const char *path, DBusGMethodInvocation *context)
{
	gchar *target_pincode = g_strdup_printf("%d", g_random_int_range(1000, 9999));

	dbus_g_method_return(context, target_pincode);
    g_signal_emit (self, signals[REQUEST_PIN_CODE], 0, target_pincode, NULL);
    g_free(target_pincode);
	return TRUE;
}

static gboolean
nm_bt_utils_request_passkey(NmBtUtils *self,
			const char *path, DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_display_passkey(NmBtUtils *self,
			const char *path, guint passkey, guint8 entered,
						DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_request_confirmation(NmBtUtils *self,
					const char *path, guint passkey,
					DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_authorize(NmBtUtils *self,
					const char *path, const char *uuid,
						DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_confirm_mode(NmBtUtils *self,
			const char *mode, DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_cancel(NmBtUtils *self,
						DBusGMethodInvocation *context)
{
	return FALSE;
}

static gboolean
nm_bt_utils_release(NmBtUtils *self,
						DBusGMethodInvocation *context)
{
	return FALSE;
}

#include "nm_agent_glue.h"

/************************************************************************
 Initialization methods
************************************************************************/

static void
nm_bt_utils_class_init (NmBtUtilsClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmBtUtilsPrivate));

    signals[DEVICE_FOUND] = g_signal_new ("device-found",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                    0, NULL, NULL,
                    nm_marshal_VOID__STRING_STRING,
                    G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING, NULL);

    signals[REQUEST_PIN_CODE] = g_signal_new ("request-pin-code",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                    0, NULL, NULL,
                    g_cclosure_marshal_VOID__STRING,
                    G_TYPE_NONE, 1, G_TYPE_STRING, NULL);

    signals[DEVICE_PAIRED] = g_signal_new ("device-paired",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
                    0, NULL, NULL,
                    g_cclosure_marshal_VOID__BOOLEAN,
                    G_TYPE_NONE, 1, G_TYPE_BOOLEAN, NULL);

    object_class->dispose = nm_bt_utils_dispose;
    object_class->finalize = nm_bt_utils_finalize;

	dbus_g_object_type_install_info(NM_BT_UTILS_TYPE,
									&dbus_glib_nm_bt_utils_object_info);
}

static void
nm_bt_utils_init (NmBtUtils *self)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (self);

    priv->connection = NULL;
}

static void
nm_bt_utils_dispose (GObject *object)
{
    NmBtUtilsPrivate *priv;

    priv = NM_BT_UTILS_GET_PRIVATE (object);

    if (priv->connection) {
        dbus_g_connection_unref (priv->connection);
    }

    G_OBJECT_CLASS (nm_bt_utils_parent_class)->dispose (object);
}

static void
nm_bt_utils_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_bt_utils_parent_class)->finalize (object);
}

NmBtUtils*
nm_bt_utils_new (void)
{
    return g_object_new (NM_BT_UTILS_TYPE, NULL);
}

