/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <string.h>
#include <hildon/hildon.h>

#include "merger-window.h"
#include "debug.h"
#include "matchcell.h"
#include "merger.h"

enum
{
    PROP_NONE,
    PROP_AGGREGATOR
};

enum
{
    COL_MATCH
};

struct _MergerWindowPrivate
{
    OssoABookAggregator *aggregator;
    GList *error_details;

    GtkWidget *toolbar;

    GtkWidget *main_area;
    GtkWidget *info_label;

    GtkListStore *store;
    GtkWidget *selector;
    GtkWidget *tree_view;

    GtkWidget *finished_dialog;

    GtkTreePath *tap_and_hold_path;
};

#define MERGER_WINDOW_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), \
                                  TYPE_MERGER_WINDOW, \
                                  MergerWindowPrivate))

#define RESPONSE_DETAILS 1

G_DEFINE_TYPE (MergerWindow, merger_window, HILDON_TYPE_STACKABLE_WINDOW)

GtkWidget *
merger_window_new (OssoABookAggregator *aggregator)
{
    return g_object_new (TYPE_MERGER_WINDOW, "aggregator", aggregator, NULL);
}

static void
arrow_clicked_cb (HildonEditToolbar *toolbar,
                  gpointer           user_data)
{
    MergerWindow *self = user_data;

    gtk_widget_destroy (GTK_WIDGET (self));
}

static GList*
merger_window_get_selection (MergerWindow *self)
{
    GList *rows;
    GList *selection = NULL;

    g_return_val_if_fail (self->priv->selector, NULL);

    rows = hildon_touch_selector_get_selected_rows (
            HILDON_TOUCH_SELECTOR (self->priv->selector), 0);
    while (rows) {
        GtkTreePath *path = rows->data;
        GtkTreeIter iter;
        Match *match;

        gtk_tree_model_get_iter (GTK_TREE_MODEL (self->priv->store),
                &iter, path);
        gtk_tree_model_get (GTK_TREE_MODEL (self->priv->store), &iter,
                COL_MATCH, &match, -1);
        selection = g_list_prepend (selection, match);

        gtk_tree_path_free (path);
        rows = g_list_delete_link (rows, rows);
    }

    return selection;
}

static void
selection_changed_cb (HildonTouchSelector *selector,
                      gint                 column,
                      gpointer             user_data)
{
    if (hildon_touch_selector_get_active (selector, column) != -1)
        hildon_touch_selector_set_active (selector, column, -1);
}

static void
error_details_delete_even_cb (GtkWidget *dialog,
                              GdkEvent  *event,
                              MergerWindow *self)
{
    gtk_widget_destroy (dialog);
    gtk_widget_show (GTK_WIDGET (self->priv->finished_dialog));
}

static void
error_details_dialog_show (MergerWindow *self)
{
    GtkWidget *selector;
    GList *l;
    GtkWidget *dialog;

    selector = hildon_touch_selector_new_text ();
    g_signal_connect (selector, "changed", G_CALLBACK (selection_changed_cb),
            NULL);
    for (l = self->priv->error_details; l; l = l->next)
        hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
               l->data);

    gtk_widget_hide (GTK_WIDGET (self->priv->finished_dialog));

    dialog = hildon_picker_dialog_new (GTK_WINDOW (self));
    gtk_window_set_title (GTK_WINDOW (dialog),
            _("Contacts that were not merged"));
    hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (dialog),
            HILDON_TOUCH_SELECTOR (selector));
    gtk_widget_show (GTK_WIDGET (dialog));
    g_signal_connect (dialog, "delete-event",
            G_CALLBACK (error_details_delete_even_cb), self);

    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, -1);
}

static void
finished_dialog_response_cb (GtkDialog    *dialog,
                             gint          response_id,
                             MergerWindow *self)
{
    if (response_id == RESPONSE_DETAILS) {
        error_details_dialog_show (self);
    } else {
        gtk_widget_destroy (GTK_WIDGET (dialog));
        gtk_widget_destroy (GTK_WIDGET (self));
    }
}

static void
finished_note_response_cb (GtkDialog    *note,
                           int           response,
                           MergerWindow *self)
{
    gtk_widget_destroy (GTK_WIDGET (note));
    gtk_widget_destroy (GTK_WIDGET (self));
}

static void
merge_finished_cb (gint      merged_count,
                   gint      failed_count,
                   GList    *failed,
                   gpointer  user_data)
{
    MergerWindow *self = user_data;
    GList *l, *m;
    gchar *msg;

    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 0);

    if (!failed_count) {
        /* All contacts merged */
        gchar *text;
        GtkWidget *note;

        text = g_strdup_printf (_("%d contacts merged"), merged_count);
        note = hildon_note_new_information (GTK_WINDOW (self), text);
        g_signal_connect (note, "response",
                G_CALLBACK (finished_note_response_cb), self);
        gtk_widget_show (note);

        g_free (text);

        return;
    }

    /* At least one contact failed */

    for (l = failed; l; l = l->next) {
        GString *str = g_string_new ("");
        for (m = l->data; m; m = m->next) {
            g_string_append (str,
                    osso_abook_contact_get_display_name (m->data));
            if (m->next)
                g_string_append (str, _(", "));
        }
        self->priv->error_details = g_list_prepend (self->priv->error_details,
                g_string_free (str, FALSE));
    }

    if (merged_count == 0)
        msg = g_strdup (_("The contacts were not merged because an error "
                "occurred or because you cancelled the merge.\n"
                "Press \"Error details\" for details."));
    else
        msg = g_strdup_printf (_("%d contacts were merged, but %d were not "
                "merged because an error occurred or because you cancelled "
                "the merge.\n"
                "Press \"Error details\" for details."),
                merged_count, failed_count);

    self->priv->finished_dialog = hildon_note_new_confirmation_add_buttons (
            GTK_WINDOW (self), msg, NULL);
    if (failed_count)
        gtk_dialog_add_button (GTK_DIALOG (self->priv->finished_dialog),
                _("Error details"), RESPONSE_DETAILS);
    gtk_dialog_add_button (GTK_DIALOG (self->priv->finished_dialog),
            _("Ok"), GTK_RESPONSE_ACCEPT);
    g_signal_connect (self->priv->finished_dialog, "response",
            G_CALLBACK (finished_dialog_response_cb), self);

    gtk_widget_show (self->priv->finished_dialog);

    g_free (msg);
}

static void
merger_window_show_error (MergerWindow *self,
                          const gchar  *msg)
{
    GtkWidget *label;
    gchar *markup;

    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 0);

    if (self->priv->main_area) {
        gtk_widget_destroy (self->priv->main_area);
        self->priv->main_area = NULL;
        self->priv->info_label = NULL;
        self->priv->selector = NULL;
    }

    g_return_if_fail (gtk_bin_get_child (GTK_BIN (self)) == NULL);

    label = gtk_label_new (NULL);
    markup = g_strdup_printf ("<big>%s</big>", msg);
    gtk_label_set_markup (GTK_LABEL (label), markup);
    g_free (markup);

    gtk_widget_show (label);
    gtk_container_add (GTK_CONTAINER (self), label);
}

static void
main_area_size_allocate_cb (GtkWidget     *main_area,
                            GtkAllocation *allocation,
                            MergerWindow  *self)
{
    GtkRequisition requisition;

    g_signal_handlers_disconnect_by_func (main_area,
            G_CALLBACK (main_area_size_allocate_cb), self);

    gtk_widget_set_size_request (self->priv->info_label,
            allocation->width - HILDON_MARGIN_DEFAULT * 3,
            -1);

    hildon_touch_selector_optimal_size_request (
            HILDON_TOUCH_SELECTOR (self->priv->selector),
            &requisition);
    gtk_widget_set_size_request (self->priv->selector,
            requisition.width, requisition.height);
}

static void
accounts_manager_ready_cb (OssoABookAccountManager *manager,
                           const GError            *error_in,
                           gpointer                 user_data)
{
    MergerWindow *self = user_data;
    GList *contacts;
    GList *matches;
    GList *l;
    GtkTreeIter tree_iter;

    if (error_in) {
        g_printerr ("Error while waiting for the accounts manager to be ready: "
                "%s\n", error_in->message);
        merger_window_show_error (self, _("Error while loading IM info"));
        return;
    }

    DEBUG ("Accounts manager ready");

    contacts = osso_abook_aggregator_list_master_contacts (
            self->priv->aggregator);
    matches = generate_merge_suggestions (contacts,
        (GList *) osso_abook_account_manager_get_primary_vcard_fields (manager));

    if (debug_is_enabled ()) {
        for (l = matches; l; l = l->next) {
            Match *m = l->data;
            OssoABookContact *c1;
            OssoABookContact *c2;

            match_get_contacts (m, &c1, &c2);
            DEBUG ("  %s (uid: %s) and %s (uid: %s) with score %d",
                osso_abook_contact_get_display_name (c1),
                osso_abook_contact_get_uid (c1),
                osso_abook_contact_get_display_name (c2),
                osso_abook_contact_get_uid (c2),
                match_get_score (m));
            DEBUG ("    %s", match_get_description (m));
        }
    }

    for (l = matches; l; l = l->next) {
        gtk_list_store_append (self->priv->store, &tree_iter);
        gtk_list_store_set (self->priv->store, &tree_iter, COL_MATCH,
                l->data, -1);
    }

    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 0);

    if (matches) {
        gchar *text;

        gtk_widget_show (self->priv->toolbar);
        gtk_window_fullscreen (GTK_WINDOW (self));

        text = g_strdup_printf (_(
                "%d possible duplicates have been found. "
                "Select the ones that actually match and press “Merge” to "
                "merge them.\n"
                "Tap and hold to see more details about the contacts."),
                g_list_length (matches));
        gtk_label_set_text (GTK_LABEL (self->priv->info_label), text);
        gtk_widget_show (self->priv->info_label);

        /* Is there any better way to do this? */
        g_signal_connect (self->priv->main_area, "size-allocate",
                G_CALLBACK (main_area_size_allocate_cb), self);
        gtk_widget_queue_resize (self->priv->main_area);

        g_free (text);
    } else {
        merger_window_show_error (self, _("No duplicate contacts found"));
    }

    match_list_free (matches);
    g_list_free (contacts);
}

static void
aggregator_ready_cb (OssoABookWaitable *aggregator,
                     const GError      *error_in,
                     gpointer           user_data)
{
    MergerWindow *self = user_data;

    if (error_in) {
        g_printerr ("Error while waiting for the aggregator to be ready: "
                "%s\n", error_in->message);
        merger_window_show_error (self, _("Error while loading contacts"));
        return;
    }

    DEBUG ("Aggregator ready");

    osso_abook_account_manager_call_when_ready (NULL,
        accounts_manager_ready_cb, self, NULL);
}

static void
merge_button_cb (HildonEditToolbar *toolbar,
                 gpointer           user_data)
{
    MergerWindow *self = user_data;
    GList *selection;

    selection = merger_window_get_selection (self);

    if (!selection) {
        hildon_banner_show_information (GTK_WIDGET (self), NULL,
                _("Select the contacts you want to merge"));
        return;
    }

    gtk_widget_set_sensitive (GTK_WIDGET (self), FALSE);
    gtk_widget_hide (self->priv->toolbar);
    gtk_window_unfullscreen (GTK_WINDOW (self));
    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 1);

    merge_contacts (selection, GTK_WINDOW (self), merge_finished_cb, self);

    match_list_free (selection);
}

static void
show_contact_starter (MergerWindow     *self,
                      OssoABookContact *contact)
{
    GtkWidget *starter;
    GtkWidget *dialog;

    starter = osso_abook_touch_contact_starter_new_not_interactive (
            NULL, contact);
    gtk_widget_show (starter);

    dialog = osso_abook_touch_contact_starter_dialog_new (GTK_WINDOW (self),
            OSSO_ABOOK_TOUCH_CONTACT_STARTER (starter));
    gtk_widget_show (dialog);
}

static Match *
get_match_for_popup (MergerWindow *self)
{
    GtkTreeIter iter;
    Match *match;

    g_return_val_if_fail (self->priv->tap_and_hold_path != NULL, NULL);

    if (!gtk_tree_model_get_iter (GTK_TREE_MODEL (self->priv->store),
                &iter, self->priv->tap_and_hold_path))
        return NULL;

    gtk_tree_model_get (GTK_TREE_MODEL (self->priv->store), &iter,
            COL_MATCH, &match, -1);

    return match;
}

static void
menu_show_contact1_cb (GtkMenuItem  *item,
                       MergerWindow *self)
{
    Match *match;
    OssoABookContact *contact1;

    match = get_match_for_popup (self);
    if (match) {
        match_get_contacts (match, &contact1, NULL);
        show_contact_starter (self, contact1);
        match_unref (match);
    }
}

static void
menu_show_contact2_cb (GtkMenuItem  *item,
                       MergerWindow *self)
{
    Match *match;
    OssoABookContact *contact2;

    match = get_match_for_popup (self);
    if (match) {
        match_get_contacts (match, NULL, &contact2);
        show_contact_starter (self, contact2);
        match_unref (match);
    }
}

static void
selector_tap_and_hold_cb (GtkWidget *widget,
                          gpointer   user_data)
{
    MergerWindow *self = user_data;
    Match *match;
    OssoABookContact *c1, *c2;
    const gchar *c1_name, *c2_name;
    gchar *text1, *text2;
    GtkWidget *menu;
    GtkWidget *item;

    match = get_match_for_popup (self);
    if (!match)
        return;

    match_get_contacts (match, &c1, &c2);
    c1_name = osso_abook_contact_get_display_name (c1);
    c2_name = osso_abook_contact_get_display_name (c2);
    if (strcmp (c1_name, c2_name) != 0) {
        text1 = g_strdup_printf ("Show %s", c1_name);
        text2 = g_strdup_printf ("Show %s", c2_name);
    } else {
        /* No point in showing the names if they are identical */
        text1 = g_strdup (_("Show first contact"));
        text2 = g_strdup (_("Show second contact"));
    }

    menu = hildon_gtk_menu_new ();

    item = gtk_menu_item_new_with_label (text1);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    g_signal_connect (item, "activate",
            G_CALLBACK (menu_show_contact1_cb), self);

    item = gtk_menu_item_new_with_label (text2);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    g_signal_connect (item, "activate",
            G_CALLBACK (menu_show_contact2_cb), self);

    gtk_widget_show_all (menu);
    gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 1,
            gtk_get_current_event_time ());

    match_unref (match);
    g_free (text1);
    g_free (text2);
}

static gboolean
selector_tap_and_hold_query_cb (GtkWidget *widget,
                                GdkEvent  *event,
                                gpointer   user_data)
{
    MergerWindow *self = user_data;
    gint x, y;
    GtkTreePath *path;

    g_signal_stop_emission_by_name (widget, "tap-and-hold-query");

    gtk_tree_view_convert_widget_to_bin_window_coords (
            GTK_TREE_VIEW (self->priv->tree_view),
            (gint) event->button.x, (gint) event->button.y,
            &x, &y);

    if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (self->priv->tree_view),
                x, y, &path, NULL, NULL, NULL)) {
        gtk_tree_path_free (self->priv->tap_and_hold_path);

        self->priv->tap_and_hold_path = path;

        /* Propagate the tap-and-hold signal */
        return FALSE;
    } else {
        /* Stop the emission of tap-and-hold */
        return TRUE;
    }
}

static void
delete_event_cb (GtkWidget *window,
                 GdkEvent  *event,
                 MergerWindow *self)
{
    gtk_widget_destroy (GTK_WIDGET (self));
}

static void
find_tree_view_cb (GtkWidget *widget,
                   gpointer   user_data)
{
    GtkWidget **tv_p = user_data;

    if (*tv_p != NULL)
        return;

    if (GTK_IS_TREE_VIEW (widget))
        *tv_p = widget;
    else if (GTK_IS_CONTAINER (widget))
        gtk_container_forall (GTK_CONTAINER (widget), find_tree_view_cb, tv_p);
}

static GtkWidget *
find_tree_view (GtkContainer *container)
{
    GtkWidget *tv = NULL;

    /* Sadly there is no proper way to get the underlying tree view from
     * a hildon touch selector, but we need it for the popup menu.
     * This makes me cry. Welcome to Hildon! */
    gtk_container_forall (container, find_tree_view_cb, &tv);

    return tv;
}

static void
merger_window_init (MergerWindow *self)
{
    GtkCellRenderer *cell;
    GtkWidget *info_label_box;
    GtkWidget *main_box;

    self->priv = MERGER_WINDOW_GET_PRIVATE (self);

    gtk_window_set_title (GTK_WINDOW (self), _("Select contacts to merge"));
    hildon_gtk_window_set_progress_indicator (GTK_WINDOW (self), 1);
    g_signal_connect (self, "delete-event", G_CALLBACK (delete_event_cb), self);

    self->priv->toolbar = hildon_edit_toolbar_new_with_text (
            _("Select contacts to merge"),
            _("Merge"));
    hildon_window_set_edit_toolbar (HILDON_WINDOW (self),
            HILDON_EDIT_TOOLBAR (self->priv->toolbar)); 
    g_signal_connect (self->priv->toolbar, "button-clicked",
            G_CALLBACK (merge_button_cb), self);
    g_signal_connect (self->priv->toolbar, "arrow-clicked",
            G_CALLBACK (arrow_clicked_cb), self);

    self->priv->info_label = gtk_label_new ("");
    gtk_label_set_line_wrap (GTK_LABEL (self->priv->info_label), TRUE);
    gtk_label_set_line_wrap_mode (GTK_LABEL (self->priv->info_label),
            PANGO_WRAP_WORD_CHAR);

    self->priv->store = gtk_list_store_new (1, TYPE_MATCH);
    self->priv->selector = hildon_touch_selector_new ();
    hildon_touch_selector_set_live_search (
            HILDON_TOUCH_SELECTOR (self->priv->selector), FALSE);
    cell = match_cell_renderer_new ();
    g_object_set (cell,
            "xalign", 0.0,
            "xpad", 6,
            NULL);
    hildon_touch_selector_append_column (
            HILDON_TOUCH_SELECTOR (self->priv->selector),
            GTK_TREE_MODEL (self->priv->store), cell, "match", COL_MATCH,
            NULL);
    hildon_touch_selector_set_column_selection_mode (
            HILDON_TOUCH_SELECTOR (self->priv->selector),
            HILDON_TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE);
    gtk_widget_show (self->priv->selector);

    self->priv->tree_view = find_tree_view (
            GTK_CONTAINER (self->priv->selector));
    gtk_widget_tap_and_hold_setup (self->priv->tree_view, NULL, NULL, 0);
    g_signal_connect (self->priv->tree_view, "tap-and-hold-query",
            G_CALLBACK (selector_tap_and_hold_query_cb), self);
    g_signal_connect (self->priv->tree_view, "tap-and-hold",
            G_CALLBACK (selector_tap_and_hold_cb), self);

    info_label_box = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (info_label_box), self->priv->info_label,
            TRUE, TRUE, HILDON_MARGIN_DEFAULT);
    gtk_widget_show (info_label_box);

    main_box = gtk_vbox_new (FALSE, HILDON_MARGIN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (main_box), info_label_box, FALSE, FALSE,
            HILDON_MARGIN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (main_box), self->priv->selector,
            TRUE, TRUE, 0);
    gtk_widget_show (main_box);

    self->priv->main_area = hildon_pannable_area_new ();
    hildon_pannable_area_add_with_viewport (
            HILDON_PANNABLE_AREA (self->priv->main_area), main_box);
    gtk_widget_show (self->priv->main_area);

    gtk_container_add (GTK_CONTAINER (self), self->priv->main_area);
} 

static void
merger_window_dispose (GObject *object)
{
    MergerWindow *self = MERGER_WINDOW (object);

    if (self->priv->aggregator) {
        g_object_unref (self->priv->aggregator);
        self->priv->aggregator = NULL;
    }

    gtk_tree_path_free (self->priv->tap_and_hold_path);
    self->priv->tap_and_hold_path = NULL;

    g_list_foreach (self->priv->error_details, (GFunc) g_free, NULL);
    g_list_free (self->priv->error_details);
    self->priv->error_details = NULL;

    G_OBJECT_CLASS (merger_window_parent_class)->dispose (object);
}

static void
merger_window_finalize (GObject *object)
{
    G_OBJECT_CLASS (merger_window_parent_class)->finalize (object);
}

static gboolean
load_aggregator_idle_cb (gpointer user_data)
{
    MergerWindow *self = user_data;

    osso_abook_waitable_call_when_ready (
            OSSO_ABOOK_WAITABLE (self->priv->aggregator),
            aggregator_ready_cb, self, NULL);

    return FALSE;
}

static void
merger_window_set_aggregator (MergerWindow        *self,
                              OssoABookAggregator *aggregator)
{
    g_return_if_fail (aggregator);

    self->priv->aggregator = g_object_ref (aggregator);

    /* If we are loaded as a plugin in the address book then the
     * aggregator is already ready. If we called _call_when_ready()
     * here the callback would be called immediately and the
     * duplicates loaded blocking the UI */
    g_idle_add (load_aggregator_idle_cb, self);
}

static void
merger_window_get_property (GObject    *object,
                            guint       param_id,
                            GValue     *value,
                            GParamSpec *pspec)
{
    MergerWindow *self = MERGER_WINDOW (object);

    switch (param_id) {
        case PROP_AGGREGATOR:
            g_value_set_object (value, self->priv->aggregator);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
merger_window_set_property (GObject      *object,
                            guint         param_id,
                            const GValue *value,
                            GParamSpec   *pspec)
{
    MergerWindow *self = MERGER_WINDOW (object);

    switch (param_id) {
        case PROP_AGGREGATOR:
            merger_window_set_aggregator (self, g_value_get_object (value));
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
merger_window_class_init (MergerWindowClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->dispose = merger_window_dispose;
    object_class->finalize = merger_window_finalize;
    object_class->get_property = merger_window_get_property;
    object_class->set_property = merger_window_set_property;

    g_object_class_install_property (
            object_class,
            PROP_AGGREGATOR,
            g_param_spec_object (
                "aggregator",
                "Aggregator",
                "The aggregator to use",
                OSSO_ABOOK_TYPE_AGGREGATOR,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (klass, sizeof (MergerWindowPrivate));
}
