#!/usr/bin/env python
# -*- coding: UTF8 -*-

from __future__ import with_statement
from __future__ import division

import os
import simplejson
import logging

from PyQt4 import QtCore
from PyQt4 import QtGui

from util import qwrappers
from util import misc as misc_utils

import constants
import config
import game_state
import game_ui
import dialogs


_moduleLogger = logging.getLogger(__name__)


class Omnom(qwrappers.ApplicationWrapper):

	def __init__(self, app):
		self._dataPath = os.path.join(os.path.dirname(__file__), "data")
		self._gameState = game_state.GameState(self._dataPath)
		self._aboutDialog = None
		qwrappers.ApplicationWrapper.__init__(self, app, constants)

	def load_settings(self):
		try:
			with open(constants._user_settings_, "r") as settingsFile:
				settings = simplejson.load(settingsFile)
		except IOError, e:
			_moduleLogger.info("No settings")
			settings = {}
		except ValueError:
			_moduleLogger.info("Settings were corrupt")
			settings = {}

		self._fullscreenAction.setChecked(settings.get("isFullScreen", False))

	def save_settings(self):
		settings = {
			"isFullScreen": self._fullscreenAction.isChecked(),
		}
		with open(constants._user_settings_, "w") as settingsFile:
			simplejson.dump(settings, settingsFile)

	@property
	def gameState(self):
		return self._gameState

	@property
	def dataPath(self):
		return self._dataPath

	def _new_main_window(self):
		return MainWindow(None, self)

	def _close_windows(self):
		if self._aboutDialog is not None:
			self._aboutDialog.close()
		qwrappers.ApplicationWrapper._close_windows(self)

	@misc_utils.log_exception(_moduleLogger)
	def _on_about(self, checked = True):
		if self._aboutDialog is None:
			self._aboutDialog = dialogs.AboutDialog(self)
		self._aboutDialog.run()


class MainWindow(qwrappers.WindowWrapper):

	def __init__(self, parent, app):
		qwrappers.WindowWrapper.__init__(self, parent, app)
		self._settingsDialog = None
		self._child = None
		self._gameState = self._app.gameState

		self._titleImagePixmap = QtGui.QPixmap(os.path.join(self._app.dataPath, "omnom", "title.png"))
		self._title = QtGui.QLabel()
		self._title.setPixmap(self._titleImagePixmap)

		self._singlePlayerButton = QtGui.QPushButton("Single Player")
		self._singlePlayerButton.clicked.connect(self._on_single_player)
		self._multiPlayerButton = QtGui.QPushButton("Multi-Player")
		self._multiPlayerButton.clicked.connect(self._on_multi_player)
		self._settingsButton = QtGui.QPushButton("Settings")
		self._settingsButton.clicked.connect(self._on_settings)
		self._butonLayout = QtGui.QHBoxLayout()
		self._butonLayout.addWidget(self._singlePlayerButton)
		self._butonLayout.addWidget(self._multiPlayerButton)
		self._butonLayout.addWidget(self._settingsButton)

		self._layout.addWidget(self._title, 100, QtCore.Qt.AlignCenter)
		self._layout.addLayout(self._butonLayout, 0)
		self._layout.setDirection(QtGui.QBoxLayout.TopToBottom)
		self._window.setWindowTitle("%s" % constants.__pretty_app_name__)
		self._window.setWindowIcon(QtGui.QIcon(os.path.join(self._app.dataPath, "omnom", "appIcon.png")))

		toolsMenu = self._window.menuBar().addMenu("&Tools")
		toolsMenu.addAction(self._app.aboutAction)

	def walk_children(self):
		if self._child is not None:
			return (self._child, )
		else:
			return ()

	def close(self):
		if self._settingsDialog is not None:
			self._settingsDialog.close()
		qwrappers.WindowWrapper.close(self)

	@misc_utils.log_exception(_moduleLogger)
	def _on_child_close(self, something = None):
		self._child = None

	@misc_utils.log_exception(_moduleLogger)
	def _on_single_player(self, checked = True):
		if self._child is not None:
			self._child.window.destroyed.disconnect(self._on_child_close)
			self._child.close()
		self._gameState.playerCount = 1
		self._child = PlayerCampaignSelector(self.window, self._app, self._gameState)
		self._child.show()
		self._child.window.destroyed.connect(self._on_child_close)

	@misc_utils.log_exception(_moduleLogger)
	def _on_multi_player(self, checked = True):
		if self._child is not None:
			self._child.window.destroyed.disconnect(self._on_child_close)
			self._child.close()
		self._gameState.playerCount = 2
		self._child = PlayerCampaignSelector(self.window, self._app, self._gameState)
		self._child.show()
		self._child.window.destroyed.connect(self._on_child_close)

	@misc_utils.log_exception(_moduleLogger)
	def _on_settings(self, checked = True):
		if self._settingsDialog is None:
			self._settingsDialog = dialogs.SettingsDialog(self._app)
		self._settingsDialog.musicEnabled = config.gameMusic
		self._settingsDialog.effectsEnabled = config.gameSound
		self._settingsDialog.playerLives = config.lives
		self._settingsDialog.playerSpeed = config.player_speed
		self._settingsDialog.enemySpeed = config.enemy_speed
		self._settingsDialog.run()
		config.gameMusic = self._settingsDialog.musicEnabled
		config.gameSound = self._settingsDialog.effectsEnabled
		config.lives = self._settingsDialog.playerLives
		config.player_speed = self._settingsDialog.playerSpeed
		config.enemy_speed = self._settingsDialog.enemySpeed


class PlayerCampaignSelector(qwrappers.WindowWrapper):

	def __init__(self, parent, app, gameState):
		qwrappers.WindowWrapper.__init__(self, parent, app)
		self._gameState = gameState
		self._child = None

		self._campaigns = self._gameState.supportedCampaigns
		self._selectedCampaign = None

		self._campaignSelector = game_ui.CampaignSelector(self._app, self._gameState)
		self._campaignSelector.buttonGroup.buttonClicked[int].connect(self._on_campaign_selected)

		self._levelSelector = game_ui.LevelSelector(self._app)
		self._levelSelector.toplevel.activated.connect(self._on_row_activated)

		self._layout.addWidget(self._campaignSelector.toplevel, 0)
		self._layout.addWidget(self._levelSelector.toplevel)
		self._window.setWindowTitle("Campaign")
		self._window.setWindowIcon(QtGui.QIcon(os.path.join(self._app.dataPath, "omnom", "appIcon.png")))

		toolsMenu = self._window.menuBar().addMenu("&Tools")
		toolsMenu.addAction(self._app.aboutAction)

	def set_orientation(self, isPortrait):
		qwrappers.WindowWrapper.set_orientation(self, isPortrait)
		if isPortrait:
			defaultLayoutOrientation = QtGui.QBoxLayout.TopToBottom
		else:
			defaultLayoutOrientation = QtGui.QBoxLayout.LeftToRight
		self._layout.setDirection(defaultLayoutOrientation)
		self._campaignSelector.set_orientation(isPortrait)

	def walk_children(self):
		if self._child is not None:
			return (self._child, )
		else:
			return ()

	def close(self):
		self._campaignSelector.close()
		qwrappers.WindowWrapper.close(self)

	@misc_utils.log_exception(_moduleLogger)
	def _on_child_close(self, something = None):
		self._child = None

	@misc_utils.log_exception(_moduleLogger)
	def _on_row_activated(self, index):
		row = index.row()
		levels = []
		try:
			i = row
			while True:
				levels.append(self._levelSelector.get_level(i))
				i += 1
		except IndexError:
			pass
		import level_ui
		levelView = level_ui.LevelView(self._app.dataPath)
		levelView.run(levels)

	@misc_utils.log_exception(_moduleLogger)
	def _on_campaign_selected(self, index):
		selectedCampaign = self._campaigns[index]
		self._levelSelector.load_campaign(selectedCampaign)


def run():
	app = QtGui.QApplication([])
	handle = Omnom(app)
	return app.exec_()


if __name__ == "__main__":
	import sys

	logFormat = '(%(relativeCreated)5d) %(levelname)-5s %(threadName)s.%(name)s.%(funcName)s: %(message)s'
	logging.basicConfig(level=logging.DEBUG, format=logFormat)
	try:
		os.makedirs(constants._data_path_)
	except OSError, e:
		if e.errno != 17:
			raise

	val = run()
	sys.exit(val)
