#!/usr/bin/env python

from __future__ import with_statement
from __future__ import division

import weakref
import logging

from PyQt4 import QtCore

from util import misc as misc_utils
import loaders


_moduleLogger = logging.getLogger(__name__)


class GameState(QtCore.QObject):

	playerCountChanged  = QtCore.pyqtSignal(int)
	supportedCampaignsChanged = QtCore.pyqtSignal()

	def __init__(self, dataPath):
		QtCore.QObject.__init__(self)
		self._dataPath = dataPath
		self._loader = loaders.GameLoader(self._dataPath)
		self._playerCount = -1
		self._campaigns = []
		self._supportedCampaigns = []

	@property
	def loader(self):
		return self._loader

	@property
	def supportedCampaigns(self):
		self._update_supported_campaigns()
		return self._supportedCampaigns

	@property
	def campaigns(self):
		self._update_supported_campaigns()
		return self._campaigns

	def _get_player_count(self):
		return self._playerCount

	def _set_player_count(self, newCount):
		oldCount = self._playerCount
		self._playerCount = newCount
		playerCountChanged = newCount != oldCount
		self._update_supported_campaigns(force=playerCountChanged)
		if playerCountChanged:
			self.playerCountChanged.emit(newCount)

	playerCount = property(_get_player_count, _set_player_count)

	def _update_campaigns(self, force=False):
		if self._campaigns and not force:
			return
		self._campaigns = [
			Campaign(self, campaign)
			for campaign in self._loader.get_campaigns()
		]

	def _update_supported_campaigns(self, force=False):
		if self._supportedCampaigns and not force:
			return
		self._update_campaigns()

		self._supportedCampaigns = []
		for campaign in self._campaigns:
			if self._is_campaign_supported(campaign):
				campaign.enable()
				self._supportedCampaigns.append(campaign)
			else:
				campaign.disable()

		self.supportedCampaignsChanged.emit()

	def _is_campaign_supported(self, campaign):
		return (self._playerCount in campaign.data.supportedPlayerCounts)


class Campaign(QtCore.QObject):

	campaignProgressChanged = QtCore.pyqtSignal()
	supportedStatusChanged = QtCore.pyqtSignal()

	def __init__(self, parent, data):
		QtCore.QObject.__init__(self)
		self._gameStateRef = weakref.ref(parent)
		self._data = data
		self._isEnabled = False
		self._levels = []

	@property
	def parent(self):
		return self._gameStateRef()

	@property
	def data(self):
		return self._data

	@property
	def levels(self):
		self._update_levels()
		return self._levels

	def enable(self, force=False):
		if self._isEnabled and not force:
			return
		self._isEnabled = True
		self.supportedStatusChanged.emit()

	def disable(self):
		if not self._isEnabled:
			return
		self._isEnabled = False
		self.supportedStatusChanged.emit()

	@property
	def isEnabled(self):
		return self._isEnabled

	def _update_levels(self):
		self._levels = [
			Level(self, levelId, levelName)
			for (levelId, levelName) in self._data.levels
		]
		for level in self._levels:
			level.levelCompleted.connect(self._on_level_state_changed)
			level.levelAvailable.connect(self._on_level_state_changed)

	@misc_utils.log_exception(_moduleLogger)
	def _on_level_state_changed(self, levelId):
		self.campaignProgressChanged.emit()


class Level(QtCore.QObject):

	levelLoaded = QtCore.pyqtSignal(str)
	levelCompleted = QtCore.pyqtSignal(str)
	levelAvailable = QtCore.pyqtSignal(str)

	def __init__(self, parent, levelId, levelName):
		QtCore.QObject.__init__(self)
		self._campaignRef = weakref.ref(parent)
		self._levelId = levelId
		self._levelName = levelName
		self._isCompleted = False
		self._isAvailable = False
		self._data = None
		self._stats = {}

	@property
	def parent(self):
		return self._campaignRef()

	@property
	def levelId(self):
		return self._levelId

	@property
	def levelName(self):
		return self._levelName

	@property
	def data(self):
		return self._data

	@property
	def stats(self):
		items = self._stats.items()
		items.sort()
		return items

	@property
	def isCompleted(self):
		return self._isCompleted

	@property
	def isAvailable(self):
		return self._isAvailable

	def load(self):
		if self._data is None:
			self._data = self.parent.data.load_level(self.levelId)
		self.levelLoaded.emit(self.levelId)

	def complete(self):
		self._isCompleted = True
		self.levelCompleted.emit(self.levelId)

	def mark_available(self):
		self._isAvailable = True
		self.levelAvailable.emit(self.levelId)


if __name__ == "__main__":
	pass

