#!/usr/bin/env python

from __future__ import with_statement
from __future__ import division

import os
import glob
import simplejson
import logging

import constants


_moduleLogger = logging.getLogger(__name__)


class GameLoader(object):

	def __init__(self, appDataPath):
		self._appDataPath = os.path.join(appDataPath, "campaigns")
		self._userDataPath = os.path.join(constants._data_path_, "campaigns")
		self._campaigns = None

	def get_campaigns(self):
		self._ensure_loaded()
		return self._campaigns

	def _ensure_loaded(self):
		if self._campaigns is None:
			self._campaigns = [
				CampaignLoader(self._appDataPath, path)
				for searchPath in (self._appDataPath, self._userDataPath)
				for path in glob.iglob(os.path.join(searchPath, "*.campaign"))
			]


class CampaignLoader(object):

	def __init__(self, appDataPath, path):
		self._id = os.path.basename(path)
		self._campaignPath = os.path.dirname(path)
		self._dataPath = appDataPath
		with open(path) as f:
			data = simplejson.load(f)
		self._name = data["name"]
		self._levels = data["levels"]
		self._imageName = data["image"]
		self._supportedPlayerCounts = data["players"]
		self._levelFiles = {}

	@property
	def name(self):
		return self._name

	@property
	def imageName(self):
		return self._imageName

	@property
	def campaignId(self):
		return self._id

	@property
	def campaignPath(self):
		return self._campaignPath

	@property
	def supportedPlayerCounts(self):
		return self._supportedPlayerCounts

	@property
	def dataSearchPaths(self):
		return self._dataSearchPaths

	@property
	def levels(self):
		"""Returns a list of level-ids and level names"""
		return [
			(level["path"], level["name"])
			for level in self._levels
		]

	def load_level(self, levelId):
		if levelId not in self._levelFiles:
			(levelData, ) = [
				level for level in self._levels
				if level["path"] == levelId
			]
			level = LevelLoader(self._dataPath, levelData)
			self._levelFiles[levelId] = level
		return self._levelFiles[levelId]


class LevelLoader(object):

	_SYMBOL_TO_IMAGE_ID = {
		"#": "block",
		"G": "greenMan",
		"P": "purpleMan",
		".": "pellet",
		"M": "monster_01",
		"*": "super_pellet",
		" ": "empty",
	}

	def __init__(self, appDataPath, levelData):
		levelPath = os.path.join(appDataPath, levelData["path"])
		self._levelDataPath = os.path.dirname(levelPath)
		with open(levelPath, "r") as f:
			self._level = simplejson.load(f)
		self._levelData = levelData

		self._dataSearchPaths = [
			self._levelDataPath,
			os.path.join(appDataPath, "common"),
		]

		self._layout =  self._level["layout"]
		self._wallLayout = self._separate_layout(self._layout, "#")
		self._itemLayout = self._separate_layout(self._layout, ".*")
		self._enemyLayout = self._separate_layout(self._layout, "M")
		self._playerLayout = self._separate_layout(self._layout, "PG")

	@property
	def name(self):
		return self._levelData["name"]

	def get_block_size(self):
		return 32

	def get_walls(self):
		return self._wallLayout

	def get_items(self):
		return self._itemLayout

	def get_enemies(self):
		return self._enemyLayout

	def get_players(self):
		return self._playerLayout

	def get_action_sound(self, action):
		return self.find_path(self._level["sound"][action])

	def get_background(self):
		return self._level["background"]

	def find_path(self, filename):
		for path in self._dataSearchPaths:
			absPath = os.path.join(path, filename)
			if os.path.exists(absPath):
				return absPath
		else:
			return ""

	def get_cell_image(self, symbol, state):
		imageId = self._SYMBOL_TO_IMAGE_ID[symbol]
		if state == "chomped":
			imageId += "Closed"
		elif state == "scared":
			parts = imageId.split("_", 1)
			imageId = "_".join([parts[0], "scared"] + parts[1:])
		elif state == "evil":
			parts = imageId.split("_", 1)
			imageId = "_".join([parts[0], "evil"] + parts[1:])
		elif state == "":
			pass
		else:
			raise RuntimeError("Unknown state: %r" % state)

		if imageId == "empty":
			path = ""
		else:
			try:
				return self.find_path(self._level["image"][imageId])
			except KeyError:
				path = ""
		return path

	def _separate_layout(self, layout, items):
		return [
			self._separate_row(row, items)
			for row in layout
		]

	def _separate_row(self, row, items):
		return "".join(
			(symbol if symbol in items else " ")
			for symbol in row
		)


if __name__ == "__main__":
	pass

