#!/usr/bin/env python

from __future__ import with_statement
from __future__ import division

import os
import logging

from PyQt4 import QtCore
from PyQt4 import QtGui

from util import misc as misc_utils


_moduleLogger = logging.getLogger(__name__)


class CampaignItem(object):

	def __init__(self, app, campaign):
		self._app = app
		self._campaign = campaign
		self._campaign.supportedStatusChanged.connect(self._on_supported_status_changed)

		campaignIcon = QtGui.QIcon(os.path.join(campaign.data.campaignPath, campaign.data.imageName))
		campaignTitle = campaign.data.name
		self._campaignButton = QtGui.QPushButton(campaignIcon, "")
		self._campaignButton.setCheckable(True)
		self._campaignButton.setIconSize(QtCore.QSize(280//2, 140//2))
		campaignLabel = QtGui.QLabel(campaignTitle)
		campaignLabel.setAlignment(QtCore.Qt.AlignCenter)
		campaignLabel.setBuddy(self._campaignButton)
		campaignLayout = QtGui.QVBoxLayout()
		campaignLayout.addWidget(self._campaignButton, 0, QtCore.Qt.AlignCenter)
		campaignLayout.addWidget(campaignLabel, 0, QtCore.Qt.AlignCenter)
		self._campaignWdiget = QtGui.QWidget()
		self._campaignWdiget.setLayout(campaignLayout)
		self._campaignWdiget.setContentsMargins(0, 0, 0, 0)
		self._set_visible_state()

	@property
	def toplevel(self):
		return self._campaignWdiget

	@property
	def button(self):
		return self._campaignButton

	def close(self):
		self._campaign.supportedStatusChanged.disconnect(self._on_supported_status_changed)

	def _set_visible_state(self):
		if self._campaign.isEnabled:
			self._campaignWdiget.show()
		else:
			self._campaignWdiget.hide()

	@misc_utils.log_exception(_moduleLogger)
	def _on_supported_status_changed(self):
		self._set_visible_state()


class CampaignSelector(object):

	def __init__(self, app, game):
		self._app = app
		self._gameState = game
		self._campaigns = self._gameState.campaigns

		self._campaignSelector = QtGui.QButtonGroup()
		self._campaignSelector.setExclusive(True)
		self._campaignSelectorLayout = QtGui.QBoxLayout(QtGui.QBoxLayout.LeftToRight)
		self._campaignSelectorWidget = QtGui.QWidget()
		self._campaignSelectorWidget.setLayout(self._campaignSelectorLayout)
		self._campaignSelectorWidget.setContentsMargins(0, 0, 0, 0)
		self._campaignScrollArea = QtGui.QScrollArea()
		self._campaignScrollArea.setWidget(self._campaignSelectorWidget)
		self._campaignScrollArea.setAlignment(QtCore.Qt.AlignLeft | QtCore.Qt.AlignTop)
		self._campaignScrollArea.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAsNeeded)
		self._campaignScrollArea.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)

		self._campaignUIItems = []
		for i, campaign in enumerate(self._campaigns):
			uiItem = CampaignItem(self._app, campaign)
			self._campaignSelectorLayout.addWidget(uiItem.toplevel, 0, QtCore.Qt.AlignCenter)
			self._campaignSelector.addButton(uiItem.button, i)
			self._campaignUIItems.append(uiItem)
		self._campaignUIItems[0].button.setChecked(True)

	@property
	def toplevel(self):
		return self._campaignSelectorWidget

	@property
	def buttonGroup(self):
		return self._campaignSelector

	def close(self):
		for uiItem in self._campaignUIItems:
			uiItem.close()

	def set_orientation(self, orientation):
		if orientation == QtCore.Qt.Horizontal:
			self._campaignSelectorLayout.setDirection(QtGui.QBoxLayout.LeftToRight)
		elif orientation == QtCore.Qt.Vertical:
			self._campaignSelectorLayout.setDirection(QtGui.QBoxLayout.TopToBottom)
		else:
			raise RuntimeError("Unknown Orientation: %r" % orientation)


class LevelSelector(object):

	def __init__(self, app):
		self._app = app
		self._campaign = None
		self._levels = []

		self._levelStore = QtGui.QStandardItemModel()
		self._levelSelector = QtGui.QTreeView()
		self._levelSelector.setModel(self._levelStore)
		self._levelSelector.setUniformRowHeights(True)
		self._levelSelector.setRootIsDecorated(False)
		self._levelSelector.setHorizontalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOff)
		self._levelSelector.setSelectionBehavior(QtGui.QAbstractItemView.SelectRows)
		self._levelSelector.setSelectionMode(QtGui.QAbstractItemView.SingleSelection)
		self._levelSelector.setHeaderHidden(True)
		self._levelSelector.setItemsExpandable(False)
		self._levelSelector.header().setResizeMode(QtGui.QHeaderView.ResizeToContents)

		self._unknownIcon = QtGui.QIcon(os.path.join(self._app.dataPath, "omnom", "unknownLevel.png"))
		self._levelSelector.setIconSize(self._unknownIcon.availableSizes()[0])

	@property
	def toplevel(self):
		return self._levelSelector

	def get_level(self, index):
		return self._levels[index]

	def load_campaign(self, campaign):
		for level in self._levels:
			level.levelCompleted.disconnect(self._on_level_completed)
			level.levelAvailable.disconnect(self._on_level_available)
		self._levelStore.clear()

		self._campaign = campaign
		if self._campaign is None:
			self._levels = []
		else:
			self._levels = self._campaign.levels

		for level in self._levels:
			level.levelCompleted.connect(self._on_level_completed)
			level.levelAvailable.connect(self._on_level_available)
			levelName = level.levelName
			item = QtGui.QStandardItem()
			item.setText(levelName)
			item.setIcon(self._unknownIcon)
			self._init_item(item, 4)

			stats = QtGui.QStandardItem()
			stats.setText("".join("%s: %s" % stat for stat in level.stats))
			self._init_item(stats, -4)

			row = (item, stats)
			self._levelStore.appendRow(row)

	def close(self):
		for level in self._levels:
			level.levelCompleted.disconnect(self._on_level_completed)
			level.levelAvailable.disconnect(self._on_level_available)
		self._levelStore.clear()

		self._campaign = None
		self._levels = []

	def _init_item(self, item, fontSize):
		itemFont = item.font()
		itemFont.setPointSize(itemFont.pointSize() + fontSize)
		item.setFont(itemFont)
		item.setEditable(False)
		item.setCheckable(False)

	def _get_level_index_from_id(self, levelId):
		levelIds = [level.levelId for level in self._levels]
		return levelIds.index(levelId)

	@misc_utils.log_exception(_moduleLogger)
	def _on_level_available(self, levelId):
		row = self._get_level_index_from_id(levelId)
		item = self._levelStore.item(row, 0)
		# item.setIcon TODO

	@misc_utils.log_exception(_moduleLogger)
	def _on_level_completed(self, levelId):
		row = self._get_level_index_from_id(levelId)
		level = self._levels[row]
		item = self._levelStore.item(row, 1)
		item.setText("".join("%s: %s" % stat for stat in level.stats))


if __name__ == "__main__":
	pass

