/*
 * This code was created by Jeff Molofee '99
 * (ported to Linux/SDL by Ti Leggett '01)
 * (ported to Maemo/SDL_gles by Till Harbaum '10)
 *
 * If you've found this code useful, please let me know.
 *
 * Visit Jeff at http://nehe.gamedev.net/
 *
 * or for port-specific comments, questions, bugreports etc.
 * email to leggett@eecs.tulane.edu or till@harbaum.org
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <SDL.h>
#include <SDL_image.h>
#include <SDL_gles.h>
#include <GLES/gl.h>
#include "glu.h"

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

/* Number of textures to load */
#define NUM_TEXTURES 3

/* This is our SDL surface */
SDL_Surface *surface;
SDL_GLES_Context *context;

/* Whether or not lighting is on */
int light = FALSE;

GLfloat xrot;      /* X Rotation */
GLfloat yrot;      /* Y Rotation */
GLfloat xspeed;    /* X Rotation Speed */
GLfloat yspeed;    /* Y Rotation Speed */
GLfloat z = -5.0f; /* Depth Into The Screen */

/* Ambient Light Values ( NEW ) */
GLfloat LightAmbient[]  = { 0.5f, 0.5f, 0.5f, 1.0f };
/* Diffuse Light Values ( NEW ) */
GLfloat LightDiffuse[]  = { 1.0f, 1.0f, 1.0f, 1.0f };
/* Light Position ( NEW ) */
GLfloat LightPosition[] = { 0.0f, 0.0f, 2.0f, 1.0f };

GLuint filter = 0;            /* Which Filter To Use    */
GLuint texture[NUM_TEXTURES]; /* Storage for 3 textures */

/* Which Filter To Use */
GLuint filter;
/* Storage For Three Types Of Fog */
GLfloat fogMode[] = { GL_EXP, GL_EXP2, GL_LINEAR };
/* Which Fog To Use */
GLuint fogFilter = 0;
/* Fog Color */
GLfloat fogColor[4] = { 0.5f, 0.5f, 0.5f, 1.0f };


/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{

    /* Clean up our textures */
    glDeleteTextures( NUM_TEXTURES, &texture[0] );

    SDL_GLES_DeleteContext(context);
    
    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures( )
{
    /* Status indicator */
    int Status = FALSE;

    /* Create storage space for the texture */
    SDL_Surface *TextureImage[1]; 

    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage[0] = IMG_Load( "/opt/nehegles/crate.png" ) ) )
        {

	    /* Set the status to true */
	    Status = TRUE;

	    /* Create The Texture */
	    glGenTextures( NUM_TEXTURES, &texture[0] );

	    /* Load in texture 1 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[0] );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_RGB,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );
	    
	    /* Nearest Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_NEAREST );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_NEAREST );

	    /* Load in texture 2 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[1] );

	    /* Linear Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_LINEAR );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_LINEAR );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_RGB,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );

	    /* Load in texture 3 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[2] );

	    /* Mipmapped Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_LINEAR_MIPMAP_NEAREST );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_LINEAR );

	    /* Generate The MipMapped Texture ( NEW ) */
	    gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB, TextureImage[0]->w,
			       TextureImage[0]->h, GL_RGB,
			       GL_UNSIGNED_BYTE, TextureImage[0]->pixels );
        }

    /* Free up any memory we may have used */
    if ( TextureImage[0] )
	    SDL_FreeSurface( TextureImage[0] );

    return Status;
}

/* function to reset our viewport after a window resize */
int resizeWindow( int width, int height )
{
    /* Height / width ration */
    GLfloat ratio;
 
    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection matrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* function to handle key press events */
void handleKeyPress( SDL_keysym *keysym )
{
    switch ( keysym->sym )
	{
	case SDLK_f:
	    /* 'f' key was pressed
	     * this pages through the different filters
	     */
	    filter = ( filter+1 ) % 3;
	    break;
	case SDLK_g:
	    /* 'g' key was pressed
	     * this pages through the different types of fog
	     */
	    fogFilter = ( fogFilter+1 ) % 3;
	    glFogf( GL_FOG_MODE, fogMode[fogFilter] );
	    break;
	case SDLK_l:
	    /* 'l' key was pressed
	     * this toggles the light
	     */
	    light = !light;
	    if ( !light )
		glDisable( GL_LIGHTING );
	    else
		glEnable( GL_LIGHTING );
	    break;
	case SDLK_i:
	    /* 'i' key was pressed
	     * this zooms into the scene
	     */
	    z += 0.02f;
	    break;
	case SDLK_o:
	    /* 'i' key was pressed
	     * this zooms out of the scene
	     */
	    z -= 0.02f;
	    break;
	case SDLK_UP:
	    /* Up arrow key was pressed
	     * this affects the x rotation
	     */
	    xspeed -= 0.01f;
	    break;
	case SDLK_DOWN:
	    /* Down arrow key was pressed
	     * this affects the x rotation
	     */
	    xspeed += 0.01f;
	    break;
	case SDLK_RIGHT:
	    /* Right arrow key was pressed
	     * this affects the y rotation
	     */
	    yspeed += 0.01f;
	    break;
	case SDLK_LEFT:
	    /* Left arrow key was pressed
	     * this affects the y rotation
	     */
	    yspeed -= 0.01f;
	    break;
	default:
	    break;
	}

    return;
}

/* general OpenGL initialization function */
int initGL( GLvoid )
{
    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    /* specify size of depth buffer (NEW) */
    SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 24);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);

    /* Load in the texture */
    if ( !LoadGLTextures( ) )
	return FALSE;

    /* Enable Texture Mapping ( NEW ) */
    glEnable( GL_TEXTURE_2D );

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.5f, 0.5f, 0.5f, 1.0f );

    /* Depth buffer setup */
    glClearDepthf( 1.0f );

    /* Enables Depth Testing */
    glEnable( GL_DEPTH_TEST );

    /* The Type Of Depth Test To Do */
    glDepthFunc( GL_LEQUAL );

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

    /* Setup The Ambient Light */
    glLightfv( GL_LIGHT1, GL_AMBIENT, LightAmbient );

    /* Setup The Diffuse Light */
    glLightfv( GL_LIGHT1, GL_DIFFUSE, LightDiffuse );

    /* Position The Light */
    glLightfv( GL_LIGHT1, GL_POSITION, LightPosition );

    /* Enable Light One */
    glEnable( GL_LIGHT1 );

    /* Setup the Fog */
    glFogf( GL_FOG_MODE, fogMode[fogFilter] ); /* Fog Mode */
    glFogfv( GL_FOG_COLOR, fogColor );         /* Set Fog Color */
    glFogf (GL_FOG_DENSITY, 0.35f );           /* How Dense Will The Fog Be */
    glHint( GL_FOG_HINT, GL_DONT_CARE );       /* Fog Hint Value */
    glFogf( GL_FOG_START, 1.0f );              /* Fog Start Depth */
    glFogf( GL_FOG_END, 5.0f );                /* Fog End Depth */
    glEnable( GL_FOG );                        /* Enables GL_FOG */

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    return( TRUE );
}

/* Here goes our drawing code */
int drawGLScene( GLvoid )
{
    /* These are to calculate our fps */
    static GLint T0     = 0;
    static GLint Frames = 0;
    int quad;

    /* Clear The Screen And The Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    /* Reset the view */
    glLoadIdentity( );

    /* Translate Into/Out Of The Screen By z */
    glTranslatef( 0.0f, 0.0f, z );

    glRotatef( xrot, 1.0f, 0.0f, 0.0f); /* Rotate On The X Axis By xrot */
    glRotatef( yrot, 0.0f, 1.0f, 0.0f); /* Rotate On The Y Axis By yrot */

    /* Select A Texture Based On filter */
    glBindTexture( GL_TEXTURE_2D, texture[filter] );

    const GLfloat quadVertices[][12] = {
      {    /* Front Face */
        -1.0f, -1.0f,  1.0f,      /* Bottom Left Of The Quad */
         1.0f, -1.0f,  1.0f,      /* Bottom Right Of The Quad */
        -1.0f,  1.0f,  1.0f,      /* Top Left Of The Quad */
         1.0f,  1.0f,  1.0f,      /* Top Right Of The Quad */
      }, { /* Back Face */
        -1.0f, -1.0f, -1.0f,      /* Bottom Right Of The Quad */
        -1.0f,  1.0f, -1.0f,      /* Top Right Of The Quad */
         1.0f, -1.0f, -1.0f,      /* Bottom Left Of The Quad */
         1.0f,  1.0f, -1.0f,      /* Top Left Of The Quad */
      }, { /* Top Face */
        -1.0f,  1.0f, -1.0f,      /* Top Left Of The Quad */
        -1.0f,  1.0f,  1.0f,      /* Bottom Left Of The Quad */
         1.0f,  1.0f, -1.0f,      /* Top Right Of The Quad */
         1.0f,  1.0f,  1.0f,      /* Bottom Right Of The Quad */
      }, { /* Bottom Face */
        -1.0f, -1.0f, -1.0f,      /* Top Right Of The Quad */
         1.0f, -1.0f, -1.0f,      /* Top Left Of The Quad */
        -1.0f, -1.0f,  1.0f,      /* Bottom Right Of The Quad */
         1.0f, -1.0f,  1.0f,      /* Bottom Left Of The Quad */
      }, { /* Right face */
         1.0f, -1.0f, -1.0f,      /* Bottom Right Of The Quad */
         1.0f,  1.0f, -1.0f,      /* Top Right Of The Quad */
         1.0f, -1.0f,  1.0f,      /* Bottom Left Of The Quad */
         1.0f,  1.0f,  1.0f,      /* Top Left Of The Quad */
      }, {/* Left Face */
        -1.0f, -1.0f, -1.0f,      /* Bottom Left Of The Quad */
        -1.0f, -1.0f,  1.0f,      /* Bottom Right Of The Quad */
        -1.0f,  1.0f, -1.0f,      /* Top Left Of The Quad */
        -1.0f,  1.0f,  1.0f,      /* Top Right Of The Quad */
      }
    };

    const GLfloat texCoords[][8] = {
      {    /* Front Face */
        0.0f, 1.0f,               /* Bottom Left Of The Texture */
        1.0f, 1.0f,               /* Bottom Right Of The Texture */
        0.0f, 0.0f,               /* Top Left Of The Texture */
        1.0f, 0.0f,               /* Top Right Of The Texture */
      },{ /* Back Face */
        0.0f, 0.0f,               /* Bottom Right Of The Texture */
        0.0f, 1.0f,               /* Top Right Of The Texture */
        1.0f, 0.0f,               /* Bottom Left Of The Texture */
        1.0f, 1.0f,               /* Top Left Of The Texture */
      },{ /* Top Face */
        1.0f, 1.0f,               /* Top Left Of The Texture */
        1.0f, 0.0f,               /* Bottom Left Of The Texture */
        0.0f, 1.0f,               /* Top Right Of The Texture */
        0.0f, 0.0f,               /* Bottom Right Of The Texture */
      },{ /* Bottom Face */
        0.0f, 1.0f,               /* Top Right Of The Texture */
        1.0f, 1.0f,               /* Top Left Of The Texture */
        0.0f, 0.0f,               /* Bottom Right Of The Texture */
        1.0f, 0.0f,               /* Bottom Left Of The Texture */
      },{ /* Right face */
        0.0f, 0.0f,               /* Bottom Right Of The Texture */
        0.0f, 1.0f,               /* Top Right Of The Texture */
        1.0f, 0.0f,               /* Bottom Left Of The Texture */
        1.0f, 1.0f,               /* Top Left Of The Texture */
      },{ /* Left Face */
        1.0f, 0.0f,               /* Bottom Left Of The Texture */
        0.0f, 0.0f,               /* Bottom Right Of The Texture */
        1.0f, 1.0f,               /* Top Left Of The Texture */
        0.0f, 1.0f,               /* Top Right Of The Texture */
      }
    };
    
    const GLfloat normVectors[][3] = {
      {  0.0f,  0.0f,  1.0f },    /* Normal Pointing Towards Viewer */
      {  0.0f,  0.0f, -1.0f },    /* Normal Pointing Away From Viewer */
      {  0.0f,  1.0f,  0.0f },    /* Normal Pointing Up */
      {  0.0f, -1.0f,  0.0f },    /* Normal Pointing Down */
      {  1.0f,  0.0f,  0.0f },    /* Normal Pointing Right */
      { -1.0f,  0.0f,  0.0f },    /* Normal Pointing Left */
    };

    /* Loop through all Quads */
    for(quad=0;quad<sizeof(quadVertices)/(12*sizeof(GLfloat));quad++)
    {
      glNormal3f(normVectors[quad][0], normVectors[quad][1], 
                 normVectors[quad][2]);

      glVertexPointer(3, GL_FLOAT, 0, quadVertices[quad]);
      glTexCoordPointer(2, GL_FLOAT, 0, texCoords[quad]);
      glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    }
  
    /* Draw it to the screen */
    SDL_GLES_SwapBuffers( );

    /* Gather our frames per second */
    Frames++;
    {
	GLint t = SDL_GetTicks();
	if (t - T0 >= 5000) {
	    GLfloat seconds = (t - T0) / 1000.0;
	    GLfloat fps = Frames / seconds;
	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
	    T0 = t;
	    Frames = 0;
	}
    }

    xrot += xspeed; /* Add xspeed To xrot */
    yrot += yspeed; /* Add yspeed To yrot */

    return( TRUE );
}

int main( int argc, char **argv )
{
    /* main loop variable */
    int done = FALSE;
    /* used to collect events */
    SDL_Event event;
    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

     SDL_EnableKeyRepeat(250, 20);

    surface = SDL_SetVideoMode(0, 0, 16, SDL_SWSURFACE | SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("NeHe OpenGL lesson 16", "NeHe16");

    /* initialize OpenGL */
    initGL( );

    /* resize the initial window */
    resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

    /* wait for events */
    while ( !done )
	{
	    /* handle the events in the queue */

	    while ( SDL_PollEvent( &event ) )
		{
		    switch( event.type )
			{
			case SDL_ACTIVEEVENT:
			    /* Something's happend with our focus
			     * If we lost focus or we are iconified, we
			     * shouldn't draw the screen
			     */
			    if ( event.active.gain == 0 )
				isActive = FALSE;
			    else
				isActive = TRUE;
			    break;
			case SDL_KEYDOWN:
			    /* handle key presses */
			    handleKeyPress( &event.key.keysym );
			    break;
			case SDL_MOUSEBUTTONDOWN:
  			case SDL_QUIT:
			    /* handle quit requests */
			    done = 1;
			    break;
			default:
			    break;
			}
		}

	    /* draw the scene */
	    if ( isActive )
		drawGLScene( );
	}

    /* clean ourselves up and exit */
    Quit( 0 );

    /* Should never get here */
    return( 0 );
}
