/*
 * This code was created by Jeff Molofee '99
 * (ported to Linux/SDL by Ti Leggett '01)
 * (ported to Maemo/SDL_gles by Till Harbaum '10)
 *
 * If you've found this code useful, please let me know.
 *
 * Visit Jeff at http://nehe.gamedev.net/
 *
 * or for port-specific comments, questions, bugreports etc.
 * email to leggett@eecs.tulane.edu or till@harbaum.org
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <SDL.h>
#include <SDL_image.h>
#include <SDL_gles.h>
#include <GLES/gl.h>
#include "glu.h"

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

/* This is our SDL surface */
SDL_Surface *surface;
SDL_GLES_Context *context;

GLuint texture[1]; /* Storage For One Texture */
GLuint xloop;      /* Loop For X Axis */
GLuint yloop;      /* Loop For Y Axis */

GLfloat xrot;      /* X Rotation */
GLfloat yrot;      /* Y Rotation */

/* Array For Box Colors */
static GLfloat boxcol[5][3] =
{
    /* Bright: */
    { 1.0f, 0.0f, 0.0f}, /* Red */
    { 1.0f, 0.5f, 0.0f}, /* Orange */
    { 1.0f, 1.0f, 0.0f}, /* Yellow */
    { 0.0f, 1.0f, 0.0f}, /* Green */
    { 0.0f, 1.0f, 1.0f}  /* Blue */
};

/* Array For Top Colors */
static GLfloat topcol[5][3] =
{
    /* Dark: */
    { 0.5f, 0.0f,  0.0f}, /* Red */
    { 0.5f, 0.25f, 0.0f}, /* Orange */
    { 0.5f, 0.5f,  0.0f}, /* Yellow */
    { 0.0f, 0.5f,  0.0f}, /* Green */
    { 0.0f, 0.5f,  0.5f}  /* Blue */
};

/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{
    SDL_GLES_DeleteContext(context);  

    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

GLfloat box_vertices[20][3];  /* Vertices array */
GLfloat box_texcoords[20][2]; /* Texture coordinates array */

GLfloat top_vertices[4][3];   /* Vertices array */
GLfloat top_texcoords[4][2];  /* Texture coordinates array */

/* function to build up our display lists */
GLvoid BuildLists( )
{
    /* Build two arrays */

    /* Bottom Face */
    box_texcoords[0][0]=0.0f; box_texcoords[0][1]=1.0f;
    box_texcoords[1][0]=1.0f; box_texcoords[1][1]=1.0f;
    box_texcoords[2][0]=0.0f; box_texcoords[2][1]=0.0f;
    box_texcoords[3][0]=1.0f; box_texcoords[3][1]=0.0f;
    box_vertices[0][0]=-1.0f; box_vertices[0][1]=-1.0f; box_vertices[0][2]=-1.0f;
    box_vertices[1][0]=1.0f;  box_vertices[1][1]=-1.0f; box_vertices[1][2]=-1.0f;
    box_vertices[2][0]=-1.0f; box_vertices[2][1]=-1.0f; box_vertices[2][2]=1.0f;
    box_vertices[3][0]=1.0f;  box_vertices[3][1]=-1.0f; box_vertices[3][2]=1.0f;

    /* Front Face */
    box_texcoords[4][0]=1.0f; box_texcoords[4][1]=0.0f;
    box_texcoords[5][0]=0.0f; box_texcoords[5][1]=0.0f;
    box_texcoords[6][0]=1.0f; box_texcoords[6][1]=1.0f;
    box_texcoords[7][0]=0.0f; box_texcoords[7][1]=1.0f;
    box_vertices[4][0]=-1.0f; box_vertices[4][1]=-1.0f; box_vertices[4][2]=1.0f;
    box_vertices[5][0]=1.0f;  box_vertices[5][1]=-1.0f; box_vertices[5][2]=1.0f;
    box_vertices[6][0]=-1.0f; box_vertices[6][1]=1.0f;  box_vertices[6][2]=1.0f;
    box_vertices[7][0]=1.0f;  box_vertices[7][1]=1.0f;  box_vertices[7][2]=1.0f;

    /* Back Face */
    box_texcoords[8][0]=0.0f; box_texcoords[8][1]=0.0f;
    box_texcoords[9][0]=0.0f; box_texcoords[9][1]=1.0f;
    box_texcoords[10][0]=1.0f; box_texcoords[10][1]=0.0f;
    box_texcoords[11][0]=1.0f; box_texcoords[11][1]=1.0f;
    box_vertices[8][0]=-1.0f; box_vertices[8][1]=-1.0f; box_vertices[8][2]=-1.0f;
    box_vertices[9][0]=-1.0f; box_vertices[9][1]=1.0f;  box_vertices[9][2]=-1.0f;
    box_vertices[10][0]=1.0f; box_vertices[10][1]=-1.0f; box_vertices[10][2]=-1.0f;
    box_vertices[11][0]=1.0f; box_vertices[11][1]=1.0f;  box_vertices[11][2]=-1.0f;

    /* Right face */
    box_texcoords[12][0]=0.0f; box_texcoords[12][1]=0.0f;
    box_texcoords[13][0]=0.0f; box_texcoords[13][1]=1.0f;
    box_texcoords[14][0]=1.0f; box_texcoords[14][1]=0.0f;
    box_texcoords[15][0]=1.0f; box_texcoords[15][1]=1.0f;
    box_vertices[12][0]=1.0f; box_vertices[12][1]=-1.0f; box_vertices[12][2]=-1.0f;
    box_vertices[13][0]=1.0f; box_vertices[13][1]=1.0f;  box_vertices[13][2]=-1.0f;
    box_vertices[14][0]=1.0f; box_vertices[14][1]=-1.0f; box_vertices[14][2]=1.0f;
    box_vertices[15][0]=1.0f; box_vertices[15][1]=1.0f;  box_vertices[15][2]=1.0f;

    /* Left Face */
    box_texcoords[16][0]=1.0f; box_texcoords[16][1]=0.0f;
    box_texcoords[17][0]=0.0f; box_texcoords[17][1]=0.0f;
    box_texcoords[18][0]=1.0f; box_texcoords[18][1]=1.0f;
    box_texcoords[19][0]=0.0f; box_texcoords[19][1]=1.0f;
    box_vertices[16][0]=-1.0f; box_vertices[16][1]=-1.0f; box_vertices[16][2]=-1.0f;
    box_vertices[17][0]=-1.0f; box_vertices[17][1]=-1.0f; box_vertices[17][2]=1.0f;
    box_vertices[18][0]=-1.0f; box_vertices[18][1]=1.0f;  box_vertices[18][2]=-1.0f;
    box_vertices[19][0]=-1.0f; box_vertices[19][1]=1.0f;  box_vertices[19][2]=1.0f;

    /* Top Face in another array */
    top_texcoords[0][0]=1.0f; top_texcoords[0][1]=1.0f;
    top_texcoords[1][0]=1.0f; top_texcoords[1][1]=0.0f;
    top_texcoords[2][0]=0.0f; top_texcoords[2][1]=1.0f;
    top_texcoords[3][0]=0.0f; top_texcoords[3][1]=0.0f;
    top_vertices[0][0]=-1.0f; top_vertices[0][1]=1.0f; top_vertices[0][2]=-1.0f;
    top_vertices[1][0]=-1.0f; top_vertices[1][1]=1.0f; top_vertices[1][2]=1.0f;
    top_vertices[2][0]=1.0f;  top_vertices[2][1]=1.0f; top_vertices[2][2]=-1.0f;
    top_vertices[3][0]=1.0f;  top_vertices[3][1]=1.0f; top_vertices[3][2]=1.0f;
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures( )
{
    /* Status indicator */
    int Status = FALSE;

    /* Create storage space for the texture */
    SDL_Surface *TextureImage[1]; 

    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage[0] = IMG_Load( "/opt/nehegles/cube.png" ) ) )
        {

	    /* Set the status to true */
	    Status = TRUE;

	    /* Create The Texture */
	    glGenTextures( 1, &texture[0] );

	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[0] );

	    /* Generate The Texture */
	    gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB, TextureImage[0]->w,
			       TextureImage[0]->h, GL_RGB,
			       GL_UNSIGNED_BYTE, TextureImage[0]->pixels );

	    /* Linear Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_LINEAR_MIPMAP_NEAREST );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_LINEAR );
        }

    /* Free up any memory we may have used */
    if ( TextureImage[0] )
	    SDL_FreeSurface( TextureImage[0] );

    return Status;
}

/* to resize the window and reset our view */
int resizeWindow( int width, int height )
{
    /* Height / width ration */
    GLfloat ratio;

    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection matrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* function to handle key press events */
void handleKeyPress( SDL_keysym *keysym )
{
    switch ( keysym->sym )
	{
	case SDLK_LEFT:
	    /* Left arrow key was pressed
	     * this affects y-rotation
	     */
	    yrot -= 0.2f;
	    break;
	case SDLK_RIGHT:
	    /* Right arrow key was pressed
	     * this affects y-rotation
	     */
	    yrot += 0.2f;
	    break;
	case SDLK_UP:
	    /* Up arrow key was pressed
	     * this affects x-rotation
	     */
	    xrot -= 0.2f;
	    break;
	case SDLK_DOWN:
	    /* Down arrow key was pressed
	     * this affects x-rotation
	     */
	    xrot += 0.2f;
	    break;
	default:
	    break;
	}

    return;
}

/* general OpenGL initialization function */
int initGL( GLvoid )
{
    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    /* specify size of depth buffer (NEW) */
    SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 24);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);

    /* Load in our textures */
    if ( !LoadGLTextures( ) )
	return FALSE;

    /* Build our display lists */
    BuildLists( );

    /* Enable Texture Mapping ( NEW ) */
    glEnable( GL_TEXTURE_2D );

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.0f, 0.0f, 0.0f, 0.5f );

    /* Depth buffer setup */
    glClearDepthf( 1.0f );

    /* Enables Depth Testing */
    glEnable( GL_DEPTH_TEST );

    /* The Type Of Depth Test To Do */
    glDepthFunc( GL_LEQUAL );

    /* Enable lighting */
    glEnable( GL_LIGHT0 );         /* Quick And Dirty Lighting */
    glEnable( GL_LIGHTING );       /* Enable Lighting */
    glEnable( GL_COLOR_MATERIAL ); /* Enable Material Coloring */

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    return( TRUE );
}

/* Here goes our drawing code */
int drawGLScene( GLvoid )
{
    /* These are to calculate our fps */
    static GLint T0     = 0;
    static GLint Frames = 0;

    /* Clear The Screen And The Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    /* Select Our Texture */
    glBindTexture( GL_TEXTURE_2D, texture[0] );

    /* Start loading in our display lists */
    for ( yloop = 0; yloop < 6; yloop++ )
	{
	    for ( xloop = 0; xloop < yloop; xloop++ )
		{
		    /* Reset the view */
		    glLoadIdentity( );

		    /* Position The Cubes On The Screen */
		    glTranslatef( 1.4f + ( ( float )xloop * 2.8f ) -
				         ( ( float )yloop * 1.4f ),
				  ( ( 6.0f - ( float )yloop ) * 2.4f ) - 7.0f,
				  -20.0f );

		    /* Tilt the cubes */
		    /* Tilt The Cubes Up And Down */
		    glRotatef(45.0f-(2.0f*yloop)+xrot, 1.0f, 0.0f, 0.0f);

		    /* Spin Cubes Left And Right */
		    glRotatef(45.0f+yrot, 0.0f, 1.0f, 0.0f);

		    /* Select A Box Color */
		    glColor4f(boxcol[yloop-1][0], boxcol[yloop-1][1], boxcol[yloop-1][2], 1.0f);
		    
		    /* Draw the box */
		    glVertexPointer(3, GL_FLOAT, 0, box_vertices);
		    glTexCoordPointer(2, GL_FLOAT, 0, box_texcoords);
		    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
		    glDrawArrays(GL_TRIANGLE_STRIP, 4, 4);
		    glDrawArrays(GL_TRIANGLE_STRIP, 8, 4);
		    glDrawArrays(GL_TRIANGLE_STRIP, 12, 4);
		    glDrawArrays(GL_TRIANGLE_STRIP, 16, 4);

		    /* Select The Top Color */
		    glColor4f(topcol[yloop-1][0], topcol[yloop-1][1], topcol[yloop-1][2], 1.0f);
		    glVertexPointer(3, GL_FLOAT, 0, top_vertices);
		    glTexCoordPointer(2, GL_FLOAT, 0, top_texcoords);
		    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
		}
	}

    /* Draw it to the screen */
    SDL_GLES_SwapBuffers( );

    /* Gather our frames per second */
    Frames++;
    {
	GLint t = SDL_GetTicks();
	if (t - T0 >= 5000) {
	    GLfloat seconds = (t - T0) / 1000.0;
	    GLfloat fps = Frames / seconds;
	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
	    T0 = t;
	    Frames = 0;
	}
    }

    return( TRUE );
}

int main( int argc, char **argv )
{
    /* main loop variable */
    int done = FALSE;
    /* used to collect events */
    SDL_Event event;
    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

    SDL_EnableKeyRepeat(250, 20);

    surface = SDL_SetVideoMode(0, 0, 16, SDL_SWSURFACE | SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("NeHe OpenGL lesson 12", "NeHe12");

    /* initialize OpenGL */
    initGL( );

    /* resize the initial window */
    resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

    /* wait for events */
    while ( !done )
	{
	    /* handle the events in the queue */

	    while ( SDL_PollEvent( &event ) )
		{
		    switch( event.type )
			{
			case SDL_ACTIVEEVENT:
			    /* Something's happend with our focus
			     * If we lost focus or we are iconified, we
			     * shouldn't draw the screen
			     */
			    if ( event.active.gain == 0 )
				isActive = FALSE;
			    else
				isActive = TRUE;
			    break;			    
			case SDL_KEYDOWN:
			    /* handle key presses */
			    handleKeyPress( &event.key.keysym );
			    break;
                        case SDL_MOUSEBUTTONDOWN:
			case SDL_QUIT:
			    /* handle quit requests */
			    done = TRUE;
			    break;
			default:
			    break;
			}
		}

	    /* draw the scene */
	    if ( isActive )
		drawGLScene( );
	}

    /* clean ourselves up and exit */
    Quit( 0 );

    /* Should never get here */
    return( 0 );
}
