/*
 *  Nameday desktop applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>

#include "nameday_calendar.h"
#include "nameday_dbparser.h"
#include "nameday_settings.h"

static void
nameday_calendar_on_window_destroy_event (GtkWidget *window,
					  GdkEvent  *event,
					  gpointer data);

static void
nameday_calendar_jump_to_entry_by_date (GtkWidget *window,
					guint month,
					guint day)
{
	GtkWidget *treeview;
	GtkTreeModel *filter;
	GtkTreePath *path;
	GtkTreeIter iter;
	gboolean iter_valid = TRUE;

	g_return_if_fail (window);

	filter = g_object_get_data (G_OBJECT (window), "filter");
	for (iter_valid = gtk_tree_model_get_iter_first (filter, &iter); iter_valid;
	     iter_valid = gtk_tree_model_iter_next (filter, &iter)) {
	    	guint column_day = 0, column_month = 0;
		gtk_tree_model_get (GTK_TREE_MODEL (filter), &iter, COLUMN_DAY, &column_day, COLUMN_MONTH, &column_month, -1);
		if ((column_day == day) && (column_month == month) )
		{
			treeview = g_object_get_data (G_OBJECT (window), "treeview");
			path = gtk_tree_model_get_path (filter, &iter);
			gtk_tree_view_set_cursor (GTK_TREE_VIEW (treeview), path, NULL, FALSE);
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview), path, NULL, TRUE,
						      0.5, 0);
			gtk_tree_path_free(path);
			break;
		}
	}
}

static void
nameday_calendar_cell_data_func (GtkTreeViewColumn *column,
				 GtkCellRenderer *cell,
				 GtkTreeModel *model,
				 GtkTreeIter *iter,
				 gpointer data)
{
	GtkTreeView *tree;
	GtkTreeSelection *selection;

	tree = (GtkTreeView *) data;
	selection = gtk_tree_view_get_selection (tree);

	if (gtk_tree_selection_iter_is_selected (selection, iter)) {
		g_object_set (cell,
			      "ellipsize", PANGO_ELLIPSIZE_NONE,
			      "wrap-width", 700,
			      NULL);
	} else {
		g_object_set (cell,
			      "ellipsize", PANGO_ELLIPSIZE_END,
			      "wrap-width", -1,
			      NULL);
	}
}

static gboolean
nameday_calendar_filered_view_visible_func (GtkTreeModel *model,
					    GtkTreeIter *iter,
					    gpointer data)
{
	NamedayDesktopApplet *priv;
	gchar *name = NULL, *ascii_name = NULL;
	const gchar *searched_name = NULL;
	gboolean found = FALSE;

	g_return_val_if_fail (data, FALSE);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	if (priv->searched_name == NULL)
		return TRUE;

	searched_name = g_ascii_strdown (priv->searched_name, strlen (priv->searched_name));

	gtk_tree_model_get (model, iter, COLUMN_NAME, &name, -1);

	if (name) {
		ascii_name = g_ascii_strdown (name,  strlen (name));
		g_free (name);
	}

	if (g_strstr_len (ascii_name, strlen (ascii_name), searched_name) != NULL)
		found = TRUE;

	if (ascii_name)
		g_free (ascii_name);

	return found;
}

static void
nameday_calendar_on_menu_search_name_event (GtkButton *button,
					    gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *search, *entry;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);
	g_return_if_fail (priv->calendar_window);
	
	/* set search menu item inactive */
	gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);

	/* show search toolbar */
	search = g_object_get_data (G_OBJECT (priv->calendar_window), "search");
	gtk_widget_show_all (search);

	/* empty search entry */
	entry = g_object_get_data (G_OBJECT (search), "entry");
	gtk_entry_set_text (GTK_ENTRY (entry), "");
	gtk_widget_grab_focus (GTK_WIDGET (entry));
}

static void
nameday_calendar_on_menu_search_date_event (GtkButton *button,
					    gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *dialog, *selector, *treeview;
	GtkTreeModel *filter;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	guint day = 0, month = 0, year = 0;
	guint selected_day, selected_month;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);
	g_return_if_fail (priv->calendar_window);

	nameday_desktop_get_current_date (&year, &month, &day);

	/* date dialog */
	dialog = gtk_dialog_new_with_buttons (_("Date"),
					      NULL,
					      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					      _("OK"), GTK_RESPONSE_OK,
					      NULL);
	gtk_widget_set_size_request(GTK_WIDGET (GTK_DIALOG (dialog)->vbox), -1, 345);

	selector = hildon_date_selector_new_with_year_range (year, year);

	/* set date to the date of selected entry, if nothing is selected, set date
	   to current date */
	treeview = g_object_get_data (G_OBJECT (priv->calendar_window), "treeview");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	if (gtk_tree_selection_get_selected (selection, &filter, &iter)) {
		filter = g_object_get_data (G_OBJECT (priv->calendar_window), "filter");
		gtk_tree_model_get (GTK_TREE_MODEL (filter), &iter, COLUMN_DAY, &day, COLUMN_MONTH, &month, -1);
	}

	hildon_date_selector_select_current_date (HILDON_DATE_SELECTOR (selector),
						  year, month - 1, day);

	hildon_touch_selector_center_on_selected (HILDON_TOUCH_SELECTOR (selector));

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), selector);

	/* show date dialog */
	gtk_widget_show_all (dialog);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		hildon_date_selector_get_date (HILDON_DATE_SELECTOR (selector), NULL,
					       &selected_month, &selected_day);
		nameday_calendar_jump_to_entry_by_date (priv->calendar_window,
							selected_month + 1,
							selected_day);
	}
	gtk_widget_destroy (dialog);
}

static void
nameday_calendat_on_entry_changed (GtkEditable *editable,
				   gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *pannable, *treeview;
	GtkTreeModel *filter;
	GtkTreeIter iter;
	GtkTreePath *path;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);
	g_return_if_fail (priv->calendar_window);

	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));

	/* refilter treeview */
	filter = g_object_get_data (G_OBJECT (priv->calendar_window), "filter");
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));

	/* jump to the first entry in the treeview */
	if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (filter), &iter)) {
		path = gtk_tree_model_get_path (GTK_TREE_MODEL (filter), &iter);
		treeview = g_object_get_data (G_OBJECT (priv->calendar_window), "treeview");
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (treeview), path, NULL, FALSE);
		gtk_tree_path_free (path);
	}

	pannable = g_object_get_data (G_OBJECT (priv->calendar_window), "pannable");
	hildon_pannable_area_jump_to (HILDON_PANNABLE_AREA(pannable), -1, -1);
}

static void
nameday_calendar_on_search_close (GtkButton *button,
				  gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *search_name, *search, *treeview;
	GtkTreeModel *filter;
	GtkTreeSelection *selection;
	GtkTreeIter iter;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	g_return_if_fail (priv->calendar_window);

	/* set search menu item inactive */
	search_name = g_object_get_data (G_OBJECT (priv->calendar_window), "search_name");
	if (!GTK_WIDGET_SENSITIVE (search_name)) {
		gtk_widget_set_sensitive (search_name, TRUE);
	}

	/* show unfiltered treeview */
	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = NULL;

	/* refilter treeview */
	filter = g_object_get_data (G_OBJECT (priv->calendar_window), "filter");
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));

	/* hide search toolbar */
	search = g_object_get_data (G_OBJECT (priv->calendar_window), "search");
	gtk_widget_hide (search);
	
	/* jump to the selected entry in treeview */
	treeview = g_object_get_data (G_OBJECT (priv->calendar_window), "treeview");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	if (gtk_tree_selection_get_selected (selection, &filter, &iter))
	{
		guint column_day, column_month;
		gtk_tree_model_get (GTK_TREE_MODEL (filter), &iter, COLUMN_DAY, &column_day, COLUMN_MONTH, &column_month, -1);
		nameday_calendar_jump_to_entry_by_date (priv->calendar_window, column_month, column_day);
	}
}

static gboolean
nameday_calendar_on_key_press_event (GtkWidget *widget,
				     GdkEventKey *event,
				     gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *search, *entry, *search_name;

	g_return_val_if_fail (data, TRUE);

	priv = NAMEDAY_DESKTOP_APPLET (data);
	
	g_return_val_if_fail (priv->calendar_window, TRUE);
	
	search = g_object_get_data (G_OBJECT (priv->calendar_window), "search");

	if ((event->keyval > GDK_space) && (event->keyval <= GDK_stricteq) && !GTK_WIDGET_VISIBLE (search)) {
		/* set search menu item inactive */
		search_name = g_object_get_data (G_OBJECT (priv->calendar_window), "search_name");
		if (GTK_WIDGET_SENSITIVE (search_name)) {
			gtk_widget_set_sensitive (search_name, FALSE);
		}

		/* show search toolbar */
		gtk_widget_show (search);
		
		/* empty search entry */
		entry = g_object_get_data (G_OBJECT (search), "entry");
		gtk_entry_set_text (GTK_ENTRY (entry), "");
		gtk_widget_grab_focus (GTK_WIDGET (entry));
	}

	return FALSE;
}

static void
nameday_calendar_on_window_destroy_event (GtkWidget *window,
					  GdkEvent  *event,
					  gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkTreeModel *filter;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);
	filter = g_object_get_data (G_OBJECT (priv->calendar_window), "filter");
	g_object_unref (filter);

	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = NULL;

	/* destory window if it was created */
	if (priv->calendar_window)
		gtk_widget_destroy (priv->calendar_window);
	priv->calendar_window = NULL;
}

GtkWidget*
nameday_calendar_window_create (NamedayDesktopApplet *priv)
{
	GtkWidget *window, *vbox, *alignment, *pannable, *treeview, *menu_item, *search, *entry, *close;
	GtkListStore *list_store;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeModel *filter;
	GdkColor style_color;
	GtkEntryCompletion *completion;
	HildonGtkInputMode input_mode;
	HildonAppMenu *menu = NULL;
	guint current_day = 0, current_month = 0;

	g_return_val_if_fail (priv, NULL);

	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = NULL;

	list_store = nameday_dbparser_liststore_create (priv);

	/* main window */
	window = hildon_stackable_window_new ();
	gtk_window_set_title (GTK_WINDOW (window), _("Nameday calendar"));
	hildon_program_add_window (hildon_program_get_instance(), HILDON_WINDOW (window));
	
	/* main vbox */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);

	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), HILDON_MARGIN_HALF, 0,
				   HILDON_MARGIN_DOUBLE, HILDON_MARGIN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (vbox), alignment, TRUE, TRUE, 0);

	/* scrolled window */
	pannable = hildon_pannable_area_new ();
	gtk_container_add (GTK_CONTAINER (alignment), pannable);
	g_object_set_data (G_OBJECT (window), "pannable", pannable);

	/* filtered view */
	filter = gtk_tree_model_filter_new (GTK_TREE_MODEL (list_store), NULL);

	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (filter),
						nameday_calendar_filered_view_visible_func,
						priv,
						NULL);
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));
	g_object_set_data (G_OBJECT (window), "filter", filter);
	g_object_unref (list_store);

	/* nameday treeview */
	treeview = hildon_gtk_tree_view_new (HILDON_UI_MODE_EDIT);
	gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), filter);
	gtk_tree_view_set_enable_search (GTK_TREE_VIEW (treeview), TRUE);
	gtk_container_add (GTK_CONTAINER (pannable), treeview);
	g_object_set_data (G_OBJECT (window), "treeview", treeview);

	/* date column */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1,
						     "Date", renderer,
						     "text", COLUMN_DATE,
						     NULL);

	if (!gtk_style_lookup_color (GTK_WIDGET (priv->label)->style, "SecondaryTextColor",
				     &style_color)) {
		gdk_color_parse ("grey", &style_color);
	}
	
	g_object_set (G_OBJECT (renderer),
		      "xalign", 1.0,
	              "width-chars", 6,
	              "foreground-gdk", &style_color,
	              "foreground-set", TRUE,
	              NULL);

	/* name column */
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						     -1,
						     "Name", renderer,
						     "text", COLUMN_NAME,
						     NULL);

	gtk_style_lookup_color (GTK_WIDGET (priv->label)->style, "ActiveTextColor",
				&style_color);
	
	g_object_set (G_OBJECT (renderer),
		      "ellipsize", PANGO_ELLIPSIZE_END,
		      "ellipsize-set", TRUE,
	              "foreground-gdk", &style_color,
	              "foreground-set", TRUE,
	              "wrap-mode", PANGO_WRAP_WORD,
		      NULL);

	column = gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), 1);
	gtk_tree_view_column_set_cell_data_func (column,
						 renderer,
						 nameday_calendar_cell_data_func,
						 treeview,
						 NULL);

	/* search bar */
	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0,
				   HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, TRUE, 0);

	search = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (alignment), search);
	g_object_set_data (G_OBJECT (window), "search", search);
	
	entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
	gtk_box_pack_start (GTK_BOX (search), entry, TRUE, TRUE, 0);

	input_mode = hildon_gtk_entry_get_input_mode (GTK_ENTRY (entry));
  
        /* Disable unsupported input modes. */
	input_mode &= ~HILDON_GTK_INPUT_MODE_MULTILINE;
        input_mode &= ~HILDON_GTK_INPUT_MODE_INVISIBLE;
        input_mode &= ~HILDON_GTK_INPUT_MODE_DICTIONARY;
          
        hildon_gtk_entry_set_input_mode (GTK_ENTRY (entry), input_mode);
            
        completion = gtk_entry_completion_new ();
        gtk_entry_completion_set_inline_completion (completion, TRUE);
        gtk_entry_completion_set_popup_completion (completion, FALSE);
        gtk_entry_set_completion (GTK_ENTRY (entry), completion);
	g_object_set_data (G_OBJECT (search), "entry", entry);

	close = GTK_WIDGET (gtk_tool_button_new (gtk_image_new_from_icon_name ("general_close",
						 HILDON_ICON_PIXEL_SIZE_FINGER), "Close"));
	gtk_box_pack_start (GTK_BOX (search), close, FALSE, TRUE, 0);

	g_signal_connect(close, "clicked",
			 G_CALLBACK(nameday_calendar_on_search_close), priv);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(nameday_calendat_on_entry_changed), priv);


	/* application menu */
	menu = HILDON_APP_MENU (hildon_app_menu_new ());
	hildon_window_set_app_menu (HILDON_WINDOW (window), menu);

	/* search by name menu item */
	menu_item = hildon_gtk_button_new (HILDON_SIZE_AUTO);
	gtk_button_set_label (GTK_BUTTON (menu_item), _("Search by name"));
	hildon_app_menu_append (menu, GTK_BUTTON (menu_item));
	g_object_set_data (G_OBJECT (window), "search_name", menu_item);

	g_signal_connect (menu_item, "clicked",
			  G_CALLBACK (nameday_calendar_on_menu_search_name_event), priv);

	/* search by date menu item */
	menu_item = hildon_gtk_button_new (HILDON_SIZE_AUTO);
	gtk_button_set_label (GTK_BUTTON (menu_item), _("Search by date"));
	hildon_app_menu_append (menu, GTK_BUTTON (menu_item));
	g_object_set_data (G_OBJECT (window), "search_date", menu_item);

	g_signal_connect (menu_item, "clicked",
			  G_CALLBACK (nameday_calendar_on_menu_search_date_event), priv);

	/* window callbacks */
	g_signal_connect (window, "destroy_event",
			  G_CALLBACK (nameday_calendar_on_window_destroy_event), priv);
	g_signal_connect (window, "delete_event",
			  G_CALLBACK (nameday_calendar_on_window_destroy_event), priv);
	g_signal_connect (window, "key-press-event",
			  G_CALLBACK (nameday_calendar_on_key_press_event), priv);

	/* show menu */
	gtk_widget_show_all (GTK_WIDGET (menu));

	/* show window */
	gtk_widget_show_all (window);

	/* hide search toolbar */
	search = g_object_get_data (G_OBJECT (window), "search");
	gtk_widget_hide (search);

	/* jump to the current day entry */
	nameday_desktop_get_current_date (NULL, &current_month, &current_day);
	nameday_calendar_jump_to_entry_by_date (window, current_month, current_day);

	return window;
}
