/*
 * db_functions.c - this file contains different functions for work with XML-DB schemas. 
 * This file is part of Maemo-DB project.
 *
 * Copyright (C) 2009 - Alexander A. Lomov
 *
 * Maemo-DB project is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Maemo-DB project is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
#include "db_functions.h"
#include "db_xml_functions.h"


/**
 * @brief Load stylescheet associated with given schema and applyed it to document.
 *
 * @param schema_name name of schema.
 * @param doc xml document.
 * @param params NULL-terminated list of strings. 
 *        format: name value name value ... NULL.
 *
 * @return xml document on success or NULL otherwise.
 */
xmlDocPtr db_apply_stylesheet(const gchar* schema_name, const xmlDocPtr doc, 
                              const gchar** params) 
{
    if (schema_name == NULL || doc == NULL) {
        return NULL;
    }

    xsltStylesheetPtr style = db_get_stylesheet(schema_name, FALSE);

    if (style == NULL) {
        return NULL;
    }

    xmlDocPtr result = xsltApplyStylesheet(style, doc, params);
	
	xsltFreeStylesheet(style);
	
	return result;                  
}


/**
 * @brief Load stylescheet associated with given schema and applyed it to document.
 *
 * @param schema_name name of schema.
 * @param doc xml document.
 * @param params NULL-terminated list of strings. 
 *        format: name value name value ... NULL.
 *
 * @return xml document on success or NULL otherwise.
 */
xmlDocPtr db_apply_revers_stylesheet(const gchar* schema_name, const xmlDocPtr doc, 
                                     const gchar** params) 
{
    if (schema_name == NULL || doc == NULL) {
        return NULL;
    }

    xsltStylesheetPtr style = db_get_stylesheet(schema_name, TRUE);

    if (style == NULL) {
        g_debug( "schema %s", schema_name);
        return NULL;
    }

    xmlDocPtr result = xsltApplyStylesheet(style, doc, params);
	xsltFreeStylesheet(style);
	
	return result;                  
}


/**
 * @brief Load stylescheet associated with given schema.
 *
 * @param schema_name name of schema.
 * @param is_revers indicates that need load the reverse transformation.
 * 
 * @return stylesheet document on success or NULL otherwise.
 */
xsltStylesheetPtr db_get_stylesheet(const gchar* schema_name, const gboolean is_revers) 
{
    if (schema_name == NULL) {
        return NULL;
    }

    gchar* xslt_path = NULL;

    if (is_revers == TRUE) {
        xslt_path = cm_get_revers_xslt_path_key(schema_name);
    } else {
        xslt_path = cm_get_xslt_path_key(schema_name);
    }    
    g_debug("XSLT: %s", xslt_path);
    if (xslt_path == NULL) {
        return NULL;
    }

    return xsltParseStylesheetFile(BAD_CAST xslt_path); 
}






/*
xmlNodePtr find_child_node_by_name(const xmlNodePtr parent_node, 
                                   const xmlChar* find_name,
                                   const xmlChar* ns)
{
    return NULL;
}
*/


gint db_add_doc_to_tree1(xmlDocPtr tree, xmlDocPtr doc)
{

    if (tree == NULL || doc == NULL) {
		return ERROR_ARGUMENT; 
	}

    xmlNodePtr doc_root = xmlDocGetRootElement(doc);
    xmlNodePtr tree_root = xmlDocGetRootElement(tree);

    if (doc_root == NULL) {
        return FAILURE;
    }   

    if (tree_root == NULL ) {
        xmlDocSetRootElement(tree, doc_root);  
        xmlDocSetRootElement(doc, NULL);
        xmlSetListDoc(doc_root, tree); 
	} else {
        xmlNodePtr tmp_nodes = xmlCopyNodeList(doc_root->children);

        tmp_nodes->nsDef = NULL;            /* FIXME: ns links to root node. */
        xmlAddChild(tree_root, tmp_nodes);
        xmlSetListDoc(tmp_nodes, tree); 
    }

    xmlFreeDoc(doc);

	return SUCCESS;
} 




/**
 * @brief Generate new id for mxl document associated with given schema.
 *
 * @param schema_name name of schema.
 * @param doc xml document for search id.
 * 
 * @retunr generated ID on success or NULL otherwise.
 */
gchar* db_generate_id(const gchar* schema_name, const xmlDocPtr doc)
{
    gchar* id = g_try_new(gchar, 15);

    if(id == NULL)
        return NULL;

	xmlXPathObjectPtr result = NULL;
   
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);
    
    gchar* id_xpath = cm_get_table_id_path_key(schema_name);
    gchar* id_name = cm_get_table_id_name_key(schema_name);
    gchar* id_ns = cm_get_table_id_ns_key(schema_name);
    gchar* max_id_xpath = cm_get_table_max_id_path_key(schema_name);
    
    if (!id_xpath || !id_name || !max_id_xpath) {
        g_message("Can't get id info from config file");

        g_free(id); 
        g_free(id_name);
        g_free(id_xpath);
        g_free(max_id_xpath);

        return NULL;
    }

    g_message("db_generate_id: before get nodes");
    result = db_get_nodeset(doc, BAD_CAST max_id_xpath, ns); 
    g_message("db_generate_id: after get nodes %s %s", id_name, id_ns);        
 
    if (result) {
	    xmlNodeSetPtr nodeset = result->nodesetval;
		
		if (nodeset->nodeNr > 0) {
            gdouble tmp_id = strtod((char*)((id_ns ==  NULL) 
                ? xmlGetNsProp(nodeset->nodeTab[0], BAD_CAST id_name, BAD_CAST id_ns) 
		        : xmlGetProp(nodeset->nodeTab[0],BAD_CAST id_name)), NULL);
              
		   snprintf(id, 15, "%i", (gint)(tmp_id + 1));
		}
		xmlXPathFreeObject(result);
	} else {
	    id[0] = '1';
	    id[1] = '\0';
	}
    
    if(ns != NULL) {
        xmlFree(ns);
    }    
    g_free(id_name);
    g_free(id_ns);
    g_free(id_xpath);
    g_free(max_id_xpath);

    return id;
}



//FIXME rename table to schema
/**
 * @brief Check unique paths in xml document.
 *
 * @param schema_name name of schema.
 * @param current_doc xml document with current data.
 * @param new_doc xml document for check it unique.
 * 
 * @retunr count of non unique paths.
 */
gint db_check_unique(const gchar* table_name, const xmlDocPtr current_doc,
                     const xmlDocPtr new_doc)
{
    gchar** unique_xpaths = cm_get_unique_columns_keys(table_name);
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(table_name);
    int not_unique_xpath_num = 0;
  
    if (!unique_xpaths) {
        return 0;
    }

    int xpath_num = 0;
    xmlChar *key;

    while(unique_xpaths[xpath_num] && !not_unique_xpath_num)
    {
        g_message("db_check_unique; unique_xpaths[%i]=%s", xpath_num, unique_xpaths[xpath_num]);
        xmlXPathObjectPtr result = 
                        db_get_nodeset(new_doc, 
                                        BAD_CAST unique_xpaths[xpath_num], ns);    
        if(result) 
    	{
            xmlNodeSetPtr nodeset = result->nodesetval;
            int i = 0;
            for(; i < nodeset->nodeNr && !not_unique_xpath_num; ++i)
            {
                key = xmlNodeListGetString(new_doc, 
                                    nodeset->nodeTab[i]->xmlChildrenNode, 1);
                        
                xmlChar* unique_xpath = 
                                BAD_CAST g_strconcat(unique_xpaths[xpath_num], 
                                                    "[.=\"", key, "\"]", NULL);        
                g_message("db_check_unique; unique_xpath: %s", unique_xpath);
                if (db_get_nodes_count(current_doc, unique_xpath, ns) )
                    not_unique_xpath_num = xpath_num + 1;
                
                xmlFree(key);
                xmlFree(unique_xpath);
            }
	        xmlXPathFreeObject(result);
        } /* End if(result) */           
        ++xpath_num;
    } /* End While */

    xmlFree(ns);
    g_strfreev(unique_xpaths);
    g_message("not_unique_xpath_num = %i", not_unique_xpath_num);
    return not_unique_xpath_num;
}                                        


/**
 * @brief Load from file or create new xml document associated with given scheme.
 *
 * @param schema_name name of schema.
 * @param db_path path to folder with data base files.
 * 
 * @return table on success or NULL otherwise.
 */
xmlDocPtr db_get_table(const gchar* schema_name, const gchar* db_path)
{
    gchar* path =  g_strconcat(db_path, G_DIR_SEPARATOR_S, schema_name, 
                               XML_FILE_EXTENSION, NULL);   
      
    xmlDocPtr doc = xmlParseFile(path);
    g_free(path);

    if (doc == NULL) {
        doc = xmlNewDoc(BAD_CAST XML_VERSION);
    }
    
    return doc;
}


/**
 * @brief Save xml document associated with given scheme.
 *
 * @param schema_name name of schema.
 * @param doc xml document for save.
 * @param db_path path for data base.
 *
 * @return 0 on success or non 0 value otherwise.
 */
gint db_save_table(const xmlDocPtr doc, const gchar* schema_name, 
                   const gchar* db_path)
{
    gchar* path =  g_strconcat(db_path, "/", schema_name, ".xml", NULL);   
      
    FILE* file = fopen(path, "w+t");
    g_message("db_save_table, path %s", path);
    
    if (file == NULL) {
        return ERROR_CANT_OPEN_FILE;
    }

    if (xmlDocDump(file, doc) < 0) {
        return ERROR_CANT_SAVE_TABLE;
    }
	
	fclose(file); 
	
    g_free(path);
//    g_free(file);
    
    return SUCCESS;
}


/**
 * Check equals of attributes. 
 *
 * @param a first attribute.
 * @param b second attribute.
 *
 * @return TRUE if properties equal or FALSE otherwise.
 */      
gboolean check_properties_equals(const xmlAttrPtr a, const xmlAttrPtr b)
{
    if (xmlStrcmp(a->name, b->name)) {
       return FALSE; 
    } 
    
    if (!(a->ns && b->ns) || !(!a->ns && !b->ns)) {
       return FALSE; 
    }
    
    if (xmlStrcmp(a->ns->href, a->ns->href)) {
        return FALSE;
    }

    return TRUE;
}


/**
 * Sets identifier attribute for node by given XPath expression. 
 *
 * @schema_name name of schema data
 * @param doc xml document.
 * @param node_id_value identifier value.
 *
 * @return 0 on success and not 0 value otherwise.
 */  
gint db_set_node_id(const gchar* schema_name, const xmlDocPtr doc, 
                    const gchar* nodeId)
{
    if(!schema_name || !doc || !nodeId)
    {
        g_message("db_set_node_id: Bad agument(s)");
        return -1;
    }
        
    gchar* id_ns = cm_get_table_id_ns_key(schema_name);    
    gchar* id_name = cm_get_table_id_name_key(schema_name);
    gchar* xpath = cm_get_table_id_path_key(schema_name);
    gchar* id_ns_prefix = cm_get_table_id_ns_prefix_key(schema_name);
    xmlChar* ns = BAD_CAST cm_get_namespaces_key(schema_name);

    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST xpath, ns);
    
    if (result != NULL) {
        xmlNodeSetPtr nodeset = result->nodesetval;
        
        if (nodeset->nodeNr > 0) {
            db_set_node_property(nodeset->nodeTab[0], id_name, 
                                  nodeId, id_ns, id_ns_prefix);         
                        
            if (nodeset->nodeTab[0]->type != XML_NAMESPACE_DECL) {
        	    nodeset->nodeTab[0] = NULL;                
        	}
        }
        xmlXPathFreeObject(result);
    }
    
    g_free(xpath);
    g_free(id_ns);
    g_free(id_name);    
    g_free(id_ns_prefix);
    xmlFree(ns);

    return 0;
}



