/*
 * db_xfunctions.c - function for work with xml data in XML-DB.
 * This file is part of Maemo-DB project.
 *
 * Copyright (C) 2009 - Alexander A. Lomov.
 *
 * Maemo-DB project is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Maemo-DB project is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maemo-DB project; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
#include "db_xml_functions.h"


#define DB_DEBUG_LEVEL 5
#define DB_DEBUG_ON
#ifdef DB_DEBUG_ON
#define DB_DEBUG(level, message ) if (level <= DB_DEBUG_LEVEL) g_debug(message);
#define DB_PDEBUG(level, message, params) if (level <= DB_DEBUG_LEVEL) g_debug(message,params);
#else
#define DB_DEBUG
#define DB_PDEBUG
#endif 



/**************************************************/
/*************** External functions ***************/


// FIXME: write 2 functions: one for "node->next" and second for "node->prev"
// FIXME: or write one function with search parameter
/**
 * @brief Find sibling node for given node with given name.
 *
 * @param start_node first node for start search.
 * @param search_name name for search.
 * @param ns namespace for search.
 *
 * @return TRUE if namespaces is equal or FALSE otherwise.
 */
xmlNodePtr db_find_sibling_node_by_name(const xmlNodePtr start_node, 
                                        const xmlChar* search_name, 
                                        const xmlNsPtr ns)
{
    DB_DEBUG(3, "DB_XMLF: find_sibling_node_by_name START");

    xmlNodePtr node_walker = start_node;

    while (node_walker) {
        if (xmlStrcmp(node_walker->name, search_name) == 0
                &&  db_check_ns_equals(node_walker->ns, ns, FALSE) == TRUE) {
            DB_PDEBUG(4, "DB_XMLF: find_sibling_node_by_name; node_walker: %s", node_walker->name);
            return node_walker;
        }
  
        node_walker = node_walker->next;
    }
    
    node_walker = start_node->prev;
    
    while(node_walker) {
        if(xmlStrcmp(node_walker->name, search_name) == 0
                &&  db_check_ns_equals(node_walker->ns, ns, FALSE) == TRUE) {
            DB_PDEBUG(4, "DB_XMLF: find_sibling_node_by_name; node_walker: %s", node_walker->name);
            
            return node_walker;
        }
  
        node_walker = node_walker->prev;
    }
    
    DB_DEBUG(3, "DB_XMLF: find_sibling_node_by_name END");
    return NULL;
}


//FIXME: need hard test...
/**
 * @brief Add xml document to xml-tree.
 *
 * Get root element from document and add it to tree as last child element. 
 *
 * @param tree xml document for inserting.
 * @param data new xml document.
 * @param stylesheet style for document or NULL.
 * @param params NULL-terminated list of strings or NULL. 
 *        format: name value name value ... NULL.
 * 
 * @return 0 on success or not 0 otherwise.
 */
gint db_add_doc_to_tree(const xmlDocPtr tree, const xmlDocPtr data, 
                        const xsltStylesheetPtr stylesheet, const char** params)
{    
	if (tree == NULL || data == NULL) {
		return ERROR_ARGUMENT; 
	}

	xmlDocPtr inner_data = (stylesheet != NULL) 
        ? xsltApplyStylesheet(stylesheet, data, params) 
	    : xmlCopyDoc(data, 1);

	if (inner_data == NULL) {
		return ERROR_CANT_PARSE_STYLESHEET;
	} 
 
    xmlNodePtr doc_root = xmlDocGetRootElement(inner_data);
    xmlNodePtr tree_root = xmlDocGetRootElement(tree);
 
    if (doc_root == NULL) {
        xmlFreeDoc(inner_data);
        return FAILURE;
    }   
 
    if (tree_root == NULL ) {
        g_message("db_add_doc_to_tree; No root");
        xmlDocSetRootElement(tree, doc_root);  
        xmlDocSetRootElement(inner_data, NULL);
        xmlSetListDoc(doc_root, tree); 
	} else {
        xmlNodePtr tmp_nodes = xmlCopyNodeList(doc_root->children);// xmlDocCopyNodeList(inner_data, doc_root->children);

        tmp_nodes->nsDef = NULL;            /* FIXME: ns links to root node. */
        xmlAddChild(tree_root, tmp_nodes);
        xmlSetListDoc(tmp_nodes, tree); 
//        xmlAddSibling(root->children, t);//new_root->children);
    }

    xmlFreeDoc(inner_data);

	return SUCCESS;
}


// -------------------------------------------------------------------------- //

// FIXME: not my function, need rewrite with normal format.
/**
 * @brief Register namespaces for xpath context.
 *
 * @param xpathCtx the pointer to an XPath context.
 * @param nsList list of known namespaces in 
 *        "<prefix1>=<href1> <prefix2>=href2> ..." format.
 * 
 * @return 0 on success and a negative value otherwise.
 */
gint db_register_xpath_namespaces(xmlXPathContextPtr xpathCtx, 
                                  const xmlChar* nsList) 
{
    xmlChar* nsListDup;
    xmlChar* prefix;
    xmlChar* href;
    xmlChar* next;
    
    nsListDup = xmlStrdup(nsList);
    if (nsListDup == NULL) {
	    g_error("Error: unable to strdup namespaces list");
	    return -1;	
    }
    
    next = nsListDup; 
    
    while (next != NULL) {
	    while ((*next) == ' ') {
	        next++;
	    }
	        
	    if ((*next) == '\0') break;

	    /* find prefix */
	    prefix = next;
	    next = (xmlChar*)xmlStrchr(next, '=');
	
	    if (next == NULL) {
	        g_error("Error: invalid namespaces list format");
	        xmlFree(nsListDup);
	        return -1;	
	    }
	
	    *(next++) = '\0';	
	
	    /* find href */
	    href = next;
	    next = (xmlChar*)xmlStrchr(next, ' ');
	
	    if(next != NULL) {
	        *(next++) = '\0';	
	    }

	    if (xmlXPathRegisterNs(xpathCtx, prefix, href) != 0) {
	        g_error("Error: unable to register NS with prefix=\"%s\" and href=\"%s\"", prefix, href);
	        xmlFree(nsListDup);
	        return -1;	
	    }
    }
    
    xmlFree(nsListDup);
    return 0;
}


/**
 * @brief Gets nodes from xml document using XPath expression.
 *
 * @param doc xml document.
 * @param xpath XPath expression.
 * @param ns_list the list of known namespaces in 
 *        "<prefix1>=<href1> <prefix2>=<href2> ..." format.
 *
 * @return not NULL on success and a NULL value otherwise.
 */
xmlXPathObjectPtr db_get_nodeset(const xmlDocPtr doc, const xmlChar* xpath, 
                                 const xmlChar* ns)
{
    DB_DEBUG(3, "DB_XMLF: db_get_nodeset START");
    if (!xmlDocGetRootElement(doc) || !xpath) {
        g_debug("db_get_nodeset; xpath: %s\nNS: %s", xpath, ns);
        return NULL;
    }
    
	xmlXPathContextPtr context;
	xmlXPathObjectPtr result;
  
	context = xmlXPathNewContext(doc);
	
	if (ns != NULL) {
        db_register_xpath_namespaces(context, ns);
    }    
	
	if (context == NULL) {
		g_debug("Error in xmlXPathNewContext\n");
		return NULL;
	}
	
	result = xmlXPathEvalExpression(xpath, context);
	xmlXPathFreeContext(context);
	
	if (result == NULL) {
		g_debug("Error in xmlXPathEvalExpression\n");
		return NULL;
	}
	
	if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
	    g_message("EMPTY XPATH\n");
		xmlXPathFreeObject(result);
		return NULL;
	}
	
    DB_DEBUG(3, "DB_XMLF: db_get_nodeset END");
	return result;
}


/**
 * @brief Counts number of nodes obtained by the XPath expression.
 *
 * @param doc xml document.
 * @param xpath XPath expression.
 * @param ns the list of known namespaces in 
 *        "<prefix1>=<href1> <prefix2>=<href2> ..." format.
 *
 * @return count of nodes and negative value otherwise.
 */            
gint db_get_nodes_count(const xmlDocPtr doc, const xmlChar* xpath, 
                        const xmlChar* ns )
{
    DB_DEBUG(4, "DB_XMLF: db_get_nodes_count START");
    xmlXPathObjectPtr result = db_get_nodeset(doc, xpath, ns);

    gint count = 0;
        
	if (result != NULL) {
	    count = result->nodesetval->nodeNr;
   		xmlXPathFreeObject(result);
	}    

    DB_PDEBUG(4, "DB_XMLF: db_get_nodes_count END (nodes count = %i)", count);
	return count;
}


/**
 * @brief Removes nodes find by XPath expression from xml document.
 *
 * @param doc xml document.
 * @param xpath_query XPath expression for select nodes witch must be deleted.
 * @param ns namespaces for xpath request.
 */
void db_remove_node_by_xpath(const xmlDocPtr doc, const gchar* xpath_query, const gchar* ns)
{
    DB_PDEBUG(4, "DB_XMLF: db_remove_node_by_xpath START (xpath = %s)", xpath_query);
    xmlXPathObjectPtr result = db_get_nodeset(doc, BAD_CAST xpath_query, BAD_CAST ns);

    if (result != NULL) {
	    xmlNodeSetPtr nodeset = result->nodesetval;
		
        gint size = (nodeset) ? nodeset->nodeNr - 1 : 0;
    
        gint i;
        for (i = size; i >= 0; --i) {
		    xmlUnlinkNode(nodeset->nodeTab[i]);
		    xmlFreeNode(nodeset->nodeTab[i]);
  	        nodeset->nodeTab[i] = NULL;
        }
        xmlXPathFreeObject(result);
    }
    
   
    
    DB_DEBUG(4, "DB_XMLF: db_remove_node_by_xpath END");
}


// -------------------------------------------------------------------------- //


//FIXME rewrite function...  
/**
 * Check nodes name. 
 *
 * @param old_node first node.
 * @param new_node second node.
 *
 * @return 0 if name of nodes names equals and non 0 value otherwise.
 */      
gint db_check_nodes_for_update(const xmlNodePtr old_node, 
                            const xmlNodePtr new_node)
{
    if (old_node == NULL || new_node == NULL) {
        return -1;
    }
    
    if (xmlStrcmp(old_node->name, new_node->name) != 0) {
        return -1;
    }
    
    return 0;    
    
}


/**
 * @brief Check names, namespace and prefixes equals. 
 *
 * @param a first node for check.
 * @param b second node for check.
 * @param check_ns TRUE if need check namespaces.
 * @param check_ns_prefix TRUE if need check namespaces prefixes.
 *
 * @return TRUE if nodes are equal or both is NULL or FALSE otherwise.
 */  
gboolean db_check_nodes_equals(const xmlNodePtr a, const xmlNodePtr b, 
                            const gboolean check_ns,
                            const gboolean check_ns_prefix)
{
    if (a == NULL && b == NULL) {
        return TRUE;
    }

    if (a == NULL || b == NULL) {
        return FALSE;
    }
    
    if (xmlStrcmp(a->name, b->name) || a->type != b->type ) {
        return FALSE;
    }
    
    if (check_ns == TRUE 
            && db_check_ns_equals(a->ns, b->ns, check_ns_prefix) == FALSE) {
        return FALSE;
    }

    return TRUE;
}


/**
 * @brief Check equal of namespaces.
 *
 * Check names and if need prefixes of namespace.
 *
 * @param a first namespace.
 * @param b second namespace.
 * @param check_prefix if TRUE - check prefixes.
 *
 * @return TRUE if namespaces is equal or both is NULL or FALSE otherwise.
 */
gboolean db_check_ns_equals(const xmlNsPtr a, const xmlNsPtr b, 
                         const gboolean check_prefix)
{
    if (a == NULL && b == NULL) {
        return TRUE;
    } else if (a == NULL || b == NULL) {
        return FALSE;
    }
    
    if (xmlStrncmp(a->href, b->href, MAX_STR_BUF_LEN) != 0) {
        return FALSE;
    }
    
    if (check_prefix == TRUE 
            && xmlStrncmp(a->prefix, b->prefix, MAX_STR_BUF_LEN) != 0) {
        return FALSE;
    }

    return TRUE;
}


// -------------------------------------------------------------------------- //


// FIXME: rewrite check nodes type
/**
 * Update one node context from another. 
 *
 * @param old_node node for set the text context.
 * @param new_node node for get the text context.
 *
 * @return 0 on success and not 0 value otherwise.
 */  
gint db_update_node_text_content(const xmlNodePtr old_node, 
                              const xmlNodePtr new_node)
{
    DB_DEBUG(4, "DB_XMLF: db_update_node_text_content START");
    if (db_check_nodes_for_update(old_node, new_node) != 0) {
        return -1;
    }
    
    if (old_node->type != XML_TEXT_NODE) {
       // return -1;
    }
    
    if (old_node->children != NULL) {    
        if (old_node->children->type != XML_TEXT_NODE) {
            return -1;
        }
    }
    
  
    if (new_node->children) {    
        if (new_node->children->type != XML_TEXT_NODE) {
            return -1;
        }
    }

    xmlNodeSetContent(old_node, xmlNodeGetContent(new_node));

    DB_DEBUG(4, "DB_XMLF: db_update_node_text_content END");
    return 0;
}


/**
 * @brief Update node attributes by attributes from other node.
 *
 * Updates founded attributes and adds new (not founded). 
 *
 * @param old_node node with old data.
 * @param new_node npde with new data.
 * 
 * @return 0 on success or not 0 otherwise.
 */
gint db_update_node_attributes(const xmlNodePtr old_node, 
                               const xmlNodePtr new_node) 
{   
    db_check_nodes_for_update(old_node, new_node);

//    xmlAttrPtr old_prop = old_node->properties;
    xmlAttrPtr new_prop = new_node->properties;
    
    while (new_prop != NULL) {
        xmlSetNsProp(old_node, new_prop->ns, new_prop->name, 
                     xmlNodeGetContent(new_prop->children));

      new_prop = new_prop->next;
    }

    return 0;
}


/**
 * @brief Update one node by another. 
 *
 * Change or sets attribtes, add or update child nodes. 
 *
 * @param old_node node for update.
 * @param new_node node for get the data for update.
 *
 * @return 0 on success and not 0 value otherwise.
 */  
gint db_update_node(const xmlNodePtr old_node, xmlNodePtr new_node)
{
    DB_DEBUG(4, "DB_XMLF: update_nodes starts START");
    xmlNodePtr tmp_node = NULL;

    if (db_check_nodes_for_update(old_node, new_node)) {
        return -1;
    }

    while (new_node != NULL) {
        tmp_node = db_find_sibling_node_by_name(old_node, 
                                                new_node->name, new_node->ns);
                                             
        if (tmp_node != NULL) {
            db_update_node_text_content(tmp_node, new_node);
            db_update_node_attributes(tmp_node, new_node);
        } else {
//            xmlUnlinkNode(xmlNodePtr cur) 
            xmlAddPrevSibling(old_node, xmlCopyNode(new_node, 1));    
        }

        if (new_node->children && tmp_node) {
            if (new_node->children->type != XML_TEXT_NODE) {
                db_update_node(tmp_node->children, new_node->children);
            }
        }
        
        tmp_node = NULL;
        new_node = new_node->next;   
    } 
    DB_DEBUG(4, "DB_XMLF: update_nodes starts END");
    return 0;
} 


/**
 * @brief Sets or update property. 
 *
 * Set or update property with or without namespace if prefix and namespace equal NULL.
 *
 * @param node node for set or update property.
 * @param name property name.
 * @param value value of property.
 * @param ns_href namespace href or NULL.
 * @param ns_prefix namespace prefix or NULL.   
 *
 * @return pointer of attribute on success and NULL otherwise.
 */  
xmlAttrPtr db_set_node_property(const xmlNodePtr node, const gchar* name, 
                                const gchar* value, const gchar* ns_href,
                                const gchar* ns_prefix)
{
    if (node == NULL || name == NULL || value == NULL) {
        return NULL;
    }
    
    if (node->type != XML_ELEMENT_NODE) {
        return NULL;
    }

    xmlAttrPtr result = NULL;
    xmlNsPtr ns = NULL;

    if (ns_href != NULL && ns_prefix != NULL) {
        ns = xmlNewNs(NULL, BAD_CAST ns_href, BAD_CAST ns_prefix);
        result = xmlSetNsProp(node, ns, BAD_CAST name, BAD_CAST value);
    } else {
        result = xmlSetProp(node, BAD_CAST name, BAD_CAST value);
    }
    
    if (result == NULL) {
        xmlFreeNs(ns);
    }
    
    return result;
}


// -------------------------------------------------------------------------- //

