/*
 * db_config_manager.c - config manager for Maemo-DB.
 * This file is part of Maemo-DB project.
 *
 * Copyright (C) 2009 - Alexander A. Lomov.
 *
 * Maemo-DB project is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Maemo-DB project is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maemo-DB project; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
#include "db_config_manager.h"


/* Key-value config file and path to file*/
static GKeyFile* config_key_file = NULL;
static gchar* config_path = NULL;

/* Config file posfixes */
#define XSLT_REVERS_FILE_POSTFIX "_rev"
#define ID_NS_POSTFIX "_id_ns"
#define ID_NS_PREFIX_POSTFIX "_id_ns_prefix"
#define ID_NAME_POSTFIX "_id_name"
#define MAX_ID_PATH_POSTFIX "_max_id"

#define REVERS_LINK_POSTFIX "_rev_links"
#define KEY_POSTFIX "_key"
#define LINK_POSTFIX "_link"
#define LINK_ACTIONS_POSTFIX "_actions"

#define WORD_DELIMETER "_"

/* Config file groups */
#define CONFIG_GROUP_XSLT_PATHS "XSLT files paths"
#define CONFIG_GROUP_UNIQUE_COLUMNS "Tables unique columns"
#define CONFIG_GROUP_DB_CONFIG "DB config"
#define CONFIG_GROUP_NAMESPACES "Tables namespaces"
#define CONFIG_GROUP_TABLES_IDS "Tables IDs"
#define CONFIG_GROUP_DB_CONFIG "DB config"
#define CONFIG_GROUP_LINKS "Links"


/* Config files keys */
#define CONFIG_KEY_DEFAULT_DB_PATH "default_db_path"
#define CONFIG_KEY_PROFILES_PATH "profiles_file_path"
#define CONFIG_KEY_SCHEMAS_NAMES "schemas_names"
#define CONFIG_KEY_BINARY_TAGS "binary_tags"


#define HOME_DIR_PREFIX "~/"


/* Not use now
enum db_config_error {
    NO_DB_CONFIG_GROUP = 1,
    NO_XSLT_GROUP = 2
} db_config_error;
*/


static gchar** get_list(const gchar* group_name, const gchar* key_name);
static gint cm_save_config_file(GKeyFile* key_file, const gchar* file_path);

// Not use now 
//static gint check_config(GKeyFile* key_file); 


/**************************************************/
/*************** External functions ***************/

/**
 * @brief Loads config file for given path. 
 *
 * @param file_path path for config file.
 *
 * @return 0 on success and not 0 value otherwise.
 */  
gint cm_load_config(const gchar* file_path)
{
    if (config_key_file != NULL) { 
    	g_key_file_free(config_key_file);
    }

    config_key_file = g_key_file_new();

    if (g_key_file_load_from_file(config_key_file, file_path,
			                	  G_KEY_FILE_KEEP_COMMENTS, NULL) == FALSE) {
    	g_key_file_free(config_key_file);
	    config_key_file = NULL;
        
        return ERROR_CANT_OPEN_CONFIG_FILE;
    }
    
    config_path = g_strdup(file_path);
    
    return SUCCESS;
}


/**
 * @brief Saves config file on given path. 
 *
 * @param file_path path for save.
 * 
 * @return 0 on success and not 0 value otherwise.
 */  
gint cm_save_config_to_file(const gchar* file_path)
{
    return cm_save_config_file(config_key_file, file_path);
}


/**
 * @brief Saves config. 
 *
 * @param file_path path for save.
 * 
 * @return 0 on success and not 0 value otherwise.
 */  
gint cm_save_config()
{
    return cm_save_config_file(config_key_file, config_path);
}


/**
 * @brief Closes config file. 
 *
 */  
void cm_close_config()
{
    g_key_file_free(config_key_file);
    g_free(config_path);
    config_path = NULL;
    config_key_file = NULL;
}


// -------------------------------------------------------------------------- //


/**
 * @brief Gets db path by name or default path of data base. 
 *
 * @param db_name name of data base.
 *
 * @return path of data base or NULL.
 */  
gchar* cm_get_db_path_by_name_or_default(const gchar* db_name)
{
    gchar* path = NULL;

    if (db_name != NULL) {  
        path = cm_get_db_path_by_name(db_name);
    }
    
    if (path == NULL) {
        path = cm_get_default_db_path();
    }

    return path;
}       


/**
 * @brief Gets db path by name. 
 *
 * Get path for given data base name.
 *
 * @param db_name name of data base.
 *
 * @return path of data base or NULL.
 */  
gchar* cm_get_db_path_by_name(const gchar* db_name)
{
    if (db_name == NULL) {
        return NULL;
    }

    return g_key_file_get_value(config_key_file, CONFIG_GROUP_DB_CONFIG,
                                db_name, NULL);
}          


/**
 * @brief Gets default db path. 
 *
 * @return path of data base or NULL.
 */
gchar* cm_get_default_db_path()
{
    gchar* path = g_key_file_get_value(config_key_file, CONFIG_GROUP_DB_CONFIG,
                                       CONFIG_KEY_DEFAULT_DB_PATH, NULL);

    if (g_str_has_prefix(path, HOME_DIR_PREFIX) == TRUE) {
        gchar** split_path = g_strsplit(path, HOME_DIR_PREFIX, 2);

        g_free(path);

        if (g_strv_length (split_path) == 2) {
            path = g_strconcat(HOME_DIR, G_DIR_SEPARATOR_S, split_path[1], NULL);
        } else {
            path = NULL;
        }
        g_strfreev(split_path);
    }

    return path;
}                                             


/**
 * @brief Sets db path by name. 
 *
 * Set path for given data base name.
 *
 * @param db_name name of data base.
 * @param db_path path of data base.
 */  
void cm_set_db_path_by_name(const gchar* db_name, const gchar* db_path)
{   
    g_key_file_set_value(config_key_file, CONFIG_GROUP_DB_CONFIG,
                         db_name, db_path);
}                                                                                



/**
 * @brief Gets db path by name or default path of data base. 
 *
 * @param db_name name of data base.
 *
 * @return path of data base or NULL.
 */
gchar* cm_get_db_config_key(const gchar* key_name)
{
     return g_key_file_get_string(config_key_file, CONFIG_GROUP_DB_CONFIG,
			                      key_name, NULL);
}


/**
 * @brief Gets db profile path. 
 *
 * @return path of data base profile file.
 */
gchar* cm_get_profiles_file_path()
{
    return cm_get_db_config_key(CONFIG_KEY_PROFILES_PATH);
}


/**
 * @brief Gets all declared shemas names. 
 *
 * @return string array with schemas names on success or NULL otherwise.
 */ 
gchar** cm_get_all_schemas_names()
{
    return get_list(CONFIG_GROUP_DB_CONFIG, CONFIG_KEY_SCHEMAS_NAMES);			                              
}


/**
 * @brief Gets binary tags names. 
 *
 * @return string array with tags names on success or NULL otherwise.
 */
gchar** cm_get_all_binary_tags_names()
{
    return get_list(CONFIG_GROUP_DB_CONFIG,CONFIG_KEY_BINARY_TAGS);			                              
}


// -------------------------------------------------------------------------- //


/**
 * @brief Gets path for xslt file associated with given data schema. 
 *
 * Get xslt file path, that using for transform to inner format of data base. 
 *
 * @param schema_name schema name.
 *
 * @return xslt file path on success or NULL otherwise.
 */
gchar* cm_get_xslt_path_key(const gchar* schema_name)
{
    return g_key_file_get_string(config_key_file, CONFIG_GROUP_XSLT_PATHS,
			                     schema_name, NULL);

}


/**
 * @brief Gets path for revers xslt file associated with given data schema. 
 *
 * Get revers xslt file path, that using for transform  from inner format to external. 
 *
 * @param schema_name schema name.
 *
 * @return revers xslt file path on success or NULL otherwise.
 */
gchar* cm_get_revers_xslt_path_key(const gchar* schema_name)
{
    gchar* xslt_path_key = g_strconcat(schema_name, XSLT_REVERS_FILE_POSTFIX, NULL);
    
    gchar* result = g_key_file_get_string(config_key_file, 
                                          CONFIG_GROUP_XSLT_PATHS,
			                              xslt_path_key, NULL);
			      
	g_free(xslt_path_key);		                              

	return result;
}


// -------------------------------------------------------------------------- //


/**
 * @brief Gets array with xpath-queries that must be unique for given data schema. 
 *
 * @param schema_name schema name.
 *
 * @return string array with queries on success or NULL otherwise.
 */
gchar** cm_get_unique_columns_keys(const gchar* schema_name)
{
    return g_key_file_get_string_list(config_key_file, 
                                      CONFIG_GROUP_UNIQUE_COLUMNS,
			                          schema_name, NULL, NULL);
}




/**
 * @brief Gets namesapaces associated with given data schema. 
 *
 * @param schema_name schema name.
 *
 * @return string with namespaces on success or NULL otherwise.
 */
gchar* cm_get_namespaces_key(const gchar* schema_name)
{
    return g_key_file_get_string(config_key_file, CONFIG_GROUP_NAMESPACES,
			                     schema_name, NULL);

}


// -------------------------------------------------------------------------- //


/**
 * @brief Gets xpath query for get tag with id associated with given schema. 
 *
 * @param schema_name schema name.
 *
 * @return xpath quere on success or NULL otherwise.
 */
gchar* cm_get_table_id_path_key(const gchar* schema_name)
{
     return g_key_file_get_string(config_key_file, CONFIG_GROUP_TABLES_IDS,
			                      schema_name, NULL);
}


/**
 * @brief Gets id name associated with given data schema. 
 *
 * @param schema_name schema name.
 *
 * @return id name on success or NULL otherwise.
 */
gchar* cm_get_table_id_name_key(const gchar* schema_name)
{
    gchar* id_name = g_strconcat(schema_name, ID_NAME_POSTFIX, NULL);

    gchar* result = g_key_file_get_string(config_key_file, 
                                          CONFIG_GROUP_TABLES_IDS,
			                              id_name, NULL);
   	g_free(id_name);		

    return result;			                              
}


/**
 * @brief Gets xpath query for get max id associated with given data schema. 
 *
 * Get xslt file path, that using for transform to inner format of data base. 
 *
 * @param schema_name schema name.
 *
 * @return xslt file path on success or NULL otherwise.
 */
gchar* cm_get_table_max_id_path_key(const gchar* schema_name)
{
    gchar* max_id_path = g_strconcat(schema_name, MAX_ID_PATH_POSTFIX, NULL);

    gchar* result = g_key_file_get_string(config_key_file, 
                                          CONFIG_GROUP_TABLES_IDS,
			                              max_id_path, NULL);
   	g_free(max_id_path);		

    return result;			                              
}


/**
 * @brief Gets namespace for id. 
 *
 * @param schema_name schema name.
 *
 * @return id namespace on success or NULL otherwise.
 */
gchar* cm_get_table_id_ns_key(const gchar* table_schema) {
    gchar* id_ns = g_strconcat(table_schema, ID_NS_POSTFIX, NULL);

    gchar* result = g_key_file_get_string(config_key_file, 
                                          CONFIG_GROUP_TABLES_IDS,
			                              id_ns, NULL);
   	g_free(id_ns);		

    return result;			                              
}


/**
 * @brief Gets namespace prefix for id associated with given data schema. 
 *
 * @param schema_name schema name.
 *
 * @return id namespace on success or NULL otherwise.
 */
gchar* cm_get_table_id_ns_prefix_key(const gchar* schema_name) 
{
    gchar* id_ns_prefix = g_strconcat(schema_name, ID_NS_PREFIX_POSTFIX, NULL);

    gchar* result = g_key_file_get_string(config_key_file, 
                                          CONFIG_GROUP_TABLES_IDS,
			                              id_ns_prefix, NULL);
   	g_free(id_ns_prefix);		

    return result;			                              
}


// -------------------------------------------------------------------------- //


/**
 * @brief Gets names of linked datas schemas. 
 *
 * @param schema_name schema name.
 *
 * @return string array with names on success or NULL otherwise.
 */
gchar** cm_get_links_names(const gchar* schema_name)
{
    return get_list(CONFIG_GROUP_LINKS, schema_name);
}


/**
 * @brief Gets names of revers-linked data schemas. 
 *
 * @param schema_name schema name.
 *
 * @return string array with names on success or NULL otherwise.
 */
gchar** cm_get_revers_links_names(const gchar* schema_name)
{
    gchar* key = g_strconcat(schema_name, REVERS_LINK_POSTFIX, NULL);
    
    gchar** result = get_list(CONFIG_GROUP_LINKS, key);
    
    g_free(key);

    return result;
}


/**
 * @brief Gets links data schema. 
 *
 * Gets xpath-queries indicates links of given schema for given link (linked schema).
 *
 * @param schema_name schema name.
 * @param schema_name linked schema name.
 *
 * @return string array with qpath-queries on success or NULL otherwise.
 */
gchar** cm_get_links(const gchar* schema_name, const gchar* linked_schema_name)
{
    gchar* key = g_strconcat(schema_name, WORD_DELIMETER, linked_schema_name, LINK_POSTFIX, NULL);

    gchar** result = get_list(CONFIG_GROUP_LINKS, key);
    
    g_free(key);

    return result;			                              
}


/**
 * @brief Gets keys of link for data schema. 
 *
 * Gets xpath-queries indicates keys of given schema for given link (linked schema).
 *
 * @param schema_name schema name.
 * @param schema_name linked schema name.
 *
 * @return string array with qpath-queries on success or NULL otherwise.
 */
gchar** cm_get_keys(const gchar* schema_name, const gchar* linked_schema_name)
{
    gchar* key = g_strconcat(schema_name, WORD_DELIMETER, linked_schema_name, KEY_POSTFIX, NULL);

    gchar** result = get_list(CONFIG_GROUP_LINKS, key);
    
    g_free(key);
    return result;			                              
}


/**************************************************/
/***************** Static functions ****************/

/**
 * @brief Gets string value array from key-value file. 
 *
 * @param group_name name of group with values.
 * @param key_name key-name of value.   
 *
 * @return string array on success or NULL otherwise.
 */
static gchar** get_list(const gchar* group_name, const gchar* key_name)
{
    gsize size = 0;

    gchar** result = g_key_file_get_string_list(config_key_file, group_name,
			                                    key_name, &size, NULL);

    if (size <= 0) {
        g_strfreev(result);
        result = NULL;
    }			                                 

    return result;
}


/**
 * @brief Saves config file on given path. 
 *
 * @param key_file key-value file.
 * @param file_path path for save.
 * 
 * @return 0 on success and not 0 value otherwise.
 */  
static gint cm_save_config_file(GKeyFile* key_file, const gchar* file_path)
{
    gsize length = 0;
    
    if (file_path == NULL || key_file == NULL) {
        return FAILURE;
    }
    
    gchar* data = g_key_file_to_data(key_file, &length, NULL);
   
    if (g_file_set_contents(file_path, data, length, NULL) == FALSE) {
        g_free(data);

        return FAILURE;
    }
   
    g_free(data);
    
    return SUCCESS;
}

 
 
/********************************************************/
/***************** Not use now functions ****************/
 
/**
 * @brief Gets actions for link.
 *
 * Gets xpath-queries indicates keys of given schema for given link (linked schema).
 *
 * @param schema_name schema name.
 * @param schema_name linked schema name.
 *
 * @return string array with qpath-queries on success or NULL otherwise.

gint* cm_get_links_actions(const gchar* schema_name, const gchar* linked_schema_name)
{
    gchar* key = g_strconcat(schema_name, WORD_DELIMETER, linked_schema_name,  
                             LINK_ACTIONS_POSTFIC, NULL);
    gsize size = 0;
    gint* result = g_key_file_get_integer_list(config_key_file, CONFIG_GROUP_LINKS, key, &size, NULL);

    if (size == 0) {
        g_free(result);
        result = NULL;
    }		

    g_free(key);
    return result;			                              
}
 */
 
 
/*
 * @brief Check config file (not use). 
 *
 * @param key_file key-value parser.
 *
 * @return 0 on success and not 0 value otherwise.
    
static gint check_config(GKeyFile* key_file)
{
    gint error_flag = 0;
    
    if(!g_key_file_has_group(key_file, CONFIG_GROUP_DB_CONFIG))
        error_flag |= NO_DB_CONFIG_GROUP;
        
    if(!g_key_file_has_group(key_file, CONFIG_GROUP_XSLT_PATHS))
        error_flag |= NO_XSLT_GROUP;
        
    if(!g_key_file_has_group(key_file, CONFIG_GROUP_XSLT_PATHS))
        error_flag |= NO_XSLT_GROUP;        

    return error_flag;

}
 */

