/*
 * db_msa_logic.h - interface fo msa db logic.
 * This file is part of DB-Maemo.
 *
 * Copyright (C) 2009 - Ilia L. Burlak
 *
 * DB-Maemo is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DB-Maemo is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DB-Maemo; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */

#include "db.h"
#include "db_config_manager.h"

#define DB_NAME "data_base"
#define DB_SEND db_send
#define DB_SHUTDOWN db_shutdown
#define DB_START db_start

/** format gconf xml **/
const char* const_gconf_xml = "\
<ListDrivers>\
</ListDrivers>\
";


/**
 * @brief prepare data base for work.
 * @param msa_module
 * @return not 0 if error.
 */
int msa_module_init(msa_module* db)
{ 
	g_debug("db_initialization: START");
    db->name = DB_NAME;
	db->module_start = DB_START;
    db->send = DB_SEND;
    db->shutdown = DB_SHUTDOWN;

	g_debug("db_initialization: 1");

    if (cm_load_config(CONFIG_FILE_PATH)) {
		g_debug("db_initialization: 33");    
	    db->status = OFF;
        db->state = BROKEN;
        return 1;
    }
    
    xmlSubstituteEntitiesDefault(1);
    xmlLoadExtDtdDefaultValue = 1;

    xmlInitParser();
    xsltInit();
    db_prepare(NULL);
    db->state = ON; 
    db->status = CONF;

	g_debug("db_initialization: END");

    return 0;
}

/**
 * @brief shutdow db 
 * @return 0 
**/
int db_shutdown()
{
    g_debug("db_shutdown START");
    cm_close_config();
    xsltCleanupGlobals();
    xmlCleanupParser();
    g_debug("db_shutdown END");
 
    return 0;
}

/**
 * @brief start db, set all enviromen variable
 * @param kernel - pointer msa_module
 * @param list_drv - pointer GList  
 * @return if start withot error 0  
**/
int db_start(gpointer _kernel, gpointer _list_drv, int argc, char* argv[])
{
	g_debug("db_start: START");

    // un used in this function
    _kernel = NULL;

    if (argc != 0)
    {
        g_debug("db parametrs - %s", argv[0]);
    }

	gboolean checked;
	GList* db_list_driver = (GList*)_list_drv;
	GList* node_glist;
	msa_module* info;
	xmlDocPtr db_xml_drivers = xmlParseDoc(BAD_CAST const_gconf_xml);
	xmlNodePtr node_root = xmlDocGetRootElement(db_xml_drivers);
	xmlNodePtr node;
	GConfClient* client;
	xmlChar* str_gconf = NULL;
	gint size;	

	/** save list drivers in gconf in xml format **/
	for (node_glist = db_list_driver; node_glist != NULL; node_glist = node_glist->next) {

		info = (msa_module*)node_glist->data;
		node = xmlNewNode(NULL,BAD_CAST DRIVER_TAG);
    
        char *drv_status = g_strdup_printf("%d", info->status);
        char *drv_state = g_strdup_printf("%d", info->state);

		xmlSetProp(node, BAD_CAST STATUS_PROP, BAD_CAST drv_status);
		xmlSetProp(node, BAD_CAST STATE_PROP, BAD_CAST drv_state);

        free(drv_status);
        free(drv_state);

		xmlNodeSetContent(node, BAD_CAST info->id);
		xmlAddChild(node_root, node);
		
	}

	xmlDocDumpMemory(db_xml_drivers, &str_gconf, &size);
	client = gconf_client_get_default ();
	gchar* gconf_key = g_strconcat(DB_PATH_GCONF, DB_LIST_DRIVERS, NULL);
	checked = gconf_client_set_string (client, gconf_key,
                                       (gchar*)str_gconf, NULL);
    g_free(gconf_key);                                       

	if (!checked) {
		g_debug("db_start: Error: couldn't create gconf settings!");
		xmlFreeDoc(db_xml_drivers);
		xmlFree(str_gconf);
		return 1;	
	}

	g_debug("db_start: END\n %s", str_gconf);
	xmlFreeDoc(db_xml_drivers);
	xmlFree(str_gconf);
	return 0;
}


/**
 * @brief send db, handler function 
 * @param request - xmlDocPtr
 * @param response - xmlDocPtr  
 * @param info - null not use in this module 
 * @return if start withot error 0  
**/
int db_send(xmlDocPtr request, xmlDocPtr *response, msa_module* db_info)
{
	g_debug("db_send: START");


        db_info = NULL;

        if (request == NULL) {
		g_debug("request is null");
		return 0;
	}

	//xmlDocPtr request = xmlCopyDoc(_request, 1);
	//xmlFreeDoc(_request);

	/** get class_name and function_name **/	
	curent_func* db_func = db_get_curent_func(request);
	
	if (db_func == NULL) {
		g_debug("db_send: error in request and response");
		xmlFreeDoc(request);
		return 1;
	}

	/** call process request and get response **/
	if (db_func->cfunc != NULL) {
		*response = db_func->cfunc(request);
                //xmlDocDump(stdout, *response);
        } else {
                /* if recive messages*/
		rem_targets(request);
		add_target(request, (xmlChar*)UI_ID);
		*response = request;
	}
	
        free_curent_func(db_func);
	g_debug("\ndb_send: END");	
	return 0;
}

/**
 * @brief generate xpath request to xmlDocPtr 
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
xmlXPathObject* db_xpath(char* req, xmlDocPtr doc)
{
    xmlXPathObject *Obj;
	xmlXPathContextPtr Ctx;
	Ctx = xmlXPathNewContext(doc);
	if (Ctx == NULL) {
		g_debug("dp_xpath: error wrong request format!");
		return NULL;	
	}
	Obj = xmlXPathEvalExpression((xmlChar*)req, Ctx);
    
	xmlXPathFreeContext(Ctx);
	return Obj;
}

/**
 * @brief create new curent func for curent request 
 * @param doc - xmlDocPtr request
 * @return curent_func 
**/
curent_func* db_get_curent_func(xmlDocPtr doc)
{
	curent_func* db_func;

	g_debug("\ndb_get_curent_func: START");
	xmlXPathObject *Obj;
	xmlNodePtr node;

	if ((db_func = create_curent_func()) == NULL) {
	    g_debug("db_get_curent_func: memmory allocation error");
	    return NULL;
	}

	/** get source node **/
	Obj = db_xpath(SOURCE_TAG_XPATH, doc);
	
	if (Obj == NULL) {
		g_debug("db_get_curent_func: response or request ERROR !");
		free_curent_func(db_func);
		return NULL;
	}

	if (Obj->nodesetval->nodeNr != 0) {
	
		node = Obj->nodesetval->nodeTab[0];

		db_func->source_id = (gchar*)xmlNodeGetContent(node);

		g_debug("db_get_curent_func: node - %s", node->name);
		g_debug("db_get_curent_func: source_id = %s", db_func->source_id);
	
	}
	
	xmlXPathFreeObject(Obj);
	
	/** get request node if source id != ui then Response else Request**/
	if (strcmp(db_func->source_id, UI_ID) != 0) {
		g_debug("db_get_curent_func: response");
		Obj = db_xpath(RESPONSE_TAG_XPATH, doc);		
	} else {
		g_debug("db_get_curent_func: request");		
		Obj = db_xpath(REQUEST_TAG_XPATH, doc);
	} 

	if (Obj->nodesetval->nodeNr == 0) {
		g_debug("db_get_curent_func: response or request ERROR !");
		free_curent_func(db_func);
    		xmlXPathFreeObject(Obj);
		return NULL;	
	}

	if (Obj->nodesetval->nodeNr != 0) {
			
		node = Obj->nodesetval->nodeTab[0];
	
		db_func->class_name = (gchar*)xmlGetProp(node, BAD_CAST "class");	
		db_func->func_name = (gchar*)xmlGetProp(node, BAD_CAST "function");
	
		g_debug("db_get_curent_func: node - %s", node->name);
		g_debug("db_get_curent_func: class = %s func = %s", 
					db_func->class_name, db_func->func_name);
				
	}
	
	g_debug("db_get_curen_func:  %s %s",db_func->class_name, db_func->func_name);

	if (db_select_curent_func(db_func) == 1) {
		g_debug("db_get_curent_func: error no function");
		db_func->cfunc = NULL;
	}

	g_debug("db_get_curent_func: END");
        xmlXPathFreeObject(Obj);

	return db_func;
}

/**
 * @brief select func 
 * @param func - pointer on struct curent func
 * @return curent_func 
**/
int db_select_curent_func(curent_func* func)
{
	
	g_debug("db_select_curent_func: START %s %s",func->class_name, func->func_name);
	
	/** if class == profile **/
	if (strcmp(func->class_name, CLASS_PROFILE) == 0) {

		/** if fuction == updatProfile **/
		if (strcmp(func->func_name, UPDATE_PROFILE) == 0) {

			/** if source id == ui then send request to drivers**/
			if (strcmp(func->source_id, UI_ID) == 0) {

				g_debug("db_select_curent_func: function %s",  UPDATE_PROFILE);
				func->cfunc = DB_SEND_REQUEST;		
				return 0;

			} else {
			/** if source id != ui then bd think it's response and try save*/

				g_debug("db_select_curent_func: function %s", UPDATE_PROFILE);
				func->cfunc = DB_SAVE_PROFILE;		
				return 0;
			
			}
		}// updateProfile
	
		/** if fuction == getProfile **/
		if (strcmp(func->func_name, GET_PROFILE) == 0) {

				func->cfunc = DB_GET_PROFILE;		
				return 0;		
		}
		
	}

	/** if class == friends **/
	if (strcmp(func->class_name, CLASS_FRIENDS) == 0) {
		
		/** if fuction == updatFriends **/
		if (strcmp(func->func_name, UPDATE_FRIENDS) == 0) {

			/** if source id == ui then send request to drivers**/
			if (strcmp(func->source_id, UI_ID) == 0) {

				g_debug("db_select_curent_func: function %s", UPDATE_FRIENDS);
				func->cfunc = DB_SEND_REQUEST;		
				return 0;

			} else {
			/** if source id != ui then bd think it's response and try save*/

				g_debug("db_select_curent_func: function %s", UPDATE_FRIENDS);
				func->cfunc = DB_SAVE_FRIENDS;		
				return 0;
			
			}
		}// updateListFriends

		/** if fuction == getListFriends **/
		if (strcmp(func->func_name, GET_FRIENDS) == 0) {

				func->cfunc = DB_GET_FRIENDS;		
				return 0;		
		}

	
	}

	/** if class == binaryData **/
	if (strcmp(func->class_name, CLASS_BINARY_DATA) == 0) {

		/** if fuction == getBinaryData **/
		if (strcmp(func->func_name, GET_BINARY_DATA) == 0) {

			/** if source id == ui then send request to drivers**/
			if (strcmp(func->source_id, UI_ID) == 0) {

				g_debug("db_select_curent_func: function %s get", GET_BINARY_DATA);
				func->cfunc = DB_SEND_REQUEST;		
				return 0;

			} else {
			/** if source id != ui then bd think it's response and try save*/

				g_debug("db_select_curent_func: function %s save", GET_BINARY_DATA);
				func->cfunc = DB_SAVE_BINARY_DATA;		
				return 0;
			
			}	
		}

	}
	
	/** if class == Message **/
	if (strcmp(func->class_name, CLASS_BOX_MESSAGE) == 0) {
	    
	
	    /** if fuction == update_inbox or upadte outox **/
		if (strcmp(func->func_name, UPDATE_INBOX_MESSAGES) == 0 || strcmp(func->func_name, UPDATE_OUTBOX_MESSAGES) == 0) {
		
		    /** if source id == ui then send request to drivers**/
            if (strcmp(func->source_id, UI_ID) == 0) {

                g_debug("db_select_curent_func: function db message update");
                func->cfunc = DB_SEND_REQUEST;		
                return 0;

            } else {
                /** if source id != ui then bd think it's response and try save*/
                g_debug("db_select_curent_func: function db message save");
                func->cfunc = DB_SAVE_INBOX_MESSAGES;		
                return 0;

            }
        }

        /** if fuction == getBinaryData **/
		if (strcmp(func->func_name, GET_NEW_MESSAGES) == 0) {
		
                g_debug("db_select_curent_func: get_new_message");
                func->cfunc = GET_NEW_MESSAGES_FUNC;
                return 0;

        }

        func->cfunc = DB_GET_INBOX_MESSAGES;
        g_debug("db: fucntion GET %s %s", (char*)DB_GET_INBOX_MESSAGES, (char*)func->func_name);
        	
        return 0;
        
	}

	g_debug("db_select_curent_func: END");
	func->cfunc = NULL;
	return 0;
}

/**
 * @brief Add target to transit data.
 *
 * @param doc xml document describing transit data.
 * @param target_id identifier of target.
 * @param order_number order number.
 */
void add_target(const xmlDocPtr doc, const xmlChar* target_id)
{
    xmlNodePtr root_node = xmlDocGetRootElement(doc);
    if (root_node == NULL || target_id == NULL) {
        return;
    }
    
    if (xmlStrlen(target_id) == 0) {
        return;
    }
    
    xmlNodePtr target = xmlNewNode(NULL, BAD_CAST TARGET_TAG);
    xmlNodeSetContent(target, BAD_CAST target_id);
    
    xmlAddChild(root_node, target);
}

/**
 * @brief remove target node in request.
 *
 * @param doc xml document describing transit data.
 */
void rem_targets(xmlDocPtr doc)
{
    g_debug("rem_targets START");
    xmlXPathObjectPtr result = get_nodeset(doc, BAD_CAST TARGET_TAG_XPATH);

    if (result == NULL) {
        return;
    }
     
    xmlNodeSetPtr nodeset = result->nodesetval;
   
    gint i = 0;
    
    for (i = 0; i < nodeset->nodeNr; ++i) {
        xmlUnlinkNode(nodeset->nodeTab[i]);
        xmlFreeNode(nodeset->nodeTab[i]);
        nodeset->nodeTab[i] = NULL;
    }
    xmlXPathFreeObject(result);
    
    g_debug("rem_targets END");
    return;
}


/**
 * @brief Gets nodes from xml document using XPath expression.
 *
 * @param doc xml document.
 * @param xpath XPath expression.
 *
 * @return find nodes or NULL otherwise.
 */
xmlXPathObjectPtr get_nodeset(const xmlDocPtr doc, const xmlChar* xpath)
{
    if (xmlDocGetRootElement(doc) == NULL || xpath == NULL) {
        g_debug("db_get_nodeset; xpath: %s\n", xpath);
        return NULL;
    }
	xmlXPathContextPtr context = xmlXPathNewContext(doc);
   
	if (context == NULL) {
		g_error("Error in xmlXPathNewContext\n");
		return NULL;
	}
	
	xmlXPathObjectPtr result = xmlXPathEvalExpression(xpath, context);
	xmlXPathFreeContext(context);
	
	if (result == NULL) {
		g_error("Error in xmlXPathEvalExpression\n");
		return NULL;
	}
	
	if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
	    g_debug("EMPTY XPATH\n");
		xmlXPathFreeObject(result);
		return NULL;
	}
	
	return result;
}

/**
 * @brief Clean and destroy curent_func structure.
 *
 * @param fnc to curent_func structure.
 */
void free_curent_func(curent_func *fnc)
{
    g_free(fnc->source_id);
    g_free(fnc->target_id);
    g_free(fnc->class_name);
    g_free(fnc->func_name);
    g_free(fnc);
}

/**
 * @brief Create empty curent_func structure.
 *
 * @return new instance of curent_func structure.
 */
curent_func *create_curent_func()
{
    curent_func *ret = g_malloc(sizeof(curent_func));
    if (ret != NULL)
    {
	ret->source_id = NULL;
        ret->target_id = NULL;
        ret->class_name = NULL;
        ret->func_name = NULL;
        ret->cfunc = NULL;
    }
    return ret;
}
