#include "servicemgr.h"

ServiceMgr::ServiceMgr(QObject *parent) :
        QObject(parent), friendsUpdate(0), albumsUpdate(0), photosUpdate(0)
{
    settings = new QSettings(Utils::getHomeDir() + QString(FILE_SETTINGS), QSettings::NativeFormat, this);
    if (!settings->contains(SETTINGS_PROXY))
        settings->setValue(SETTINGS_PROXY, this->SystemProxy);
    loadAccounts();

    qDebug()<< "loaded " << this->accounts.length() << " accounts";

    updateDriverSettings();

    qRegisterMetaType<FriendList>("FriendList");
    qRegisterMetaType<AlbumList>("AlbumList");
    qRegisterMetaType<PhotoList>("PhotoList");
    qRegisterMetaType<Friend>("Friend");
    qRegisterMetaType<VkTransport::Action>("VkTransport::Action");

    for(int i = 0; i < this->accounts.length(); i++) {
        QObject::connect(this->accounts.at(i)->transport, SIGNAL(friendsReceived(QString, FriendList, bool)), this, SLOT(gotFriends(QString, FriendList, bool)));
        QObject::connect(this->accounts.at(i)->transport, SIGNAL(albumsReceived(QString, QString, AlbumList, bool)), this, SLOT(gotAlbumList(QString, QString, AlbumList, bool)));
        QObject::connect(this->accounts.at(i)->transport, SIGNAL(photosReceived(QString, QString, QString, PhotoList, bool)), this, SLOT(gotPhotoList(QString, QString, QString, PhotoList, bool)));
        QObject::connect(this->accounts.at(i), SIGNAL(updateProfile(QString,Friend)), this, SLOT(gotProfile(QString,Friend)));
        QObject::connect(this->accounts.at(i)->transport, SIGNAL(errorOccurred(QString,int,QString, VkTransport::Action)), this, SLOT(gotErrorMsg(QString,int,QString, VkTransport::Action)));

        // load profiles for each account
        this->accounts[i]->getProfile(false);
    }

    connect(this, SIGNAL(updateAccounts(AccountList)), this, SLOT(storeAccounts(AccountList)));

}

ServiceMgr::ServiceMgr(const ServiceMgr &src) : QObject(src.parent())
{
    this->accounts = src.accounts;
    this->threads = src.threads;
    this->settings = src.settings;
    this->friendsUpdate = src.friendsUpdate;
    this->albumsUpdate = src.albumsUpdate;
    this->photosUpdate = src.photosUpdate;
}

ServiceMgr::~ServiceMgr()
{
    for (int i = 0; i < this->accounts.length(); i++)
    {
        delete this->accounts.at(i);
    }
}

void ServiceMgr::storeAccounts(AccountList)
{
    QDomDocument out("MyDoc");
    QDomElement profile = out.createElement(NODE_ACCOUNTS_ROOT);
    out.appendChild(profile);
    for (int i = 0; i < this->accounts.length(); i++) {
        Account *curAcc = this->accounts[i];
        curAcc->saveAccount();
        QDomElement anode = out.createElement(NODE_ACCOUNT);
        QDomText t = out.createTextNode(curAcc->transport->accountId);
        anode.appendChild(t);
        profile.appendChild(anode);
    }

    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    qDebug() << file.fileName();
    if (file.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate) == false) {
        qDebug() << "can't open file: " + file.errorString();
    }
    file.write(out.toByteArray());
    file.close();
}

void ServiceMgr::loadAccounts()
{
    QDomDocument doc;
    QFile file(Utils::getHomeDir() + QString(FILE_ACCOUNTS_LIST));
    if (file.open(QIODevice::ReadOnly) == false) {
        qDebug() << "can't open file: " + file.errorString();
        return;
    }

    doc.setContent(&file);
    file.close();

    QDomElement doc_elem = doc.documentElement();
    QDomNodeList lst = doc_elem.elementsByTagName("profile");

    qDebug() << "accounts:" << lst.count();
    for (int i = 0; i < lst.count(); i++)
    {
        QString accId = lst.at(i).firstChild().nodeValue();
        qDebug() << "Load account:" << accId;
        Account *prf = Account::loadAccount(accId);

        if (prf != NULL)
            this->accounts.append(prf);
    }
}

void ServiceMgr::cleanThreads() {
    QList<QString> keys;
    QHashIterator<QString, QFuture<void> > i(this->threads);
    while (i.hasNext()) {
        i.next();
        if (i.value().isFinished())
            keys.append(i.key());
    }

    for(int i = 0; i < keys.length(); i++) {
        this->threads.remove(keys[i]);
    }
}

void ServiceMgr::updateDriverSettings() {
    QString curProxyHost = QString("");
    int curProxyPort = 0;

    ProxyType proxy = (ProxyType)settings->value(SETTINGS_PROXY).toInt();

    switch (proxy) {
    case ServiceMgr::NoProxy:
        break;
    case ServiceMgr::UserProxy:
        curProxyHost = settings->value(SETTINGS_PROXYHOST).toString();
        curProxyPort = settings->value(SETTINGS_PROXYPORT).toInt();
        break;
    case ServiceMgr::SystemProxy: {
        QNetworkProxyQuery npq(QUrl(QLatin1String("http://maemo.org")));
        QList<QNetworkProxy> proxyList = QNetworkProxyFactory::systemProxyForQuery(npq);
        curProxyHost = proxyList.at(0).hostName();
        curProxyPort = proxyList.at(0).port();
        if (curProxyHost.isEmpty())
            curProxyHost = QString::null;
        break;
    }
    default:
        qWarning() << "Unknown type of proxy";
    }
    qDebug() << "using proxy: \""<< curProxyHost << "\":" << curProxyPort;

    for (int i = 0; i < this->accounts.length(); i++) {
        this->accounts.at(i)->setProxy(curProxyHost, curProxyPort);
    }

}

FriendList ServiceMgr::getFriends(bool isNeedUpdate, bool useSignal){
    FriendList ret;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        FriendList curList = this->accounts.at(i)->getFriendList();
        ret.append(curList);
        if (isNeedUpdate || curList.isEmpty()) {
            QString threadId = QString("getFriends") + this->accounts[i]->transport->accountId;
            // check that thread shouldn't already running
            if (!this->threads.contains(threadId) && this->accounts.at(i)->isNetworkEnabled) {
                // add thread to list
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &VkTransport::vkGetFriends);
                this->threads.insert(threadId, status);
                this->friendsUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updateFriends(ret, this->friendsUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotFriends(QString accountid, FriendList list, bool isLastUpdate){

    // create list of merged lists of friends
    FriendList ret;
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountid) {
            this->accounts[i]->setFriendList(list);
            ret.append(list);
            qDebug() << "Add to friends " << list.length() << ". Total:" << ret.length();
        }else{
            FriendList lst = this->accounts[i]->getFriendList();
            ret.append(lst);
            qDebug() << "Add to friends " << lst.length() << ". Total:" << ret.length();
        }
    }

    if (isLastUpdate)
        this->friendsUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", friendsUpdate=" << friendsUpdate;

    emit updateFriends(ret, (this->friendsUpdate > 0 ? false: true));
}

AlbumList ServiceMgr::getMyAlbums(bool isNeedUpdate, bool useSignal) {
    AlbumList ret;

    // generate list of albums from all accounts
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->getAlbumList().isEmpty() ||
            isNeedUpdate)
            this->accounts.at(i)->setAlbumList(this->getAlbums(this->accounts.at(i)->getProfile(false), isNeedUpdate, useSignal));
        ret.append(this->accounts.at(i)->getAlbumList());
    }

    if (useSignal)
        emit this->updateAlbumList(QString(), ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

AlbumList ServiceMgr::getAlbums(Friend curFriend, bool isNeedUpdate, bool useSignal){
    AlbumList ret;

    qDebug() << "Load album list from cache for " << curFriend.id  << " from " << curFriend.accountId << " needUpdate=" << isNeedUpdate;

    ret = curFriend.getAlbumList();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (curFriend.accountId == this->accounts[i]->transport->accountId) {
                qDebug() << "Download album list for " << curFriend.id;
                if (this->threads.contains(QString("albums") + curFriend.id) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &VkTransport::vkGetAlbums, curFriend.id);
                this->threads.insert(QString("albums") + curFriend.id, status);
                this->albumsUpdate++;
                break;
            }
        }
    }

    if (useSignal)
        emit this->updateAlbumList(curFriend.id, ret, this->albumsUpdate > 0 ? false: true);

    return ret;
}

void ServiceMgr::gotAlbumList(QString accountId, QString owner, AlbumList list, bool isLastUpdate) {


    // store album list to cache
    Friend::setAlbumList(list, accountId, owner);

    // check that it is not a my album list
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts[i]->transport->accountId == accountId &&
            this->accounts[i]->getProfile(false).id == owner) {
            // if it is my album list then store, generate new full list and send it
            this->accounts[i]->setAlbumList(list);
            AlbumList ret;
            for (int j = 0; j < this->accounts.length(); j++) {
                ret.append(this->accounts[j]->getAlbumList());
            }

            if (isLastUpdate)
                this->albumsUpdate--;
            qDebug() << "islastUpdate=" << isLastUpdate << ", albumsUpdate=" << albumsUpdate;

            emit updateAlbumList(QString::QString(), ret, this->albumsUpdate > 0 ? false: true);
            return;
        }
    }

    if (isLastUpdate)
        this->albumsUpdate--;
    qDebug() << "islastUpdate=" << isLastUpdate << ", albumsUpdate=" << albumsUpdate;

    emit updateAlbumList(owner, list, this->albumsUpdate > 0 ? false: true);
}

PhotoList ServiceMgr::getPhotos(Album album, bool isNeedUpdate, bool useSignal) {
    PhotoList ret;

    qDebug() << "Load photo list from cache for " << album.albumId;

    ret = album.getPhotoList();

    this->cleanThreads();

    if (isNeedUpdate || ret.isEmpty()) {
        for (int i = 0; i < this->accounts.length(); i++) {
            if (album.accountId == this->accounts[i]->transport->accountId) {
                qDebug() << "Download photo list for " << album.albumId;
                if (this->threads.contains(Utils::getPhotoListKey(album)) || !this->accounts.at(i)->isNetworkEnabled)
                    return ret;
                QFuture<void> status = QtConcurrent::run(this->accounts[i]->transport, &VkTransport::vkGetPhotos, album.ownerId, album.albumId);
                this->threads.insert(Utils::getPhotoListKey(album), status);
                this->photosUpdate++;
            }
        }
    }

    if (useSignal)
        emit this->updatePhotoList(album.albumId, ret, this->photosUpdate > 0? false: true);

    return ret;
}

void ServiceMgr::gotPhotoList(QString accountId, QString friendId, QString albumId, PhotoList list, bool isLastUpdate) {

    Album::setPhotoList(list, accountId, friendId, albumId);

    if (isLastUpdate)
        this->photosUpdate--;
    if (this->photosUpdate < 0)
        this->photosUpdate = 0;
    qDebug() << "islastUpdate=" << isLastUpdate << ", photosUpdate=" << photosUpdate;

    emit updatePhotoList(albumId, list, photosUpdate > 0 ? false : true);
    emit updatePhoto(accountId, friendId, albumId, list, isLastUpdate);
}

void ServiceMgr::gotProfile(QString, Friend) {
    emit updateAccounts(this->accounts);
}

AccountList ServiceMgr::getAccounts() {
    // check that all accounts has profiles
    for (int i = 0; i < this->accounts.length(); i++) {
        this->accounts.at(i)->getProfile(false);
    }

    return this->accounts;
}

void ServiceMgr::addAccount(QString accountName, DriverInfo *driver)
{
    Account *newAcc = new Account(driver->library, accountName);
    QObject::connect(newAcc->transport, SIGNAL(friendsReceived(QString, FriendList, bool)), this, SLOT(gotFriends(QString, FriendList, bool)));
    QObject::connect(newAcc->transport, SIGNAL(albumsReceived(QString, QString, AlbumList, bool)), this, SLOT(gotAlbumList(QString, QString, AlbumList, bool)));
    QObject::connect(newAcc->transport, SIGNAL(photosReceived(QString, QString, QString, PhotoList, bool)), this, SLOT(gotPhotoList(QString, QString, QString, PhotoList, bool)));
    QObject::connect(newAcc, SIGNAL(updateProfile(QString,Friend)), this, SLOT(gotProfile(QString,Friend)));
    QObject::connect(newAcc->transport, SIGNAL(errorOccurred(QString,int,QString, VkTransport::Action)), this, SLOT(gotErrorMsg(QString,int,QString, VkTransport::Action)));

    if (newAcc != NULL) {
        this->accounts.append(newAcc);
        this->updateDriverSettings();

        this->updateAccounts(this->accounts);
    }
}

void ServiceMgr::deleteAccount(Account *oldAcc)
{
    int i = -1;
    for (int j = 0; j < this->accounts.length(); j++) {
        if (oldAcc->transport->accountId == this->accounts.at(j)->transport->accountId)
            i = j;
    }

    if (i >= 0) {
        this->accounts.removeAt(i);
        QDir dir(Utils::getAccountDir(oldAcc->transport->accountId));

        delete oldAcc;
        qDebug() << "Try to remove dir " << dir.path();
        Utils::RemoveDirectory(dir);
        //execute accounts update process
        this->updateAccounts(this->accounts);
    }
}

QList<DriverInfo*> ServiceMgr::getDrivers() {
    QList<DriverInfo*> ret;

    QDir dir;
    dir.setFilter(QDir::Dirs|QDir::NoDotAndDotDot);
    dir.setPath(Utils::getDriversDir());
    QFileInfoList lst = dir.entryInfoList();

    qDebug() << "Found " << lst.length() << " entries at " << dir.path();

    for (int i = 0; i < lst.length(); i++) {
        QFileInfo dir = lst.at(i);
        if (dir.isDir() && dir.fileName() != "." && dir.fileName() != "..")
        {
            QDir files;
            files.setFilter(QDir::Files);
            files.setNameFilters(QStringList(QString("*.so")));
            files.setPath(dir.filePath());
            QFileInfoList lstFiles = files.entryInfoList();
            for (int j = 0; j < lstFiles.length(); j++) {
                QFileInfo file = lstFiles.at(j);
                if (file.isFile()) {
                    DriverInfo *curDr = VkTransport::getDriverInfo(file.filePath());
                    if (curDr != NULL) {
                        qDebug() << "found driver " << curDr->name;
                        ret.append(curDr);
                    }
                }
            }
        }
    }

    return ret;
}

PhotoList ServiceMgr::getPhotosForAlbum(Photo curPhoto) {
    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == curPhoto.accountId && this->accounts.at(i)->isNetworkEnabled) {
            return this->getPhotosForAlbum(this->accounts.at(i), curPhoto);
        }
    }

    PhotoList ret;
    return ret;
}

PhotoList ServiceMgr::getPhotosForAlbum(Account *curAcc, Photo curPhoto) {
    PhotoList ret;

    AlbumList albums;

    // check that it can be my album
    if (curAcc->getProfile(false).id == curPhoto.ownerId) {
        albums = curAcc->getAlbumList();

    }else{
        FriendList friends;
        friends = curAcc->getFriendList();

        for (int i = 0; i < friends.length(); i++) {
            if (friends.at(i).id == curPhoto.ownerId) {
                albums = friends[i].getAlbumList();
                break;
            }
        }
    }

    for (int i = 0; i < albums.length(); i++) {
        if (albums.at(i).albumId == curPhoto.albumId) {
            return albums[i].getPhotoList();
        }
    }

    return ret;
}

void ServiceMgr::downloadPhotos(Photo startPhoto, int nearest) {
    qDebug() << "Start to download photo images from " << startPhoto.photoId << " for " << startPhoto.albumId;

    this->cleanThreads();

    for (int i = 0; i < this->accounts.length(); i++) {
        if (this->accounts.at(i)->transport->accountId == startPhoto.accountId && this->accounts.at(i)->isNetworkEnabled) {
            PhotoList photos = this->getPhotosForAlbum(this->accounts.at(i), startPhoto);
            if (!this->threads.contains(Utils::getPhotoListKey(photos)))
            {
                QFuture<void> ret = QtConcurrent::run(this->accounts.at(i)->transport, &VkTransport::vkDownloadPhotoList, photos, startPhoto, nearest);
                this->threads.insert(Utils::getPhotoListKey(photos), ret);
                this->photosUpdate++;
            }
            break;
        }
    }
}

void ServiceMgr::gotErrorMsg(QString accountId, int errNum, QString errMsg, VkTransport::Action acc) {
    QString prefix("During request ");
    QString comment;

    switch (acc) {
    case VkTransport::updateListFriendsAction:
        this->friendsUpdate--;
        prefix += tr("\"Update list of friends\"");
        break;
    case VkTransport::getAlbumsAction:
        this->albumsUpdate--;
        prefix += tr("\"Update list of albums\"");
        break;
    case VkTransport::getPhotosAction:
        this->photosUpdate--;
        prefix += tr("\"Update list of images\"");
        break;
    case VkTransport::getProfilesAction:
        prefix += tr("\"Update user profile\"");
        comment = tr("Account swiched to offline mode. Please check network settings and try again.");
        break;
    }

    prefix += tr("from account \"%1\" the error %2 was caused: ").arg(accountId).arg(QString::number(errNum)) + errMsg;
    if (!comment.isEmpty())
        prefix += tr("\n") + comment;

    emit this->errorOccured(prefix, acc);
}
