/*
 * This file is part of mydicto
 *
 * Copyright (C) 2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gconf/gconf-client.h>

#include <libosso.h>

#include <hildon/hildon.h>
#include <hildon-extras/he-about-dialog.h>

#ifdef WITH_SHARING

#ifdef WITH_SHARING_H
#include <sharingdialog/sharing-dialog.h>
#endif

#else
void sharing_dialog_with_file(osso_context_t *osso, GtkWindow *parent, const gchar *uri);
#endif

#include <dicto/dicto.h>
#include <dicto/dicto-ui.h>
#include "config.h"

/* #define PHONE_SRC "liveadder name=phonemixer ! pulsesrc device=sink.hw0.monitor ! phonemixer. pulsesrc device=source.hw0 ! phonemixer." */
#define PHONE_SRC "adder name=phonemixer pulsesrc device=sink.hw0.monitor ! queue ! phonemixer. pulsesrc device=source.hw0 ! queue ! phonemixer. phonemixer. ! queue"
#define SYSTEM_SRC "pulsesrc device=sink.hw0.monitor ! audioconvert ! queue "

typedef enum {
	SRC_MIC=0,
	SRC_PHONE=1,
	SRC_SYSTEM=2,
	SRC_END=999
} myDSrcId;

typedef struct _MyDicto MyDicto;
struct _MyDicto {
	HildonProgram *program;
	HildonWindow *window;
	osso_context_t *osso;
	GtkWidget *dw;
	GtkWidget *source_button, *sharing_button, *format_button, *about_button;
	Dicto *d, *dm, *dp;
	GtkTreeModel *formats;
	DictoFormats f_mic;
	DictoFormats f_phone;
	gchar *sink;
	gchar *mic_src;
	gchar *phone_src;
};

MyDicto mapp;

typedef struct _MyDictoSrc MyDictoSrc;
struct _MyDictoSrc {
	myDSrcId id;
	gchar *name;
	gchar *src;
};

static MyDictoSrc sources[]={
	{ SRC_MIC,	"Microphone", AUDIO_SRC },
	{ SRC_PHONE,	"Phone", PHONE_SRC },
	{ SRC_SYSTEM,	"System", SYSTEM_SRC },
	{ SRC_END,	NULL, NULL }
};

#define MYDICTO_GCONF_PATH "/apps/mydicto"
#define MYDICTO_GCONF_AUDIO_FORMAT_MIC MYDICTO_GCONF_PATH  "/audio-format"
#define MYDICTO_GCONF_AUDIO_FORMAT_PHONE MYDICTO_GCONF_PATH  "/audio-format-phone"

static void
mydicto_gconf_store(MyDicto *md)
{
GConfClient *client;

client=gconf_client_get_default();
gconf_client_set_int(client, MYDICTO_GCONF_AUDIO_FORMAT_MIC, md->f_mic, NULL);
gconf_client_set_int(client, MYDICTO_GCONF_AUDIO_FORMAT_PHONE, md->f_phone, NULL);
}

static gboolean
get_selected_id_from_selector(HildonTouchSelector *selector, gint *id, gchar **desc)
{
GtkTreePath *path;
GtkTreeModel *model;
GtkTreeIter iter;

#if 0
path=hildon_touch_selector_get_last_activated_row(selector, 0);
if (path==NULL)
	return FALSE;
model=hildon_touch_selector_get_model(selector, 0);
gtk_tree_model_get_iter(model, &iter, path);
gtk_tree_model_get(model, &iter, 0, id, 1, desc, -1);
gtk_tree_path_free(path);
#else
if (!hildon_touch_selector_get_selected(selector, 0, &iter))
	return FALSE;

model=hildon_touch_selector_get_model(selector, 0);
gtk_tree_model_get(model, &iter, 0, id, 1, desc, -1);
#endif

return TRUE;
}

static gint
mydicto_dbus_req_handler(const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval)
{
return OSSO_OK;
}

/***************************************************
 ** Dicto handlers
 **/

static gboolean
dicto_error_cb(Dicto *d, gpointer err, gpointer data)
{
gchar *msg;
GtkWidget *info;
GError *e=(GError *)err;

g_debug("Dicto reported error: (%d) %s", e->code, e->message);

msg=g_strdup_printf("Error:\n%s", e->message);
info=hildon_note_new_information(GTK_WINDOW(mapp.window), msg);
gtk_dialog_run(GTK_DIALOG(info));
g_free(msg);
return FALSE;
}

static gboolean
dicto_delete_query_cb(GtkWidget *dw, const gchar *file, gpointer data)
{
return dicto_widget_delete_query_dialog(GTK_WINDOW(mapp.window), file);
}

static gboolean
dicto_stop_cb(Dicto *d, gpointer data)
{
gtk_widget_show(mapp.sharing_button);
gtk_widget_show(mapp.format_button);
hildon_banner_show_information(GTK_WIDGET(mapp.window), "", _("Stopped"));
return FALSE;
}

static gboolean
dicto_playing_cb(Dicto *d, gpointer data)
{
gtk_widget_hide(mapp.sharing_button);
gtk_widget_hide(mapp.format_button);
hildon_banner_show_information(GTK_WIDGET(mapp.window), "", _("Playing"));
return FALSE;
}

static gboolean
dicto_recording_cb(Dicto *d, gpointer data)
{
gtk_widget_hide(mapp.sharing_button);
gtk_widget_hide(mapp.format_button);
hildon_banner_show_information(GTK_WIDGET(mapp.window), "", _("Recording"));
return FALSE;
}

/*********************************************************
 ** mydicto callbacks
 **/

#define URL_MYDICTO "http://maemo.tal.org/"
#define URL_BUGS "http://bugzilla.tal.org/"

static gboolean
mydicto_about_cb()
{
he_about_dialog_present(GTK_WINDOW(mapp.window),
    "myDicto",
    "mydicto",
    VERSION,
    "Dictation application",
    "Mapper is free software, GPLv2\n"
    "Copyright 2010 Kaj-Michael Lang <milang@tal.org>",
    URL_MYDICTO,
    URL_BUGS,
    NULL);
return TRUE;
}

static void
mydicto_source_selection_cb(HildonTouchSelector *selector, gint c, gpointer data)
{
gint srcid;

srcid=hildon_touch_selector_get_active(selector, 0);
g_debug("Src: %d", srcid);
switch (srcid) {
	case SRC_MIC:
		mapp.d=mapp.dm;
	break;
	case SRC_PHONE:
		mapp.d=mapp.dp;
	break;
	default:
		g_return_if_reached();
	break;
}
g_object_set(mapp.dw, "dicto", mapp.d, NULL);
}

static void
mydicto_format_selection_cb(HildonTouchSelector *selector, gint c, gpointer data)
{
gint id;
gchar *desc, *msg;

if (!get_selected_id_from_selector(selector, &id, &desc))
	return;
}

static void
mydicto_format_cb(HildonPickerButton *button, gpointer data)
{
HildonTouchSelector *selector=(HildonTouchSelector *)data;
gint id;
gchar *desc, *msg;

if (!get_selected_id_from_selector(selector, &id, &desc))
	return;
if (!dicto_set_format(mapp.d, (DictoFormats)id)) {
	hildon_banner_show_information(GTK_WIDGET(mapp.window), "", _("Failed to set recording format"));
	return;
}
msg=g_strdup_printf("Using format:\n%s", desc);
hildon_banner_show_information(GTK_WIDGET(mapp.window), "", msg);
g_free(msg);
}

#if HILDON_CHECK_VERSION(2,2,0)
static void
mydicto_menu_button_share_cb(GtkWidget *w, gpointer data)
{
DictoWidget *dw=(DictoWidget *)data;
gchar *file;
#ifdef WITH_SHARING
g_object_get(G_OBJECT(mapp.d), "current-file", &file, NULL);
if (file && strlen(file)>0 && g_file_test(file, G_FILE_TEST_IS_REGULAR)) {
	sharing_dialog_with_file(mapp.osso, GTK_WINDOW(mapp.window), file);
} else {
	hildon_banner_show_information(GTK_WIDGET(mapp.window), "", _("No file selected"));
}
g_free(file);
#else
hildon_banner_show_information(GTK_WIDGET(mapp.window), "", "Sharing disabled");
#endif
}
#else
static void
mydicto_menu_share_email_cb(GtkWidget *w, gpointer data)
{
}

static void
mydicto_menu_share_bt_cb(GtkWidget *w, gpointer data)
{
}
#endif

#if HILDON_CHECK_VERSION(2,2,0)
static HildonAppMenu *
mydicto_menu(void)
{
HildonAppMenu *menu;
HildonTouchSelector *f_selector, *src_selector;
HildonTouchSelectorColumn *column;

menu=HILDON_APP_MENU(hildon_app_menu_new());

/* Sharing menu button */
mapp.sharing_button=gtk_button_new_with_label("Share");
g_signal_connect_after(mapp.sharing_button, "clicked", G_CALLBACK(mydicto_menu_button_share_cb), mapp.dw);
hildon_app_menu_append(menu, GTK_BUTTON(mapp.sharing_button));

mapp.source_button=hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
hildon_button_set_title(HILDON_BUTTON(mapp.source_button), _("Recording source"));
src_selector=hildon_touch_selector_new_text();
hildon_touch_selector_append_text(src_selector, _("Microphone"));
hildon_touch_selector_append_text(src_selector, _("Phone"));
hildon_touch_selector_set_active(src_selector, 0, 0);
g_signal_connect_after(src_selector, "changed", G_CALLBACK(mydicto_source_selection_cb), NULL);
hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(mapp.source_button), HILDON_TOUCH_SELECTOR(src_selector));
hildon_app_menu_append(menu, GTK_BUTTON(mapp.source_button));

mapp.format_button=hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
hildon_button_set_title(HILDON_BUTTON(mapp.format_button), _("Recording format"));
f_selector=hildon_touch_selector_new();
column=hildon_touch_selector_append_column(f_selector, GTK_TREE_MODEL(mapp.formats), gtk_cell_renderer_text_new(), "text", 1, NULL);
hildon_touch_selector_column_set_text_column(column, 1);
hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(mapp.format_button), HILDON_TOUCH_SELECTOR(f_selector));
hildon_touch_selector_set_column_selection_mode(f_selector, HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);
hildon_app_menu_append(menu, GTK_BUTTON(mapp.format_button));

mapp.about_button=hildon_gtk_button_new(HILDON_SIZE_AUTO);
gtk_button_set_label(GTK_BUTTON(mapp.about_button), _("About"));
hildon_app_menu_append(menu, GTK_BUTTON(mapp.about_button));

g_signal_connect(G_OBJECT(mapp.format_button), "value-changed", G_CALLBACK(mydicto_format_cb), f_selector);
g_signal_connect(G_OBJECT(mapp.about_button), "clicked", G_CALLBACK(mydicto_about_cb), NULL);
g_signal_connect(G_OBJECT(f_selector), "changed", G_CALLBACK(mydicto_format_selection_cb), NULL);

gtk_widget_show_all(GTK_WIDGET(menu));
return menu;
}
#else
static GtkMenu *
mydicto_menu(void)
{
GtkWidget *menu, *email, *bt;

menu=gtk_menu_new();
email=gtk_menu_item_new_with_label("Send via E-mail");
bt=gtk_menu_item_new_with_label("Send via Bluetooth");
gtk_menu_append(menu, email);
gtk_menu_append(menu, bt);

g_signal_connect(G_OBJECT(email), "activate", GTK_SIGNAL_FUNC(mydicto_menu_share_email_cb), mapp.dw);
g_signal_connect(G_OBJECT(bt), "activate", GTK_SIGNAL_FUNC(mydicto_menu_share_bt_cb), mapp.dw);

return menu;
}
#endif

static void
mydicto_connect_dicto(Dicto *d, MyDicto *md)
{
g_signal_connect(G_OBJECT(d), "error", G_CALLBACK(dicto_error_cb), md->dw);
g_signal_connect(G_OBJECT(d), "playing", G_CALLBACK(dicto_playing_cb), md->dw);
g_signal_connect(G_OBJECT(d), "recording", G_CALLBACK(dicto_recording_cb), md->dw);
g_signal_connect(G_OBJECT(d), "stopped-play", G_CALLBACK(dicto_stop_cb), md->dw);
g_signal_connect(G_OBJECT(d), "stopped-record", G_CALLBACK(dicto_stop_cb), md->dw);
}

static gboolean
mydicto_init(MyDicto *md)
{
osso_return_t result;

g_set_application_name("myDicto");
gtk_window_set_default_icon_name("mydicto");

md->program=HILDON_PROGRAM(hildon_program_get_instance());

md->dm=dicto_new_full(NOTES_BASEDIR, md->f_mic, md->sink, md->mic_src);
if (!md->dm) {
	g_error("Failed to create microphone dicto");
	return FALSE;
}
md->dp=dicto_new_full(NOTES_PHONE_BASEDIR, md->f_phone, md->sink, md->phone_src);
if (!md->dp) {
	g_error("Failed to create phone dicto");
	return FALSE;
}
md->d=md->dm;

md->dw=GTK_WIDGET(dicto_widget_new(md->d));
md->formats=dicto_widget_get_formats(DICTO_WIDGET(md->dw));

#if HILDON_CHECK_VERSION(2,2,0)
md->window=HILDON_WINDOW(hildon_stackable_window_new());
hildon_window_set_app_menu(HILDON_WINDOW(md->window), mydicto_menu());
#else
md->window=HILDON_WINDOW(hildon_window_new());
hildon_window_set_menu(HILDON_WINDOW(md->window), mydicto_menu());
#endif

hildon_program_add_window(md->program, md->window);

gtk_window_set_title(GTK_WINDOW(md->window), "myDicto");
g_signal_connect(G_OBJECT(md->window), "destroy", G_CALLBACK(gtk_main_quit), NULL);

md->osso=osso_initialize("org.tal.mydicto", VERSION, FALSE, NULL);
if (!md->osso)
	return FALSE;

result=osso_rpc_set_cb_f(md->osso, "org.tal.mydicto", "/org/tal/mydicto", "org.tal.mydicto", mydicto_dbus_req_handler, NULL);
if (result!=OSSO_OK)
	return FALSE;

mydicto_connect_dicto(md->dm, md);
mydicto_connect_dicto(md->dp, md);

g_signal_connect(G_OBJECT(md->dw), "delete-query", G_CALLBACK(dicto_delete_query_cb), NULL);
g_object_set(md->dw, "enable-delete", TRUE, NULL);

gtk_container_add(GTK_CONTAINER(md->window), md->dw);
gtk_widget_show_all(GTK_WIDGET(md->window));

return TRUE;
}

gint 
main(gint argc, gchar * argv[])
{
GConfClient *client;

bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
textdomain(GETTEXT_PACKAGE);

#if HILDON_CHECK_VERSION(2,2,0)
hildon_gtk_init(&argc, &argv);
#else
gtk_init(&argc, &argv);
#endif
gst_init(&argc, &argv);

mapp.f_mic=FORMAT_WAV_8K_16B_M;
mapp.f_phone=FORMAT_WAV_22K_16B_M;
mapp.sink=AUDIO_SINK;
mapp.mic_src=AUDIO_SRC;
mapp.phone_src=PHONE_SRC;

/* Veeery simple argument passing, mainly for getting working sinks under sdk */
if (argc>1) {
	mapp.f_mic=mapp.f_phone=atoi(argv[1]);
	if (argc>2)
		mapp.mic_src=argv[2];
	if (argc>3)
		mapp.sink=argv[3];
} else {
	GError *error=NULL;
	gint v;

	client=gconf_client_get_default();
	v=gconf_client_get_int(client, MYDICTO_GCONF_AUDIO_FORMAT_MIC, &error);
	if (error!=NULL) {
		g_debug("Failed to get mic format from gconf.");
		g_error_free(error);
	} else {
		mapp.f_mic=v;
	}
	v=gconf_client_get_int(client, MYDICTO_GCONF_AUDIO_FORMAT_PHONE, &error);
	if (error!=NULL) {
		g_debug("Failed to get phone format from gconf.");
		g_error_free(error);
	} else {
		mapp.f_mic=v;
	}
}

if (argc>4) {
	g_setenv("PULSE_PROP_media.role", argv[4], TRUE);
	g_debug("Pulseaudio role set to: %s", argv[4]);
}

if (!mydicto_init(&mapp)) {
	g_warning("myDicto init failed");
	return 1;
}

gtk_main();

mydicto_gconf_store(&mapp);

return 0;
}

