


/* Copuright 2010 Mikko Antila */

/* And
 
 * md2.c -- md2 model loader
 * last modification: aug. 14, 2007
 *

 
 * Copyright (c) 2005-2007 David HENRY

 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * gcc -Wall -ansi -lGL -lGLU -lglut md2.c -o md2
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "md2.h"
#include <QDebug>
#include <QtCore>
#include <QtGui>
#include <QGLWidget>
/*** An MD2 model ***/

/* Table of precalculated normals */
vec3_t anorms_table[162] = { 
#include "anorms.h" 
};




/**
 * Load an MD2 model from file.
 *
 * Note: MD2 format stores model's data in little-endian ordering.  On
 * big-endian machines, you'll have to perform proper conversions.
 */
MD2Model::MD2Model() {
}

int
MD2Model::ReadMD2Model (const char *filename, const char *texturefile, struct md2_model_t *mdl)
{
  FILE *fp;
  int i;

  fp = fopen (filename, "rb");
  if (!fp)
    {
      fprintf (stderr, "Error: couldn't open \"%s\"!\n", filename);
      return 0;
    }

  /* Read header */
  fread (&mdl->header, 1, sizeof (struct md2_header_t), fp);

  if ((mdl->header.ident != 844121161) ||
      (mdl->header.version != 8))
    {
      /* Error! */
      fprintf (stderr, "Error: bad version or identifier\n");
      fclose (fp);
      return 0;
    }

  /* Memory allocations */
  mdl->skins = (struct md2_skin_t *)
    malloc (sizeof (struct md2_skin_t) * mdl->header.num_skins);
  mdl->texcoords = (struct md2_texCoord_t *)
    malloc (sizeof (struct md2_texCoord_t) * mdl->header.num_st);
  mdl->triangles = (struct md2_triangle_t *)
    malloc (sizeof (struct md2_triangle_t) * mdl->header.num_tris);
  mdl->frames = (struct md2_frame_t *)
    malloc (sizeof (struct md2_frame_t) * mdl->header.num_frames);
  mdl->glcmds = (int *)malloc (sizeof (int) * mdl->header.num_glcmds);

  /* Read model data */
  fseek (fp, mdl->header.offset_skins, SEEK_SET);
  fread (mdl->skins, sizeof (struct md2_skin_t),
	 mdl->header.num_skins, fp);

  fseek (fp, mdl->header.offset_st, SEEK_SET);
  fread (mdl->texcoords, sizeof (struct md2_texCoord_t),
	 mdl->header.num_st, fp);

  fseek (fp, mdl->header.offset_tris, SEEK_SET);
  fread (mdl->triangles, sizeof (struct md2_triangle_t),
	 mdl->header.num_tris, fp);

  fseek (fp, mdl->header.offset_glcmds, SEEK_SET);
  fread (mdl->glcmds, sizeof (int), mdl->header.num_glcmds, fp);

  /* Read frames */
  fseek (fp, mdl->header.offset_frames, SEEK_SET);
  for (i = 0; i < mdl->header.num_frames; ++i)
    {
      /* Memory allocation for vertices of this frame */
      mdl->frames[i].verts = (struct md2_vertex_t *)
	malloc (sizeof (struct md2_vertex_t) * mdl->header.num_vertices);

      /* Read frame data */
      fread (mdl->frames[i].scale, sizeof (vec3_t), 1, fp);
      fread (mdl->frames[i].translate, sizeof (vec3_t), 1, fp);
      fread (mdl->frames[i].name, sizeof (char), 16, fp);
      fread (mdl->frames[i].verts, sizeof (struct md2_vertex_t),
	     mdl->header.num_vertices, fp);
    }

  fclose (fp);
	
	QImage buf;
	buf.load(QString(texturefile));
	QImage textImage = QGLWidget::convertToGLFormat(buf);
	glGenTextures(1, &texture);
	glBindTexture(GL_TEXTURE_2D, texture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, textImage.width(), textImage.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE, textImage.bits());
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glBindTexture(GL_TEXTURE_2D, 0);

  return 1;
}





/**
 * Free resources allocated for the model.
 */
void
MD2Model::FreeModel (struct md2_model_t *mdl)
{
	
  int i;
  	
  glDeleteBuffers(mdl->header.num_frames, fbuffers);

  if (mdl->skins)
    {
      free (mdl->skins);
      mdl->skins = NULL;
    }

  if (mdl->texcoords)
    {
      free (mdl->texcoords);
      mdl->texcoords = NULL;
    }

  if (mdl->triangles)
    {
      free (mdl->triangles);
      mdl->triangles = NULL;
    }

  if (mdl->glcmds)
    {
      free (mdl->glcmds);
      mdl->glcmds = NULL;
    }

  if (mdl->frames)
    {
      for (i = 0; i < mdl->header.num_frames; ++i)
	{
	  free (mdl->frames[i].verts);
	  mdl->frames[i].verts = NULL;
	}

      free (mdl->frames);
      mdl->frames = NULL;
    }
}

/**
 * Render the model at frame n.
 */
void
MD2Model::RenderFrame (int n, const struct md2_model_t *mdl)
{
}


/**
 * generete buffers and so on
 **/

void MD2Model::DumpData (struct md2_model_t *mdl) {
	fbuffers = new GLuint[mdl->header.num_frames];
	pmitta = 0;
	glGenBuffers(mdl->header.num_frames, fbuffers);
	
  int i, j;
  struct md2_frame_t *pframe;
  struct md2_vertex_t *pvert;
  Vertex *vpc;
  for(int n=0; n<mdl->header.num_frames;n++) {

  vpc  = new Vertex[mdl->header.num_tris*3];
  int ide = 0;
	for (i = 0; i < mdl->header.num_tris; ++i)
      {
    for (j = 0; j < 3; ++j)
	  {
	    pframe = &mdl->frames[n];
	    pvert = &pframe->verts[mdl->triangles[i].vertex[j]];

	    /* Compute texture coordinates */
		
	    vpc[ide].u = (float)mdl->texcoords[mdl->triangles[i].st[j]].s / mdl->header.skinwidth;
	    vpc[ide].v = (float)mdl->texcoords[mdl->triangles[i].st[j]].t / mdl->header.skinheight;
		vpc[ide].nx = anorms_table[pvert->normalIndex][0];
		vpc[ide].ny = anorms_table[pvert->normalIndex][1];
		vpc[ide].nz = anorms_table[pvert->normalIndex][2];
		vpc[ide].x = (pframe->scale[0] * pvert->v[0]) + pframe->translate[0];
	    vpc[ide].y = (pframe->scale[1] * pvert->v[1]) + pframe->translate[1];
	    vpc[ide].z = (pframe->scale[2] * pvert->v[2]) + pframe->translate[2];
		ide++;

	  }
	}
	ide--;	
	pmitta = ide;
	glBindBuffer(GL_ARRAY_BUFFER, fbuffers[n]);
	glBufferData(GL_ARRAY_BUFFER, ide*sizeof(Vertex), vpc, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	free(vpc);
	vpc = NULL;

  }
}	

void MD2Model::RenderBuffers(int n, GLuint program) {
	glBindTexture(GL_TEXTURE_2D, texture);
	glEnableVertexAttribArray(0);
  	glEnableVertexAttribArray(1);
  	glEnableVertexAttribArray(2);

	glBindBuffer(GL_ARRAY_BUFFER, fbuffers[n]);
 	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), ((char *)NULL + (0)));
	glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex), ((char *)NULL + (sizeof(float)*3)));
    	glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), ((char *)NULL + (sizeof(float)*5)));
	glBindAttribLocation(program, 0, "vertex");
	glBindAttribLocation(program, 1, "texture");
	glBindAttribLocation(program, 2, "normaali");
	glDrawArrays(GL_TRIANGLES, 0, pmitta);
	glBindTexture(GL_TEXTURE_2D, 0);
    	glBindBuffer(GL_ARRAY_BUFFER, 0);
}
	

/**
 * Calculate the current frame in animation beginning at frame
 * 'start' and ending at frame 'end', given interpolation percent.
 * interp will be reseted to 0.0 if the next frame is reached.
 */



MD2Model::~MD2Model ()
{
}

